﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ProgramOptionParser.h"
#include <cstdio>
#include <cstdlib>
#include <getopt.h>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nnt/nnt_Argument.h>

namespace nns {

const int CharBufferLength = 512;

class ProgramOptionParser::ProgramOptionParserImpl
{
public:
    typedef struct option Option;
    typedef std::vector<Option> OptionArray;
    OptionArray m_Option;

    typedef struct
    {
        const char* helpMessage;
        bool isOptionAssigned;
        char argument[CharBufferLength];
    } OptionAux;
    typedef std::vector<OptionAux> OptionAuxArray;
    OptionAuxArray m_OptionAux;

    void AddHelpOption() NN_NOEXCEPT;
    void CreateOptionString(char* optionString) NN_NOEXCEPT;
    bool CheckOptionValidity(int argc, char** argv) NN_NOEXCEPT;
    bool ParseArgument(const char* optionString) NN_NOEXCEPT;
};

NN_IMPLICIT ProgramOptionParser::ProgramOptionParser() NN_NOEXCEPT
:
m_Impl(new ProgramOptionParser::ProgramOptionParserImpl())
{
}

ProgramOptionParser::~ProgramOptionParser() NN_NOEXCEPT
{
}

void ProgramOptionParser::ProgramOptionParserImpl::AddHelpOption() NN_NOEXCEPT
{
    for (auto iter=m_Option.begin(); iter!=m_Option.end(); ++iter)
    {
        if (iter->val == 'h')
        {
            // 既に help が追加されているなら追加しない。
            return;
        }
    }
    m_Option.push_back({"help", no_argument, 0, 'h'});
    m_OptionAux.push_back({ "show usage of this application.", false });
}

void ProgramOptionParser::ProgramOptionParserImpl::CreateOptionString(char* optionString) NN_NOEXCEPT
{
    int position = 0;
    for (auto iter=m_Option.begin(); iter!=m_Option.end(); ++iter)
    {
        // 動的確保を避けた実装だが、エラー処理をきっちりやるコストが高いので、
        // NN_ASSERT_LESS を使って範囲内のアクセスを保証。
        NN_ASSERT_LESS(position, CharBufferLength);
        optionString[position++] = static_cast<char>(iter->val);
        if (required_argument == iter->has_arg)
        {
            NN_ASSERT_LESS(position, CharBufferLength);
            optionString[position++] = ':';
        }
    }
}

bool ProgramOptionParser::ProgramOptionParserImpl::CheckOptionValidity(int argc, char** argv) NN_NOEXCEPT
{
    bool validity = true;
    for (int i1=1; i1<argc; ++i1)
    {
        const char* option = argv[i1];
        if ('-' == option[0])
        {
            if (2 < strnlen(option, CharBufferLength) && '-' != option[1])
            {
                // Short name は 2文字でないとダメ。
                NN_LOG("Warning: Find invalid long option:[%s]. Two '-'s are necessary.\n", option);
                validity = false;
                continue;
            }
            // 一意な Long name オプションであるかどうか。
            for (int i2=(i1 + 1); i2<argc; ++i2)
            {
                if (0 == strncmp(option, argv[i2], CharBufferLength))
                {
                    NN_LOG("Warning: Find not unique option:[%s]\n", option);
                    validity = false;
                    break;
                }
            }
        }
    }
    return validity;
}

bool ProgramOptionParser::ProgramOptionParserImpl::ParseArgument(const char* optionString) NN_NOEXCEPT
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    if (!ProgramOptionParserImpl::CheckOptionValidity(argc, argv))
    {
        return false;
    }

    bool parseResult = true;
    int option;
    int longOptionIndex = 0;
    while ((option = getopt_long(argc, argv, optionString, &(m_Option[0]), &longOptionIndex)) != -1)
    {
        bool isOptionAssigned = false;
        for (int i1=0; i1<m_Option.size(); ++i1)
        {
            if (m_Option[i1].val == option)
            {
                if (m_Option[i1].has_arg != no_argument)
                {
                    std::strncpy(m_OptionAux[i1].argument, optarg, CharBufferLength);
                }
                m_OptionAux[i1].isOptionAssigned = isOptionAssigned = true;
            }
        }
        if ('h' == option)
        {
            // ヘルプが指定されたら終了。
            NN_LOG("Detected help option.\n");
            parseResult = false;
        }
        if (!isOptionAssigned)
        {
            // 異常な引数が指定されたら終了。
            NN_LOG("Warning: Detected unknown option:[%c].\n", option);
            parseResult = false;
        }
    }

    if (optind < argc)
    {
        NN_LOG("Warning: Non-option argv elements: ");
        parseResult = false;
        while (optind < argc)
        {
            NN_LOG("[%s] ", argv[optind++]);
        }
        NN_LOG("\n");
    }
    return parseResult;
}

void ProgramOptionParser::AddOption(const char shortName, const char* longName, bool isArgumentRequired, const char* helpMessage) NN_NOEXCEPT
{
    // help 用のオプションは指定不可。
    NN_ASSERT_NOT_EQUAL(shortName, 'h');
    NN_ASSERT(0 != std::strncmp(longName, "help", CharBufferLength));

    // optional_argument は非対応。
    m_Impl->m_Option.push_back(
        {longName, isArgumentRequired ? required_argument : no_argument, 0, shortName}
    );
    m_Impl->m_OptionAux.push_back({ helpMessage, false });
}

bool ProgramOptionParser::ParseOption() NN_NOEXCEPT
{
    m_Impl->AddHelpOption();
    char optionString[CharBufferLength];
    m_Impl->CreateOptionString(optionString);
    return m_Impl->ParseArgument(optionString);
}

bool ProgramOptionParser::GetBooleanOptionalValue(const char shortName, bool defaultValue) NN_NOEXCEPT
{
    for (int i1=0; i1<m_Impl->m_Option.size(); ++i1)
    {
        const ProgramOptionParserImpl::Option& option = m_Impl->m_Option[i1];
        const ProgramOptionParserImpl::OptionAux& aux = m_Impl->m_OptionAux[i1];

        if (option.val == shortName)
        {
            if (aux.isOptionAssigned)
            {
                if (option.has_arg == no_argument)
                {
                    // デフォルトと逆の評価を返す。
                    return !defaultValue;
                }
                return 0 < std::atoi(aux.argument);
            }
            // 指定されていなかった場合は default の値を返す。
            return defaultValue;
        }
    }
    NN_LOG("Warning: Cannot find the short option name:[%c]\n", shortName);
    return defaultValue;
}

int ProgramOptionParser::GetIntegerOptionalValue(const char shortName, int defaultValue) NN_NOEXCEPT
{
    for (int i1=0; i1<m_Impl->m_Option.size(); ++i1)
    {
        const ProgramOptionParserImpl::Option& option = m_Impl->m_Option[i1];
        const ProgramOptionParserImpl::OptionAux& aux = m_Impl->m_OptionAux[i1];

        if (option.val == shortName)
        {
            if (aux.isOptionAssigned)
            {
                NN_ASSERT_NOT_EQUAL(option.has_arg, no_argument);
                return std::atoi(aux.argument);
            }
            // 指定されていなかった場合は default の値を返す。
            return defaultValue;
        }
    }
    NN_LOG("Warning: Cannot find the short option name:[%c]\n", shortName);
    return defaultValue;
}

float ProgramOptionParser::GetFloatOptionalValue(const char shortName, float defaultValue) NN_NOEXCEPT
{
    for (int i1=0; i1<m_Impl->m_Option.size(); ++i1)
    {
        const ProgramOptionParserImpl::Option& option = m_Impl->m_Option[i1];
        const ProgramOptionParserImpl::OptionAux& aux = m_Impl->m_OptionAux[i1];

        if (option.val == shortName)
        {
            if (aux.isOptionAssigned)
            {
                NN_ASSERT_NOT_EQUAL(option.has_arg, no_argument);
                return std::atof(aux.argument);
            }
            // 指定されていなかった場合は default の値を返す。
            return defaultValue;
        }
    }
    NN_LOG("Warning: Cannot find the short option name:[%c]\n", shortName);
    return defaultValue;
}

const char* ProgramOptionParser::GetStringOptionalValue(const char shortName, const char* defaultValue) NN_NOEXCEPT
{
    for (int i1=0; i1<m_Impl->m_Option.size(); ++i1)
    {
        const ProgramOptionParserImpl::Option& option = m_Impl->m_Option[i1];
        const ProgramOptionParserImpl::OptionAux& aux = m_Impl->m_OptionAux[i1];

        if (option.val == shortName)
        {
            if (aux.isOptionAssigned)
            {
                NN_ASSERT_NOT_EQUAL(option.has_arg, no_argument);
                return aux.argument;
            }
            // 指定されていなかった場合は default の値を返す。
            return defaultValue;
        }
    }
    NN_LOG("Warning: Cannot find the short option name:[%c]\n", shortName);
    return defaultValue;
}

void ProgramOptionParser::ShowHelp() NN_NOEXCEPT
{
    NN_LOG("Usage: %s\n", nnt::GetHostArgv()[0]);
    for (int i1=0; i1<m_Impl->m_Option.size(); ++i1)
    {
        const ProgramOptionParserImpl::Option& option = m_Impl->m_Option[i1];
        const ProgramOptionParserImpl::OptionAux& aux = m_Impl->m_OptionAux[i1];
        char optionView[CharBufferLength];
        std::snprintf(optionView, CharBufferLength, "-%c, --%s %s",
            option.val, option.name, (required_argument == option.has_arg) ? "<param>" : "");
        NN_LOG("    %-30s: %s\n", optionView, aux.helpMessage);
    }

}

}
