﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "FilePathViewer.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h> // SD への FAT アクセス

namespace nns {

namespace {

}

NN_IMPLICIT FilePathViewer::FilePathViewer(const char* hostPath) NN_NOEXCEPT
{
    m_SdCardOpenResult = nn::fs::MountSdCardForDebug("SdCard");
    m_HostOpenResult = nn::fs::MountHost("Host", hostPath);

    nn::Result result = nn::fs::QueryMountRomCacheSize(&m_MountRomCacheBufferSize);
    if (result.IsSuccess())
    {
        m_MountRomCacheBuffer = malloc(m_MountRomCacheBufferSize);
        if (nullptr != m_MountRomCacheBuffer)
        {
            m_RomOpenResult = nn::fs::MountRom("Contents", m_MountRomCacheBuffer, m_MountRomCacheBufferSize);
        }
    }
}

FilePathViewer::~FilePathViewer() NN_NOEXCEPT
{
    if (m_SdCardOpenResult.IsSuccess())
    {
        nn::fs::Unmount("SdCard");
    }
    if (m_HostOpenResult.IsSuccess())
    {
        nn::fs::Unmount("Host");
    }
    if (m_RomOpenResult.IsSuccess())
    {
        nn::fs::Unmount("Contents");
        free(m_MountRomCacheBuffer);
    }
}

void FilePathViewer::SearchImageFileFromAttachedDevice(std::vector<std::string>* pFileList) NN_NOEXCEPT
{
    pFileList->clear();
    pFileList->reserve(128);

    if (m_SdCardOpenResult.IsSuccess())
    {
        SearchImageFile(pFileList, "SdCard:/");
    }
    else
    {
        NN_LOG("Cannot mount SD card.\n");
    }
    if (m_HostOpenResult.IsSuccess())
    {
        SearchImageFile(pFileList, "Host:/");
    }
    else
    {
        NN_LOG("Cannot mount host.\n");
    }
}

bool FilePathViewer::SearchImageFile(std::vector<std::string>* pFileList, const char* directoryPath) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle handle;
    auto result = nn::fs::OpenDirectory(&handle, directoryPath, nn::fs::OpenDirectoryMode_File);
    if (!result.IsSuccess())
    {
        NN_LOG("Warning: Cannot open directory:[%s]\n", directoryPath);
        return false;
    }

    while (true)
    {
        int64_t value = 0;
        nn::fs::DirectoryEntry entry;
        result = nn::fs::ReadDirectory(&value, &entry, handle, 1);
        if (!result.IsSuccess() || 0 == value)
        {
            break;
        }
        std::string filePath = std::string(directoryPath) + std::string(entry.name);
        std::string extension = ExtractExtension(filePath);
        if (".bmp" == extension || ".BMP" == extension ||
            ".jpg" == extension || ".JPG" == extension ||
            ".jpeg" == extension || ".JPEG" == extension)
        {
            pFileList->push_back(filePath);
        }
    }
    nn::fs::CloseDirectory(handle);
    return true;
}

const std::string FilePathViewer::ExtractExtension(const std::string& path) NN_NOEXCEPT
{
    std::string::size_type dotPosition = path.rfind(".");
    if (std::string::npos != dotPosition)
    {
        return path.substr(dotPosition);
    }
    return "";
}

}
