﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// Simple CEC API test app

// このテストでは、同一のテストプロセス内で、サーバとクライアントを動作させている。
// 実環境では、一般に、サーバとクライアントは別プロセスとなる。

#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_EventApi.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Argument.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_SdkThreadApi.h>
#include <nn/nn_SdkLog.h>
#include <nn/settings/system/settings_Tv.h>

#include <nn/cec/cec_Api.h>

namespace nn { namespace cec {
namespace {

    const size_t    s_CancelTestStackSize = 8 * 1024;
    NN_ALIGNAS(os::MemoryPageSize) char s_CancelTestStack[s_CancelTestStackSize];

    typedef struct
    {
        os::EventType   event;
    } CecCancelTestDataType;

    const char BusEventNameIgnore[] = "Ignore";
    const char BusEventNameActiveSourceChangedToActive[] = "Active Source --> active";
    const char BusEventNameActiveSourceChangedToInactive[] = "Active Source --> inactive";
    const char BusEventNameGoStandby[] = "Standby";
    const char BusEventNameSuspending[] = "Suspending";
    const char BusEventNameConnectionChange[] = "Connection Change";
    const char BusEventNameOtpFeatureAbort[] = "One Touch Play Feature Abort.";
    const char BusEventNameStandbyFeatureAbort[] = "Standby Feature Abort.";
    const char BusEventNameSetOnScreenStringFeatureAbort[] = "Set On Screen String Feature Abort";
    const char BusEventNameStarted[] = "Started";

    const char* BusEvent2String(BusEventType busEvent) NN_NOEXCEPT
    {
        const char* rval;

        switch(busEvent)
        {
            case BusEventType_Ignore:
                rval = BusEventNameIgnore;
                break;
            case BusEventType_ActiveSourceChangedToActive:
                rval = BusEventNameActiveSourceChangedToActive;
                break;
            case BusEventType_ActiveSourceChangedToInactive:
                rval = BusEventNameActiveSourceChangedToInactive;
                break;
            case BusEventType_GoStandby:
                rval = BusEventNameGoStandby;
                break;
            case BusEventType_Suspending:
                rval = BusEventNameSuspending;
                break;
            case BusEventType_ConnectionChange:
                rval = BusEventNameConnectionChange;
                break;
            case BusEventType_FeatureAbortOneTouchPlay:
                rval = BusEventNameOtpFeatureAbort;
                break;
            case BusEventType_FeatureAbortStandby:
                rval = BusEventNameStandbyFeatureAbort;
                break;
            case BusEventType_FeatureAbortSetOnScreenString:
                rval = BusEventNameSetOnScreenStringFeatureAbort;
                break;
            case BusEventType_Started:
                rval = BusEventNameStarted;
                break;
            default:
                rval = nullptr;
                break;
        }
        return rval;
    }

    const char PowerStateNameOn[] = "On";
    const char PowerStateNameStandby[] = "Standby";
    const char PowerStateNameGoingOn[] = "Going On";
    const char PowerStateNameGoingStandby[] = "Going Standby";

    const char* PowerState2String(PowerState powerState) NN_NOEXCEPT
    {
        const char* rval;

        switch(powerState)
        {
            case PowerState_On:
                rval = PowerStateNameOn;
                break;
            case PowerState_Standby:
                rval = PowerStateNameStandby;
                break;
            case PowerState_GoingOn:
                rval = PowerStateNameGoingOn;
                break;
            case PowerState_GoingStandby:
                rval = PowerStateNameGoingStandby;
                break;
            default:
                rval = nullptr;
                break;
        }
        return rval;
    }

    const char ConnectionStateNameNotConnected[] = "Not Connected";
    const char ConnectionStateNameOnlyCradleConnected[] = "Cradle Connected";
    const char ConnectionStateNameCradleAndTvConnected[] = "Cradle And TV Connected";

    const char* ConnectionState2String(ConnectionState connectionState) NN_NOEXCEPT
    {
        const char* rval;

        switch(connectionState)
        {
            case ConnectionState_NotConnected:
                rval = ConnectionStateNameNotConnected;
                break;
            case ConnectionState_OnlyCradleConnected:
                rval = ConnectionStateNameOnlyCradleConnected;
                break;
            case ConnectionState_CradleAndTvConnected:
                rval = ConnectionStateNameCradleAndTvConnected;
                break;
            default:
                rval = nullptr;
                break;
        }
        return rval;
    }

    bool    wait = false;
    char*   pArg2;
    int     arg2Value;
    bool    isNumericalArg;

    void    CecApiSequenceThread(void* pv) NN_NOEXCEPT
    {
        Result  result;
        CecCancelTestDataType*  pCancelTestData;
        bool    isSuccessfullyPinged;
        PowerState  powerState;
        const char* name;

        pCancelTestData = reinterpret_cast<CecCancelTestDataType*>(pv);

        os::SignalEvent(&pCancelTestData->event);
        result = RestartManager();
        NN_SDK_LOG("%s: RestartManager --> %d\n", __func__, result.GetDescription());
        os::SleepThread(TimeSpan::FromMilliSeconds(500));
        result = RestartManager();
        NN_SDK_LOG("%s: RestartManager --> %d\n", __func__, result.GetDescription());

        os::SignalEvent(&pCancelTestData->event);
        result = IsTvResponsive(&isSuccessfullyPinged);
        NN_SDK_LOG("%s: IsTvResponsive %d --> %d\n", __func__, isSuccessfullyPinged, result.GetDescription());
        os::SleepThread(TimeSpan::FromMilliSeconds(500));
        result = IsTvResponsive(&isSuccessfullyPinged);
        NN_SDK_LOG("%s: IsTvResponsive %d --> %d\n", __func__, isSuccessfullyPinged, result.GetDescription());
        os::SleepThread(TimeSpan::FromMilliSeconds(500));

        os::SignalEvent(&pCancelTestData->event);
        result = PerformOneTouchPlay();
        NN_SDK_LOG("%s: PerformOneTouchPlay --> %d\n", __func__, result.GetDescription());
        os::SleepThread(TimeSpan::FromMilliSeconds(500));
        result = PerformOneTouchPlay();
        NN_SDK_LOG("%s: PerformOneTouchPlay --> %d\n", __func__, result.GetDescription());

        os::SignalEvent(&pCancelTestData->event);
        result = GetTvPowerState(&powerState);
        name = PowerState2String(powerState);
        NN_SDK_LOG("%s: Tv Power State %s --> %d\n", __func__, name, result.GetDescription());
        os::SleepThread(TimeSpan::FromMilliSeconds(1500));
        result = GetTvPowerState(&powerState);
        name = PowerState2String(powerState);
        NN_SDK_LOG("%s: Tv Power State %s --> %d\n", __func__, name, result.GetDescription());
    }

    nn::Result  HandleApiCancelCommand() NN_NOEXCEPT
    {
        nn::settings::system::TvSettings    settings;
        CecCancelTestDataType               cancelTestData;
        os::ThreadType                      thread;
        bool                                isCanceling;

        os::InitializeEvent(&cancelTestData.event, false, os::EventClearMode_AutoClear);
        SuspendManager();
        nn::settings::system::GetTvSettings(&settings);
        settings.flags.Set<nn::settings::system::TvFlag::AllowsCec>();
        nn::settings::system::SetTvSettings(settings);

        os::CreateThread(&thread, CecApiSequenceThread, &cancelTestData,
                         s_CancelTestStack, s_CancelTestStackSize,
                         os::GetThreadPriority(os::GetCurrentThread()) + 1);
        os::StartThread(&thread);

        os::WaitEvent(&cancelTestData.event);
        os::SleepThread(TimeSpan::FromMilliSeconds(100));
        CancelCurrentApiCall(&isCanceling);
        NN_SDK_LOG("Past Restart: %d\n", isCanceling);

        os::WaitEvent(&cancelTestData.event);
        os::SleepThread(TimeSpan::FromMilliSeconds(1));
        CancelCurrentApiCall(&isCanceling);
        NN_SDK_LOG("Past Ping: %d\n", isCanceling);

        os::WaitEvent(&cancelTestData.event);
        os::SleepThread(TimeSpan::FromMilliSeconds(5));
        CancelCurrentApiCall(&isCanceling);
        NN_SDK_LOG("Past OTP: %d\n", isCanceling);

        os::WaitEvent(&cancelTestData.event);
        os::SleepThread(TimeSpan::FromMilliSeconds(1));
        CancelCurrentApiCall(&isCanceling);
        NN_SDK_LOG("Past Get TV power state: %d\n", isCanceling);

        os::WaitThread(&thread);
        os::DestroyThread(&thread);
        os::FinalizeEvent(&cancelTestData.event);
        NN_RESULT_SUCCESS;
    }
    nn::Result  HandleOtpCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return PerformOneTouchPlay();
    }
    nn::Result  HandleGoStandbyCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return PerformGoStandby(true);
    }
    nn::Result  HandleAllGoStandbyCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return PerformGoStandby(false);
    }
    nn::Result  HandleInactiveSourceCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return PerformInactiveSource();
    }
    nn::Result  HandleSetPowerStateOnCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return SetPowerState(PowerState_On);
    }
    nn::Result  HandleSetPowerStateGoingOnCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return SetPowerState(PowerState_GoingOn);
    }
    nn::Result  HandleSetPowerStateGoingStandbyCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return SetPowerState(PowerState_GoingStandby);
    }
    nn::Result  HandleSetPowerStateStandbyCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return SetPowerState(PowerState_Standby);
    }
    nn::Result  HandleIsActiveSourceCommand() NN_NOEXCEPT
    {
        bool        isActiveSource;
        nn::Result  result;

        result = IsActiveSource(&isActiveSource);
        NN_SDK_LOG("%s: %d\n", __func__, isActiveSource);
        return result;
    }
    nn::Result  HandleIsStartedCommand() NN_NOEXCEPT
    {
        bool        isStarted;
        nn::Result  result;

        result = IsStarted(&isStarted);
        NN_SDK_LOG("%s: %d\n", __func__, isStarted);
        return result;
    }
    nn::Result  HandleGetConnectionStateCommand() NN_NOEXCEPT
    {
        ConnectionState connectionState;
        nn::Result      result;

        result = GetConnectionState(&connectionState);
        if(result.GetDescription() == 0)
        {
            const char* name;

            name = ConnectionState2String(connectionState);
            NN_SDK_LOG("%s: %s\n", __func__, name);
        }
        return result;
    }
    nn::Result  HandleGetPowerStateCommand() NN_NOEXCEPT
    {
        PowerState  powerState;
        nn::Result  result;

        result = GetPowerState(&powerState);
        if(result.GetDescription() == 0)
        {
            const char* name;

            name = PowerState2String(powerState);
            NN_SDK_LOG("%s: %s\n", __func__, name);
        }
        return result;
    }
    nn::Result  HandleGetTvPowerStateCommand() NN_NOEXCEPT
    {
        PowerState  powerState;
        nn::Result  result;

        result = GetTvPowerState(&powerState);
        if(result.GetDescription() == 0)
        {
            const char* name;

            name = PowerState2String(powerState);
            NN_SDK_LOG("%s: %s\n", __func__, name);
        }
        return result;
    }
    nn::Result  HandlePingTvCommand() NN_NOEXCEPT
    {
        bool        isSuccessfullyPinged;
        nn::Result  result;

        result = IsTvResponsive(&isSuccessfullyPinged);
        if(result.GetDescription() == 0)
        {
            NN_SDK_LOG("%s: ping -> %d\n", __func__, isSuccessfullyPinged);
        }
        return result;
    }
    nn::Result  HandleIsEnabledCommand() NN_NOEXCEPT
    {
        bool        isEnabled;
        nn::Result  result;

        result = IsEnabled(&isEnabled);
        NN_SDK_LOG("%s: %d\n", __func__, isEnabled);
        return result;
    }
    nn::Result  HandleSuspendCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return SuspendManager();
    }
    nn::Result  HandleRestartCommand() NN_NOEXCEPT
    {
        NN_SDK_LOG("%s\n", __func__);
        return RestartManager();
    }
    nn::Result  HandleEnableCecCommand() NN_NOEXCEPT
    {
        bool        isEnabled = false;
        nn::Result  result;

        result = IsEnabled(&isEnabled);
        if(result.GetDescription() == 0)
        {
            if(isEnabled == false)
            {
                nn::settings::system::TvSettings    settings;

                nn::settings::system::GetTvSettings(&settings);
                settings.flags.Set<nn::settings::system::TvFlag::AllowsCec>();
                nn::settings::system::SetTvSettings(settings);
                result = IsEnabled(&isEnabled);
            }
        }
        NN_SDK_LOG("%s: %d\n", __func__, isEnabled);
        return result;
    }
    nn::Result  HandleDisableCecCommand() NN_NOEXCEPT
    {
        bool        isEnabled = false;
        nn::Result  result;

        result = IsEnabled(&isEnabled);
        if(result.GetDescription() == 0)
        {
            if(isEnabled == true)
            {
                nn::settings::system::TvSettings    settings;

                SuspendManager();
                nn::settings::system::GetTvSettings(&settings);
                settings.flags.Reset<nn::settings::system::TvFlag::AllowsCec>();
                nn::settings::system::SetTvSettings(settings);
                result = IsEnabled(&isEnabled);
            }
        }
        NN_SDK_LOG("%s: %d\n", __func__, isEnabled);
        return result;
    }
    nn::Result  HandleSetOnScreenStringCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        if(pArg2 != nullptr)
        {
            result = SetOnScreenString(pArg2);
            NN_SDK_LOG("%s: %s\n", __func__, pArg2);
        }
        else
        {
            result = ResultSuccess();
        }
        return result;
    }

    nn::Result  HandleRemoteControlCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        if(isNumericalArg)
        {
            result = PerformSendRemoteControlCommand(true, arg2Value);
            if(result.IsSuccess())
            {
                result = PerformSendRemoteControlCommand(false, arg2Value);
            }
            NN_SDK_LOG("%s: %s (%X)\n", __func__, pArg2, arg2Value);
        }
        else
        {
            result = ResultSuccess();
        }
        return result;
    }

    nn::Result  HandleVolumeUpCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        result = PerformSendRemoteControlCommand(true, RemoteControlVolumeUp);
        if(result.IsSuccess())
        {
            result = PerformSendRemoteControlCommand(false, 0);
        }
        return result;
    }

    nn::Result  HandleVolumeDownCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        result = PerformSendRemoteControlCommand(true, RemoteControlVolumeDown);
        if(result.IsSuccess())
        {
            result = PerformSendRemoteControlCommand(false, 0);
        }
        return result;
    }

    nn::Result  HandleVolumeMuteCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        result = PerformSendRemoteControlCommand(true, RemoteControlVolumeMute);
        if(result.IsSuccess())
        {
            result = PerformSendRemoteControlCommand(false, 0);
        }
        return result;
    }

    nn::Result  HandleDeviceMenuActivateCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        result = PerformSendDeviceMenuStateCommand(true);
        return result;
    }

    nn::Result  HandleDeviceMenuDeactivateCommand() NN_NOEXCEPT
    {
        nn::Result  result;

        result = PerformSendDeviceMenuStateCommand(false);
        return result;
    }

    nn::Result  HandleWaitArgument() NN_NOEXCEPT
    {
        wait = true;
        NN_SDK_LOG("Waiting For Event Post Command Completion\n");
        NN_RESULT_SUCCESS;
    }

    typedef nn::Result (* CommandHandlerFunctionPointerType)();

    typedef struct
    {
        const char* name;
        const int   nameLength;
        const CommandHandlerFunctionPointerType pfnCommand;
    } CommandEntryType;

    const char  nameCancelApi[] = "cancel";
    const char  nameOneTouchPlay[] = "otp";
    const char  nameGoStandby[] = "standby";
    const char  nameAllGoStandby[] = "standby_all";
    const char  nameInactiveSource[] = "inactive";
    const char  nameSetPowerStateOn[] = "set_power_state_on";
    const char  nameSetPowerStateGoingOn[] = "set_power_state_going_on";
    const char  nameSetPowerStateGoingStandby[] = "set_power_state_going_standby";
    const char  nameSetPowerStateStandby[] = "set_power_state_standby";
    const char  nameIsActiveSource[] = "is_active_source";
    const char  nameIsStarted[] = "is_started";
    const char  nameGetConnectionState[] = "get_connection_state";
    const char  nameGetPowerState[] = "get_power_state";
    const char  nameGetTvPowerState[] = "get_tv_power_state";
    const char  namePingTv[] = "ping_tv";
    const char  nameIsEnabled[] = "is_enabled";
    const char  nameSuspend[] = "suspend";
    const char  nameRestart[] = "restart";
    const char  nameEnableCec[] = "enable_cec";
    const char  nameDisableCec[] = "disable_cec";
    const char  nameSetOnScreenString[] = "set_osd";
    const char  nameTvVolumeUp[] = "tv_volume_up";
    const char  nameTvVolumeDown[] = "tv_volume_down";
    const char  nameTvVolumeMute[] = "tv_volume_mute";
    const char  nameRemoteControl[] = "remote_control";
    const char  nameDeviceMenuActive[] = "device_menu_active";
    const char  nameDeviceMenuInactive[] = "device_menu_inactive";
    const char  nameWait[] = "wait";

    const CommandEntryType  commands[] = {
        {nameCancelApi, sizeof(nameCancelApi) - 1, HandleApiCancelCommand},
        {nameOneTouchPlay, sizeof(nameOneTouchPlay) - 1, HandleOtpCommand},
        {nameAllGoStandby, sizeof(nameAllGoStandby) - 1, HandleAllGoStandbyCommand},
        {nameGoStandby, sizeof(nameGoStandby) - 1, HandleGoStandbyCommand},
        {nameInactiveSource, sizeof(nameInactiveSource) - 1, HandleInactiveSourceCommand},
        {nameSetPowerStateOn, sizeof(nameSetPowerStateOn) - 1, HandleSetPowerStateOnCommand},
        {nameSetPowerStateGoingOn, sizeof(nameSetPowerStateGoingOn) - 1, HandleSetPowerStateGoingOnCommand},
        {nameSetPowerStateGoingStandby, sizeof(nameSetPowerStateGoingStandby) - 1, HandleSetPowerStateGoingStandbyCommand},
        {nameSetPowerStateStandby, sizeof(nameSetPowerStateStandby) - 1, HandleSetPowerStateStandbyCommand},
        {nameIsActiveSource, sizeof(nameIsActiveSource) - 1, HandleIsActiveSourceCommand},
        {nameIsStarted, sizeof(nameIsStarted) - 1, HandleIsStartedCommand},
        {nameGetConnectionState, sizeof(nameGetConnectionState) - 1, HandleGetConnectionStateCommand},
        {nameGetPowerState, sizeof(nameGetPowerState) - 1, HandleGetPowerStateCommand},
        {nameGetTvPowerState, sizeof(nameGetTvPowerState) - 1, HandleGetTvPowerStateCommand},
        {namePingTv, sizeof(namePingTv) - 1, HandlePingTvCommand},
        {nameIsEnabled, sizeof(nameIsEnabled) - 1, HandleIsEnabledCommand},
        {nameSuspend, sizeof(nameSuspend) - 1, HandleSuspendCommand},
        {nameRestart, sizeof(nameRestart) - 1, HandleRestartCommand},
        {nameEnableCec, sizeof(nameEnableCec) - 1, HandleEnableCecCommand},
        {nameDisableCec, sizeof(nameDisableCec) - 1, HandleDisableCecCommand},
        {nameSetOnScreenString, sizeof(nameSetOnScreenString) - 1, HandleSetOnScreenStringCommand},
        {nameTvVolumeUp, sizeof(nameTvVolumeUp) - 1, HandleVolumeUpCommand},
        {nameRemoteControl, sizeof(nameRemoteControl) - 1, HandleRemoteControlCommand},
        {nameTvVolumeDown, sizeof(nameTvVolumeDown) - 1, HandleVolumeDownCommand},
        {nameTvVolumeMute, sizeof(nameTvVolumeMute) - 1, HandleVolumeMuteCommand},
        {nameDeviceMenuActive, sizeof(nameDeviceMenuActive) - 1, HandleDeviceMenuActivateCommand},
        {nameDeviceMenuInactive, sizeof(nameDeviceMenuInactive) - 1, HandleDeviceMenuDeactivateCommand}
    };
    const int   numberOfCommands = (sizeof(commands) / sizeof(CommandEntryType));

    const CommandEntryType  optionalCommands[] = {
        {nameWait, sizeof(nameWait) - 1, HandleWaitArgument},
    };
    const int   numberOfOptionalCommands = (sizeof(optionalCommands) / sizeof(CommandEntryType));

    void    ShowCommands() NN_NOEXCEPT
    {
        int index;

        NN_SDK_LOG("At least one argument required.  Recognized Commands Are:\n");
        for(index=0; index<numberOfCommands; index++)
        {
            NN_SDK_LOG("%s\n", commands[index].name);
        }
    }

    nn::Result  ProcessCommand(int argc, char* argv[]) NN_NOEXCEPT
    {
        nn::Result  result = ResultSuccess();

        if(argc >= 1)
        {
            int index;

            for(index=0; index<numberOfCommands; index++)
            {
                if(strncmp(argv[0], commands[index].name, commands[index].nameLength) == 0)
                {
                    result = commands[index].pfnCommand();
                    break;
                }
            }
        }
        return result;
    }

    nn::Result  ProcessOptionalArgument(int argc, char* argv[]) NN_NOEXCEPT
    {
        nn::Result  result = ResultSuccess();

        if(argc >= 1)
        {
            int index;

            for(index=0; index<numberOfOptionalCommands; index++)
            {
                if(strncmp(argv[0], optionalCommands[index].name, optionalCommands[index].nameLength) == 0)
                {
                    result = optionalCommands[index].pfnCommand();
                    break;
                }
            }
        }
        return result;
    }

    bool    IsNumericalOptionalArgument(int argc, char* argv[]) NN_NOEXCEPT
    {
        bool    rval = false;

        if(argc >= 1)
        {
            char*   endPtr;
            int argValue;

            argValue = ::std::strtol(pArg2, &endPtr, 0);
            if(endPtr != pArg2)
            {
                arg2Value = argValue;
                rval = true;
            }
        }
        return rval;
    }


extern "C" void nnMain()
{
    nn::os::SystemEventType systemEvent;
    int argc;
    char** argv;
    int index;

    argc = nn::os::GetHostArgc();
    argv = nn::os::GetHostArgv();
    NN_SDK_LOG("%s: test cec entry\n", argv[0]);
    NN_SDK_LOG("argc %d\n", argc);
    for(index=0; index<argc; index++)
    {
        NN_SDK_LOG("%d -> %s\n", index, argv[index]);
    }

    if(argc > 1)
    {
        nn::Result  result;

        Initialize(&systemEvent);
        NN_SDK_LOG("%s: initialized cec manager\n", __func__);

        pArg2 = (argc > 2) ? argv[2] : nullptr;
        isNumericalArg = IsNumericalOptionalArgument(argc - 2, &argv[2]);
        result = ProcessCommand(argc - 1, &argv[1]);
        NN_SDK_LOG("%s: %s --> %d\n", __func__, argv[1], result.GetDescription());

        if((!isNumericalArg) && (argc > 2))
        {
            ProcessOptionalArgument(argc - 2, &argv[2]);
        }

        if((wait == true) && (result.GetDescription() == 0))
        {
            BusEventType    busEvent;

            nn::os::WaitSystemEvent(&systemEvent);
            result = GetBusEventType(&busEvent);
            if(result.GetDescription() == 0)
            {
                const char* name;

                name = BusEvent2String(busEvent);
                if(busEvent == BusEventType_ConnectionChange)
                {
                    ConnectionState connectionState;

                    result = GetConnectionState(&connectionState);
                    if(result.GetDescription() == 0)
                    {
                        const char* connectionName;

                        connectionName = ConnectionState2String(connectionState);
                        NN_SDK_LOG("%s %s\n", name, connectionName);
                    }
                }
                else
                {
                    NN_SDK_LOG("%s\n", name);
                }
            }
        }

        Finalize(&systemEvent);
    }
    else
    {
        ShowCommands();
    }
    NN_SDK_LOG("Test App: %s: Leave\n", __func__);
}

}
}}
