<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        HDMI dock-undock test (SDEV only)

    .DESCRIPTION
        Verifies the HDMI works properly following dock-undock.
#>

# Pre-defined and custom windows commands
$WM_USER_TAKE_PICTURE = 0x0400 + 1
$WM_QUIT = 0x12

# Amount of time to wait in milliseconds for certain operations to complete
# The long wait is for any operation that should take the SDEV some amount of time to complete, such as docking or displaying after restarting
# The short wait is for any operation where the SDEV has already matched one picture, but it needs to render another frame to match the next picture
$LONG_WAIT = 30000
$SHORT_WAIT = 15000

# Since most iterations involve 3 or more pictures, if at least two of them fail, this will trigger after 30 iterations
# This number is leniant in case something fails in multiple ways
$MAX_FAILURE_LIMIT = 30

$externs = @"
[DllImport("user32.dll", EntryPoint="FindWindow", SetLastError = true)]
public static extern IntPtr FindWindowByClass(String lpClassName, IntPtr Zero);

[return: MarshalAs(UnmanagedType.Bool)]
[DllImport("user32.dll", SetLastError = true)]
public static extern bool PostThreadMessage(uint threadId, uint msg, UIntPtr wParam, IntPtr lParam);

[DllImport("user32.dll")]
public static extern uint GetWindowThreadProcessId(IntPtr hWnd, IntPtr ProcessId);
"@

function Initialize-Consts
{
    $script:scriptRootPath      = "$env:NINTENDO_SDK_ROOT"
    $script:toolsDirectoryPath  = "${scriptRootPath}\Tools\CommandLineTools"

    if ($scriptRootPath -eq "")
    {
        Write-Error "Error: NINTENDO_SDK_ROOT not set"
        Write-Error "Aborting"
        exit 1
    }

    if ($ArtifactPath -eq "")
    {
        $script:ArtifactPath = ${scriptDirectoryPath}
    }

    $script:ControlTargetExe           = "${toolsDirectoryPath}\ControlTarget.exe"
    $script:ControlTargetPrivateExe    = "${toolsDirectoryPath}\ControlTargetPrivate.exe"
    $script:RunOnTargetExe             = "${toolsDirectoryPath}\RunOnTarget.exe"

    $script:InitLog                    = "${ArtifactPath}\init.log"
    $script:CleanUpLog                 = "${ArtifactPath}\cleanup.log"
}

function Init-Transition-Test([string]$scriptDirectoryPath, [string]$outputFile, [string]$targetAddress)
{
    Initialize-Consts
    
    $script:Functions = Add-Type -Namespace Win32 -Name Funcs -MemberDefinition $externs -PassThru

    $script:transitionFailureCount = 0
    $script:failure = 0
    $script:totalStopwatch = [diagnostics.stopwatch]::StartNew()
    Remove-Item $InitLog -ErrorAction SilentlyContinue

###############################################################################################
    Write-Host "Artifact Path: ${script:ArtifactPath}"
    Write-Host "Target Address: $targetAddress"
    Write-Host "Transition: ${script:TransitionTest}"

###############################################################################################
    Write-Host "Starting capture..."
    $script:captureProcess = [diagnostics.process]::start("${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DisplayCapture\Develop\DisplayCapture.exe", "-outputFile ${outputFile}")

    # Get the thread ID so we can send messages
    $script:threadId = $script:captureProcess.Threads[0].Id

###############################################################################################
    Write-Host "Initiating Serial Logging..."
    $ScriptBlock =
    {
        param($artifactPath, $targetAddress)
        
        $Socket = New-Object System.Net.Sockets.TcpClient($targetAddress, "10023")
        $Stream = $Socket.GetStream()
        $Reader = New-Object System.IO.StreamReader($Stream)
        
        Remove-Item "${artifactPath}\serial.log" -ErrorAction SilentlyContinue
        
        while (1) 
        {
            $line = $Reader.ReadLine()
            
            # Remove the color information
            $line = $line -replace '\e\[\d\d?m', ''
            
            $line >> "${artifactPath}\serial.log"
        }
    }

    $SerialJob = Start-Job $ScriptBlock -ArgumentList $ArtifactPath, $targetAddress

###############################################################################################
    # This needs to be done early enough for the SDEV to notice the change
    Write-Host "Connecting HDMI and resetting input EDID to output..."
    Write-HDMI-Switcher "s 1"
    Write-HDMI-Switcher "#set_edid output 1 input 1"
    
###############################################################################################
    Write-Host "Restarting SDEV..."

    & $ControlTargetExe reset >> $InitLog

    & $ControlTargetExe connect >> $InitLog

###############################################################################################
    Write-Host "Formatting SD card..."

    & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -- sdcard format >> $InitLog

    Write-Host "Initiating SD card logging..."

    & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -- debug enable-sd-card-logging >> $InitLog

###############################################################################################
    Write-Host "Restarting SDEV..."

    & $ControlTargetExe reset >> $InitLog

    & $ControlTargetExe connect >> $InitLog

###############################################################################################
    Write-Host "Redocking SDEV..."
    $ret = & $ControlTargetExe enable-cradle

###############################################################################################
    Write-Host "Setting display to ${script:startRes}p with no underscan..."

    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution ${script:startRes} -underscan 0 -update_underscan -update_display -update_nand

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${script:startRes}_1" "r1a_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${script:startRes}_2" "r1b_" $LONG_WAIT
}

function Cleanup-Transition-Test([string]$scriptDirectoryPath)
{
    Write-Host "Cleaning up..."
    Remove-Item $CleanUpLog -ErrorAction SilentlyContinue

    Write-Host "Disabling SD card logging..."

    $ret = & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -- debug disable-sd-card-logging

###############################################################################################
    Write-Host "Restarting SDEV..."

    & $ControlTargetExe reset >> $CleanUpLog

    & $ControlTargetExe connect >> $CleanUpLog

###############################################################################################
    Write-Host "Collecting logs..."

    & $RunOnTargetExe "${scriptRootPath}/Tests/Outputs/NX-NXFP2-a64/Tests/GetSdCardFiles/Develop/GetSdCardFiles.nsp" -- --read_dir=sd:/NxBinLogs --write_dir=${ArtifactPath} >> $CleanUpLog
    & "${scriptRootPath}\Tests\Outputs\AnyCPU\Tools\ParseNxBinLogs\Release\ParseNxBinLogs.exe" "${ArtifactPath}" >> $CleanUpLog
    
###############################################################################################
    Write-Host "Closing capture..."
    $ret = $Functions::PostThreadMessage($threadId, $WM_QUIT, [UIntPtr]::Zero, [IntPtr]::Zero)

    Wait-Process -id $script:captureProcess.Id -ErrorAction SilentlyContinue

###############################################################################################
    $totalSeconds = $script:totalStopwatch.elapsed.TotalSeconds
    Write-Host "Finished in $totalSeconds seconds"
}

function Write-HDMI-Switcher([string]$command)
{
    $Socket = New-Object System.Net.Sockets.TcpClient("192.168.1.72", "23")
    $Stream = $Socket.GetStream()
    $Writer = New-Object System.IO.StreamWriter($Stream)
    
    $Writer.WriteLine($command)
    $Writer.Flush()
    $Writer.Close()
    
    # To prevent HDMI Switcher bugs, wait a second after closing the connection
    Start-Sleep -s 1
}

function Get-SDEV-Serial([string]$ipAddress)
{
    $Socket = New-Object System.Net.Sockets.TcpClient($ipAddress, "10023")
    $Stream = $Socket.GetStream()
    $Reader = New-Object System.IO.StreamReader($Stream)
    
    return $Reader
}

function Copy-No-Report([string]$source, [string]$folder, [string]$destination, [string]$name)
{
    if (Test-Path($source))
    {
        Copy-Item $source -Destination "${folder}/${destination}"
    }
}

function Copy-And-Report([string]$source, [string]$folder, [string]$destination, [string]$name)
{
    if (Test-Path($source))
    {
        Copy-Item $source -Destination "${folder}/${destination}"
        Write-Host "Copied $name to $destination"
    }
}
    
function Verify-Picture([string]$folder, [string]$out, [string]$name, [string]$unique)
{
    $golden = "${folder}\golden_${name}.png"
    $mask = "${folder}\mask_${name}.png"
    $diff = "${folder}\diff_${name}.png"
    
    $timeoutTime = [TimeSpan]::FromMilliseconds(5000)
    $stopwatch = [diagnostics.stopwatch]::StartNew()
    
    # Delete the old bitmap in the event there is one
    Remove-Item $out -ErrorAction SilentlyContinue

    Write-Host "Taking picture..."
    $ret = $script:Functions::PostThreadMessage($script:threadId, $WM_USER_TAKE_PICTURE, [UIntPtr]::Zero, [IntPtr]::Zero)
        
    # Wait for the bitmap to be created
    $ret = Test-Path $out -ErrorAction SilentlyContinue
    while ($ret -eq 0 -and $stopwatch.elapsed -lt $timeoutTime)
    {
        Start-Sleep -s 0.02
        $ret = Test-Path $out -ErrorAction SilentlyContinue
    }

    Write-Host "Verifying picture..."
        
    # If there's no mask, don't use one
    $ret = Test-Path $mask
    if ($ret -ne 0)
    {
        & "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -inputFile $out -goldenFile $golden -maskFile $mask -diffFile $diff
    }
    else
    {
        & "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -inputFile $out -goldenFile $golden -diffFile $diff
    }
    
    if ($lastexitcode -ne 0)
    {
        "***DIFF FAILED FOR $golden"
        $script:failure = 1
        
        if ($ArtifactPath -ne "")
        {
            # Copy all the bitmaps to the artifact folder
            Copy-And-Report $golden "${ArtifactPath}" "${unique}golden_${name}.png" "golden"
            Copy-And-Report $out "${ArtifactPath}" "${unique}out_${name}.png" "out"
            Copy-And-Report $mask "${ArtifactPath}" "${unique}mask_${name}.png" "mask"
            Copy-And-Report $diff "${ArtifactPath}" "${unique}diff_${name}.png" "diff"
        }
        else
        {
            # Make a copy of all the bitmaps with unique names
            Copy-And-Report $golden "${folder}" "${unique}golden_${name}.png" "golden"
            Copy-And-Report $out "${folder}" "${unique}out_${name}.png" "out"
            Copy-And-Report $mask "${folder}" "${unique}mask_${name}.png" "mask"
            Copy-And-Report $diff "${folder}" "${unique}diff_${name}.png" "diff"
        }
    }
    
    # Delete the output bitmap and diff bitmap as needed
    Remove-Item $out -ErrorAction SilentlyContinue
    Remove-Item $diff -ErrorAction SilentlyContinue
}

function Wait-Picture([string]$folder, [string]$out, [string]$name, [string]$unique, [int]$timeout)
{
    $golden = "${folder}\golden_${name}.png"
    $mask = "${folder}\mask_${name}.png"
    $diff = "${folder}\diff_${name}.png"
    
    $minPictures = $timeout / 500
    $timeoutExtraTime = [TimeSpan]::FromMilliseconds($timeout + 30000)
    $timeoutTime = [TimeSpan]::FromMilliseconds($timeout)
    $stopwatch = [diagnostics.stopwatch]::StartNew()
    
    # Write-Host "Waiting ${timeout} milliseconds for $golden to appear..."
    
    $attempts = 0
    while (1)
    {
        if ($stopwatch.elapsed -gt $timeoutTime -and $attempts -ge $minPictures)
        {
            "***TIMED OUT WAITING FOR golden_${name}.png (${attempts} attempts)"
            $script:failure = 1
        
            if ($ArtifactPath -ne "")
            {
                # Copy all the bitmaps to the artifact folder
                Copy-And-Report $golden "${ArtifactPath}" "${unique}golden_${name}.png" "golden"
                Copy-And-Report $mask "${ArtifactPath}" "${unique}mask_${name}.png" "mask"
                Copy-And-Report $diff "${ArtifactPath}" "${unique}diff_${name}.png" "diff"
                
                for ($i = 0; $i -lt $attempts; $i++)
                {
                    Copy-No-Report "${out}${i}" "${ArtifactPath}" "${unique}out${i}_${name}.png" "out${i}"
                }
                
                Write-Host "Copied $attempts out picture(s) to ${ArtifactPath}"
            }
            else
            {
                # Make a copy of all the bitmaps with unique names
                Copy-And-Report $golden "${folder}" "${unique}golden_${name}.png" "golden"
                Copy-And-Report $mask "${folder}" "${unique}mask_${name}.png" "mask"
                Copy-And-Report $diff "${folder}" "${unique}diff_${name}.png" "diff"
                
                for ($i = 0; $i -lt $attempts; $i++)
                {
                    Copy-No-Report "${out}${i}" "${folder}" "${unique}out${i}_${name}.png" "out${i}"
                }
                
                Write-Host "Copied $attempts out picture(s) to ${folder}"
            }
            
            # Check to see if it's a black screen
            & "${scriptDirectoryPath}\..\..\..\..\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $tempOut -goldenFile "${folder}\golden_no_signal480.png" -diffFile $diff
            $error480 = $lastexitcode
            & "${scriptDirectoryPath}\..\..\..\..\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $tempOut -goldenFile "${folder}\golden_no_signal720.png" -diffFile $diff
            $error720 = $lastexitcode
            & "${scriptDirectoryPath}\..\..\..\..\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $tempOut -goldenFile "${folder}\golden_no_signal1080.png" -diffFile $diff
            $error1080 = $lastexitcode
            
            if ($error480 -eq 0 -or $error720 -eq 0 -or $error1080 -eq 0)
            {
                Write-Host "Note: Black screen instead of expected output.  Possibly SIGLO-67107?"
            }
            
            ++$script:transitionFailureCount
            
            if ($script:transitionFailureCount -ge $MAX_FAILURE_LIMIT)
            {
                Write-Host "TOO MANY CONSECUTIVE FAILURES OCCURRED!  ABORTING"
                Cleanup-Transition-Test $scriptDirectoryPath
                
                Write-Host "FAILURES OCCURRED"
                exit 1
            }
            
            break
        }
    
        # Delete the old bitmap in the event there is one
        Remove-Item $out -ErrorAction SilentlyContinue

        $ret = $script:Functions::PostThreadMessage($script:threadId, $WM_USER_TAKE_PICTURE, [UIntPtr]::Zero, [IntPtr]::Zero)
        
        # Wait for the bitmap to be created (a little extra time is given for the timeout here)
        $ret = Test-Path $out -ErrorAction SilentlyContinue
        while ($ret -eq 0 -and $stopwatch.elapsed -lt $timeoutExtraTime)
        {
            Start-Sleep -s 0.02
            $ret = Test-Path $out -ErrorAction SilentlyContinue
        }
        
        if ($ret -ne 0)
        {
            $tempOut = "${out}${attempts}"
            Remove-Item $tempOut -ErrorAction SilentlyContinue
            Rename-Item $out -NewName $tempOut
            
            # If there's no mask, don't use one
            $ret = Test-Path $mask
            if ($ret -ne 0)
            {
                & "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $tempOut -goldenFile $golden -maskFile $mask -diffFile $diff
            }
            else
            {
                & "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $tempOut -goldenFile $golden -diffFile $diff
            }
        
            if ($lastexitcode -eq 0)
            {
                $attempts = $attempts + 1
            
                if (${attempts} -eq 1)
                {
                    Write-Host "  (matched ${name} immediately)"
                }
                else
                {
                    Write-Host "  (matched ${name} after ${attempts} attempts)"
                }
        
                # Slowly decrease the failure count on successful pictures
                if ($script:transitionFailureCount -gt 0)
                {
                    --$script:transitionFailureCount
                }
                break
            }
        }
        
        $attempts = $attempts + 1
    }
        
    # Delete the output bitmaps and diff bitmap as needed
                
    for ($i = 0; $i -lt $attempts; $i++)
    {
        Remove-Item "${out}${i}" -ErrorAction SilentlyContinue
    }
    Remove-Item $diff -ErrorAction SilentlyContinue
}

function Wait-Picture-Negative([string]$folder, [string]$out, [string]$name, [int]$timeout)
{
    $golden = "${folder}\golden_${name}.png"
    $mask = "${folder}\mask_${name}.png"
    
    $minPictures = $timeout / 500
    $timeoutExtraTime = [TimeSpan]::FromMilliseconds($timeout + 30000)
    $timeoutTime = [TimeSpan]::FromMilliseconds($timeout)
    $stopwatch = [diagnostics.stopwatch]::StartNew()
    
    # Write-Host "Waiting ${timeout} milliseconds for $golden to appear..."
    
    $attempts = 0
    while (1)
    {
        if ($stopwatch.elapsed -gt $timeoutTime -and $attempts -ge $minPictures)
        {
            Write-Host "  (picture did not appear)"
            break
        }
    
        # Delete the old bitmap in the event there is one
        Remove-Item $out -ErrorAction SilentlyContinue

        $ret = $script:Functions::PostThreadMessage($script:threadId, $WM_USER_TAKE_PICTURE, [UIntPtr]::Zero, [IntPtr]::Zero)
        
        # Wait for the bitmap to be created (a little extra time is given for the timeout here)
        $ret = Test-Path $out -ErrorAction SilentlyContinue
        while ($ret -eq 0 -and $stopwatch.elapsed -lt $timeoutExtraTime)
        {
            Start-Sleep -s 0.02
            $ret = Test-Path $out -ErrorAction SilentlyContinue
        }
        
        if ($ret -ne 0)
        {
            Rename-Item $out -NewName $tempOut
            
            # If there's no mask, don't use one
            $ret = Test-Path $mask
            if ($ret -ne 0)
            {
                & "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $out -goldenFile $golden -maskFile $mask
            }
            else
            {
                & "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\DiffTool\Develop\DiffTool.exe" -silent -inputFile $out -goldenFile $golden
            }
        }
        
        $attempts = $attempts + 1
        
        if ($lastexitcode -eq 0)
        {
            "***MATCHED BAD PICTURE AFTER ${attempts} ATTEMPT(S)"
            $script:failure = 1
        
            break
        }
    }
        
    # Delete the output bitmap as needed
    Remove-Item "${out}${i}" -ErrorAction SilentlyContinue
}
