﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nvntutorial/TutorialBaseClass.h>
#include <nvntutorial/MemoryPool.h>
#include <nvntutorial/TutorialUtil.h>

#include <nn/nn_Abort.h>
#include <nn/audio.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/vi.private.h>

#include <nn/oe.h>
#include <nn/oe/oe_HdcpApis.private.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

//////////////////////////////////////

// Threads for various parts
const size_t ThreadStackSize = 32 * 1024;

NN_OS_ALIGNAS_THREAD_STACK char g_HdcpStateThreadStack[ThreadStackSize];
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

nn::os::ThreadType  g_HdcpStateThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_AudioThread;

//////////////////////////////////////

static volatile bool    g_hdcpAuthenticated = false;

static const size_t     g_CommandMemorySize = 64 * 1024;
static const size_t     g_ControlMemorySize = 64 * 1024;
static const int        g_NumColorBuffers = 2;

static const int        g_NumColors = 3;
static int              g_Frame = 0;

char g_HeapBuffer[128 * 1024];

const double PI = 3.14159265358979;

//
// This sounds terrible, but it's a very clear tone
//
void GenerateSineWave(void* buffer, int sampleRate, int frequency, int sampleCount, int amplitude)
{
    NN_ASSERT_NOT_NULL(buffer);
    static int s_TotalSampleCount = 0;

    int16_t* buf = reinterpret_cast<int16_t*>(buffer);
    double increment = 2 * PI * frequency / static_cast<double>(sampleRate);

    for (int sample = 0; sample < sampleCount; sample++)
    {
        int16_t value = static_cast<int16_t>(sin(s_TotalSampleCount * increment) * amplitude);
        buf[sample] = value;
        s_TotalSampleCount++;
    }
}

class GettingStartedWithNVN : public TutorialBaseClass
{
    public:
        GettingStartedWithNVN() ;
        virtual ~GettingStartedWithNVN() ;
        virtual void Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow);
        virtual void Shutdown();

        virtual void Draw(uint64_t millisec) ;
        virtual void Resize(int width, int height) ;

    private:
        static void NVNAPIENTRY DebugLayerCallback(
            NVNdebugCallbackSource source,
            NVNdebugCallbackType type,
            int id,
            NVNdebugCallbackSeverity severity,
            const char* message,
            void* pUser
            ) ;

        int UpdateRenderTargets() ;

        NVNdevice         m_Device;
        NVNqueue          m_Queue;

        MemoryPool*       m_pCommandMemoryPool;

        ptrdiff_t         m_CommandPoolOffset;
        void*             m_pControlPool;
        NVNcommandBuffer  m_CommandBuffer;
        NVNcommandHandle  m_CommandHandle[g_NumColors];
        NVNcommandHandle  m_CommandHandleNoHDCP;

        ptrdiff_t         m_RenderTargetCommandPoolOffset;
        void*             m_pRenderTargetControlPool;
        NVNcommandBuffer  m_RenderTargetCommandBuffer;
        NVNcommandHandle  m_RenderTargetCommandHandle;

        NVNtextureBuilder m_RenderTargetBuilder;
        NVNtexture*       m_RenderTargets[g_NumColorBuffers];

        NVNsync           m_CommandBufferSync;

        MemoryPool*       m_pRenderTargetMemoryPool;

        NVNwindow*        m_pWindow;
        NVNwindowBuilder  m_WindowBuilder;
        size_t            m_ColorTargetSize;
};

GettingStartedWithNVN::GettingStartedWithNVN() :
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pWindow(NULL)
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }

    for (int i = 0; i < g_NumColors; ++i)
    {
        m_CommandHandle[i] = NULL;
    }

    m_CommandHandleNoHDCP = NULL;
}

GettingStartedWithNVN::~GettingStartedWithNVN()
{
}

void GettingStartedWithNVN::Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(loader != NULL, "Bootstrap loader function pointer is NULL\n");

    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*loader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*loader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

    nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress );

    int deviceFlags = 0;
#ifdef _DEBUG
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT | NVN_DEVICE_FLAG_DEBUG_SKIP_CALLS_ON_ERROR_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress );

    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    if(nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

    if(!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    if(!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(
        NULL,
        Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        &m_Device);

    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    if(!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    if(!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "nvnSyncInitialize");
    }

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

    nvnTextureBuilderSetFlags (&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
}

void GettingStartedWithNVN::Shutdown()
{
    nvnQueueFinish(&m_Queue);

    nvnSyncFinalize(&m_CommandBufferSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if(m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);
    nvnCommandBufferFinalize(&m_CommandBuffer);

    if(m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if(m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if (m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_RenderTargets[i])
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if(m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    nvnQueueFinalize(&m_Queue);

    nvnDeviceFinalize(&m_Device);
}

void GettingStartedWithNVN::Resize(int width, int height)
{
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);

    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (!m_RenderTargets[i])
        {
            m_RenderTargets[i] = new NVNtexture;
        }
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

        nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder);
    }

    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_RenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

    const int TileSize = 100;

    const float   Red[4] = { 1.0f, 0.0f, 0.0f, 1.0f };
    const float Green[4] = { 0.0f, 1.0f, 0.0f, 1.0f };
    const float  Blue[4] = { 0.0f, 0.0f, 1.0f, 1.0f };

    const float* ColorList[g_NumColors] = { Red, Green, Blue };

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    for (int i = 0; i < g_NumColors; ++i)
    {
        nvnCommandBufferBeginRecording(&m_CommandBuffer);

        for (int y = 0; y < height; y += TileSize)
        {
            for (int x = 0; x < width; x += TileSize)
            {
                int subWidth = (x + TileSize > width ? width - x : TileSize);
                int subHeight = (y + TileSize > height ? height - y : TileSize);

                nvnCommandBufferSetScissor(&m_CommandBuffer, x, y, subWidth, subHeight);
                nvnCommandBufferClearColor(&m_CommandBuffer, 0, ColorList[((x + y) / TileSize + i) % g_NumColors], NVN_CLEAR_COLOR_MASK_RGBA);
            }
        }

        m_CommandHandle[i] = nvnCommandBufferEndRecording(&m_CommandBuffer);
    }

    // Without HDCP, it's all red
    nvnCommandBufferBeginRecording(&m_CommandBuffer);

    nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, width, height);
    nvnCommandBufferClearColor(&m_CommandBuffer, 0, Red, NVN_CLEAR_COLOR_MASK_RGBA);

    m_CommandHandleNoHDCP = nvnCommandBufferEndRecording(&m_CommandBuffer);
}

void GettingStartedWithNVN::Draw(uint64_t)
{
    int index = UpdateRenderTargets();

    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

    // Continually monitor HDCP and only display the proper colors if it's working
    if (g_hdcpAuthenticated)
    {
        nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle[(g_Frame / 30) % g_NumColors]);
    }
    else
    {
        nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandleNoHDCP);
    }

    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);
}

int GettingStartedWithNVN::UpdateRenderTargets()
{
    int bufferIndex;

    nvnQueueAcquireTexture(&m_Queue, m_pWindow, &bufferIndex);

    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_RenderTargets[bufferIndex], NULL, NULL, NULL);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

    return bufferIndex;
}

void GettingStartedWithNVN::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "NVN Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static GettingStartedWithNVN tut;
    return (&tut);
}

void EndFrameCallbackFunction(bool* pOutExitLoop, const CallbackFunctionType::EndFrameCallbackParameter* param, void*)
{
    g_Frame = param->currentFrame;
}

///////////////////////////////////////////////////////////////////////////////
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));

    nn::audio::AudioOut audioOut;

    int frequency = 1000;
    int amplitude = 3500;

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    for(int i = 0; i < argc; ++i)
    {
        if(strcmp("-frequency", argv[i]) == 0)
        {
            NN_ASSERT(i < argc, "Expected argument after frequency!");
            frequency = atoi(argv[i + 1]);

            ++i;
            continue;
        }

        if(strcmp("-amplitude", argv[i]) == 0)
        {
            NN_ASSERT(i < argc, "Expected argument after amplitude!");
            amplitude = atoi(argv[i + 1]);

            ++i;
            continue;
        }
    }

    // Open the default audio (there's only one on the device)
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.channelCount = 2;
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsFailure())
    {
        parameter.sampleRate = 0;
        parameter.channelCount = 0;
        NN_ABORT_UNLESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsSuccess(), "Failed to open AudioOut.");
    }

    // Some constants
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);

    const int frameRate = 20;                             // 20fps
    const int frameSampleCount = sampleRate / frameRate;  // 50msecs (in samples)
    const size_t dataSize = frameSampleCount * nn::audio::GetSampleByteSize(nn::audio::SampleFormat_PcmInt16);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    const int bufferCount = 4;

    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    void* outBuffer[bufferCount];
    for (int i = 0; i < bufferCount; ++i)
    {
        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        GenerateSineWave(outBuffer[i], sampleRate, frequency, frameSampleCount, amplitude);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }

    // Start playing the sound
    NN_ABORT_UNLESS(nn::audio::StartAudioOut(&audioOut).IsSuccess(), "Failed to start playback.");

    NN_LOG("Start audio playback with sample rate = %d Hz\n", sampleRate);

    while (1)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

        if (!g_hdcpAuthenticated)
        {
            nn::audio::StopAudioOut(&audioOut);

            while (!g_hdcpAuthenticated)
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

            // Restart playing the sound
            NN_ABORT_UNLESS(nn::audio::StartAudioOut(&audioOut).IsSuccess(), "Failed to start playback.");
        }

        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        while (pAudioOutBuffer)
        {
            // Append to the audio buffer
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
            NN_ASSERT(outSize == frameSampleCount * nn::audio::GetSampleByteSize(nn::audio::SampleFormat_PcmInt16));
            GenerateSineWave(pOutBuffer, sampleRate, frequency, frameSampleCount, amplitude);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);

            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
void HdcpStateThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    nn::os::SystemEvent hdcpAuthenticationStateChangeEvent;
    nn::oe::GetHdcpAuthenticationStateChangeEvent(&hdcpAuthenticationStateChangeEvent);

    g_hdcpAuthenticated = (nn::oe::GetHdcpAuthenticationState() == nn::oe::HdcpAuthenticationState_Authenticated);

    while (1)
    {
        hdcpAuthenticationStateChangeEvent.Wait();
        g_hdcpAuthenticated = (nn::oe::GetHdcpAuthenticationState() == nn::oe::HdcpAuthenticationState_Authenticated);
    }
}

///////////////////////////////////////////////////////////////////////////////
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    CallbackFunctionType cbFunctions;
    cbFunctions.endFrameCallbackFunction = EndFrameCallbackFunction;
    cbFunctions.endFrameCallbackUserParam = NULL;
    TutorialRun(&cbFunctions);
}

void SetDisplayMode(int resolution)
{
    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::vi::OpenDisplay( &pDisplay, "External" ) );

    nn::vi::DisplayModeInfo modes[ nn::vi::DisplayModeCountMax ];
    int modeCount = nn::vi::ListDisplayModes( modes, sizeof( modes ) / sizeof( modes[ 0 ] ), pDisplay );

    NN_ASSERT(modeCount > 0);

    nn::vi::DisplayModeInfo mode = modes[0];

    switch (resolution)
    {
    case 480:
        mode.width = 720;
        mode.height = 480;
        break;
    case 720:
        mode.width = 1280;
        mode.height = 720;
        break;
    case 1080:
        mode.width = 1920;
        mode.height = 1080;
        break;
    default:
        for ( int i = 1; i < modeCount; ++i )
        {
            if ( modes[i].width > mode.width )
            {
                mode = modes[i];
            }
        }
        break;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::SetDisplayMode( pDisplay, &mode ));

    nn::vi::CloseDisplay(pDisplay);
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    int setResolution = -1;

    for(int i = 0; i < argc; ++i)
    {
        if (!strcmp(argv[i], "-set_resolution"))
        {
            NN_ASSERT(i + 1 < argc);

            setResolution = atoi(argv[i + 1]);
            continue;
        }
    }

    nn::oe::Initialize();

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) + 1 ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) + 1) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_HdcpStateThread, HdcpStateThreadFunction, NULL, g_HdcpStateThreadStack, ThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) + 1) );

    nn::os::StartThread( &g_HdcpStateThread );

    if (setResolution != -1)
    {
        while (!g_hdcpAuthenticated)
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        SetDisplayMode(setResolution);
    }

    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_AudioThread );

    // Run for a maximum of 5 minutes
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(300));
}
