﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VideoConfigUtility.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <binder/Parcel.h>
#include <hardware/hwcomposer_defs.h>
#include <hos/nvdisp_output.h>
#include <cstdio>
#include <string>

namespace nns {

namespace {

const int DisplayContextHead = 1;

}

NN_IMPLICIT VideoConfigUtility::VideoConfigUtility() NN_NOEXCEPT
:
m_Video(android::SurfaceComposerClient::getBuiltInDisplay(android::ISurfaceComposer::eDisplayIdHdmi)),
m_Client(new android::SurfaceComposerClient()),
m_NvdcHandle(nvdcOpen(NvRm_MemmgrGetIoctlFile())),
m_NvdcUtilHandle(nvdcutilOpen())
{
    if (nullptr == m_NvdcHandle)
    {
        NN_LOG("Warning: m_NvdcHandle is nullptr.\n");
    }
    if (nullptr == m_NvdcUtilHandle)
    {
        NN_LOG("Warning: m_NvdcUtilHandle is nullptr.\n");
    }
}

VideoConfigUtility::~VideoConfigUtility() NN_NOEXCEPT
{
    if (nullptr != m_NvdcUtilHandle)
    {
        nvdcutilClose(m_NvdcUtilHandle);
    }
    if (nullptr != m_NvdcHandle)
    {
        nvdcClose(m_NvdcHandle);
    }
}

bool VideoConfigUtility::ShowDisplayInfo() NN_NOEXCEPT
{
    auto getAspectRatioString = [](int aspectRatioIndex) {
        static std::string buffer;
        buffer = "";
        if (HWC_DISPLAY_ASPECT_RATIO_4_3 & aspectRatioIndex)
        {
            buffer += "4:3,";
        }
        if (HWC_DISPLAY_ASPECT_RATIO_16_9 & aspectRatioIndex)
        {
            buffer += "16:9,";
        }
        if (HWC_DISPLAY_ASPECT_RATIO_64_27 & aspectRatioIndex)
        {
            buffer += "64:27,";
        }
        if (HWC_DISPLAY_ASPECT_RATIO_256_135 & aspectRatioIndex)
        {
            buffer += "256:135,";
        }
        if (0 < buffer.size())
        {
            buffer = buffer.substr(0, buffer.size() - 1);
        }
        return buffer.c_str();
    };
    auto getDisplayModeString = [](int displayModeIndex) {
        static std::string buffer;
        buffer = "";
        if (HWC_DISPLAY_MODE_RGB & displayModeIndex)
        {
            buffer += "RGB,";
        }
        if (HWC_DISPLAY_MODE_YUV420 & displayModeIndex)
        {
            buffer += "YUV420,";
        }
        if (HWC_DISPLAY_MODE_YUV420_ONLY & displayModeIndex)
        {
            buffer += "YUV420_ONLY,";
        }
        if (HWC_DISPLAY_MODE_IS_CEA & displayModeIndex)
        {
            buffer += "CEA,";
        }
        if (HWC_DISPLAY_MODE_IS_DETAILED & displayModeIndex)
        {
            buffer += "DETAILED,";
        }
        if (HWC_DISPLAY_MODE_IS_HDMI_EXT & displayModeIndex)
        {
            buffer += "HDMI_EXT,";
        }
        if (0 < buffer.size())
        {
            buffer = buffer.substr(0, buffer.size() - 1);
        }
        return buffer.c_str();
    };

    int activeConfig = android::SurfaceComposerClient::getActiveConfig(m_Video);
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(0 == status);
    NN_LOG("===============================================================================================\n");
    NN_LOG("The number of EXTERNAL DISPLAY INFOs:[%d]\n", configs.size());
    if (0 < configs.size())
    {
        NN_LOG("===============================================================================================\n");
        for (int i1=0; i1<configs.size(); ++i1)
        {
            const android::DisplayInfo& info = configs[i1];
            NN_LOG("%s [%2d/%2u] resolution:[%4dx%4d] fps:[%2.5f] aspect_ratio:[%s]@[%d] display_mode:[%s]@[%d]\n",
                (activeConfig == i1) ? "*" : " ", i1, configs.size(),
                info.w, info.h, info.fps,
                getAspectRatioString(info.aspectRatioFlag), info.aspectRatioFlag,
                getDisplayModeString(info.modeFlags), info.modeFlags);
        }
        NN_LOG("===============================================================================================\n");
        return true;
    }
    NN_LOG("No external displays...\n");
    NN_LOG("===============================================================================================\n");
    return false;
}

bool VideoConfigUtility::SetActiveDisplay(int id) NN_NOEXCEPT
{
    android::status_t status = android::SurfaceComposerClient::setActiveConfig(m_Video, id);
    return android::NO_ERROR == status;
}

int VideoConfigUtility::GetActiveDisplayId() NN_NOEXCEPT
{
    return android::SurfaceComposerClient::getActiveConfig(m_Video);
}

android::DisplayInfo VideoConfigUtility::GetDisplayInfo(int id) NN_NOEXCEPT
{
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(android::NO_ERROR == status);
    return configs[id];
}

int VideoConfigUtility::GetDisplayInfoCount() NN_NOEXCEPT
{
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(android::NO_ERROR == status);
    return configs.size();
}

bool VideoConfigUtility::SetVirtualEdid(const char* edidPath) NN_NOEXCEPT
{
    if (nullptr == m_NvdcUtilHandle)
    {
        NN_LOG("Warning: m_NvdcUtilHandle is nullptr.\n");
        return false;
    }

    if (nvdcutilVirtualEdidEnable(m_NvdcUtilHandle, DisplayContextHead, edidPath))
    {
        NN_LOG("nvdcutilVirtualEdidEnable failed. No such file:[%s]\n", edidPath);
        return false;
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    if (nvdcutilSwHotplugIn(m_NvdcUtilHandle, DisplayContextHead))
    {
        NN_LOG("sw hotplug in fail !\n");
        return false;
    }

    // Wait for upper s/w stack construct modeDB.
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

    struct nvdcutilMode mode;
    if (nvdcutilGetMode(m_NvdcUtilHandle, DisplayContextHead, &mode))
    {
        NN_LOG("getmode fail !\n");
        return false;
    }
    return true;
}

bool VideoConfigUtility::ResetVirtualEdid() NN_NOEXCEPT
{
    if (nvdcutilVirtualEdidDisable(m_NvdcUtilHandle, DisplayContextHead))
    {
        NN_LOG("nvdcutilVirtualEdidDisable fail !\n");
        return false;
    }
    if (nvdcutilSwHotplugOut(m_NvdcUtilHandle, DisplayContextHead))
    {
        NN_LOG("sw hotplug out fail !\n");
        return false;
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    return true;
}

}
