﻿using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace TargetUtility
{
    class Runner
    {
        private static int IncrementalCountValue = 0;
        private static int IncrementalCount
        {
            get { return Interlocked.Increment(ref IncrementalCountValue); }
        }

        public class RunApplicationResult
        {
            public int ExitCode;
            public string MixedOutputString;
            public string StandardOutputString;
            public string StandardErrorString;
        }

        private static string StripInvalidFileNameChars(string text)
        {
            foreach (var c in Path.GetInvalidFileNameChars())
            {
                text = text.Replace(c.ToString(), "_");
            }
            return text.Replace(" ", "_").Replace("__", "_");
        }

        private static string MakeLogFileName(string fileName, string arguments)
        {
            var baseName = $"{IncrementalCount:00}_{Path.GetFileNameWithoutExtension(fileName)}_{StripInvalidFileNameChars(arguments)}";
            const int baseNameLengthMax = 80;
            if (baseName.Length > baseNameLengthMax)
            {
                baseName = baseName.Substring(0, baseNameLengthMax / 2) + baseName.Substring(baseName.Length - baseNameLengthMax / 2);
            }
            return  Path.Combine(EnvironmentInfo.LogDirectory, $"{baseName}.log");
        }

        public static RunApplicationResult RunApplication(string fileName, string arguments)
        {
            return RunApplicationImpl(fileName, arguments, null);
        }

        public static RunApplicationResult RunApplication(string fileName, string arguments, TimeSpan timeout)
        {
            return RunApplicationImpl(fileName, arguments, timeout);
        }

        private static RunApplicationResult RunApplicationImpl(
            string fileName, string arguments, TimeSpan? timeout)
        {
            using (var process = new Process())
            {
                process.StartInfo.FileName = fileName;
                process.StartInfo.Arguments = arguments;
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardOutput = true;
                process.StartInfo.RedirectStandardError = true;

                var sbForMixed = new StringBuilder();
                var sbForStandardOutput = new StringBuilder();
                var sbForStandardError = new StringBuilder();

                process.OutputDataReceived += (sender, e) =>
                {
                    sbForStandardOutput.AppendLine(e.Data);
                    lock (sbForMixed) sbForMixed.AppendLine(e.Data);
                };
                process.ErrorDataReceived += (sender, e) =>
                {
                    sbForStandardError.AppendLine(e.Data);
                    lock (sbForMixed) sbForMixed.AppendLine(e.Data);
                };

                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();

                var isTimedOut = !process.WaitForExit((int)(timeout?.TotalMilliseconds ?? int.MaxValue));
                if (isTimedOut)
                {
                    process.Kill();
                }

                using (var sw = new StreamWriter(MakeLogFileName(fileName, arguments), false, new UTF8Encoding(false)))
                {
                    sw.WriteLine($"FileName: {fileName}");
                    sw.WriteLine($"Arguments: {arguments}");
                    sw.WriteLine($"ExitCode: {(isTimedOut ? "none (timeout)" : process.ExitCode.ToString())}");
                    sw.WriteLine(new string('-', Console.WindowWidth - 1));
                    sw.Write(sbForMixed.ToString());
                }

                if (isTimedOut)
                {
                    throw new TimeoutException(
                        $"Timeout to run Application\n" +
                        $"FileName={fileName}\n" +
                        $"Arguments={arguments}\n" +
                        new string('-', Console.WindowWidth - 1) + "\n" +
                        sbForMixed.ToString());
                }

                return new RunApplicationResult()
                {
                    ExitCode = process.ExitCode,
                    MixedOutputString = sbForMixed.ToString(),
                    StandardOutputString = sbForStandardOutput.ToString(),
                    StandardErrorString = sbForStandardError.ToString()
                };
            }
        }
    }
}
