﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  I/O マッピングアドレスの取得 API のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Log.h>
#include <nn/dd.h>

#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "test_IoMappingAddress-os.horizon.h"
#endif

namespace nnt { namespace dd { namespace ioMapAddress {

//----------------------------------------------------------------------------
// 非 I/O な物理アドレスを返すテスト補助関数。
// Win32 環境では 0 を返し、Horizon 環境では明らかに DDR メモリな
// 物理アドレスを返す。
//
nn::dd::PhysicalAddress GetNonIoPhysAddress()
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    return 0;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::dd::PhysicalAddress padrs;
    nn::Result result = nn::dd::QuerySinglePhysicalAddress(&padrs, &padrs, 1);
    EXPECT_TRUE( result.IsSuccess() );
    return padrs;
#endif
}

//----------------------------------------------------------------------------
//
void PrintJudgement( bool result )
{
    if (result)
    {
        NN_LOG(" ... OK\n");
        SUCCEED();
    }
    else
    {
        NN_LOG(" ... NG\n");
        ADD_FAILURE();
    }
}


//----------------------------------------------------------------------------
// 非 I/O な物理アドレスを返す。
// Win32 環境では 0 を返し、Horizon 環境では明らかに DDR メモリな
// 物理アドレスを返す。
//
TEST(QueryIoMappingAddress, QueryIoMappingAddress)
{
    NN_LOG(NN_TEXT("I/O アドレス変換関連のテスト開始\n"));

    NN_LOG(NN_TEXT("I/O ではない物理アドレスを渡すと 0 を返すか\n"));
    {
        nn::dd::PhysicalAddress nonIoPhysAddress = GetNonIoPhysAddress();
        uintptr_t vadrs = nn::dd::QueryIoMappingAddress(nonIoPhysAddress, 1);
        NN_LOG("PhysicalAddress = 0x%016llx\n", nonIoPhysAddress);
        NN_LOG("VirtualAddress  = 0x%p", vadrs);
        PrintJudgement( vadrs == 0 );
        NN_LOG("\n");
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    NN_LOG(NN_TEXT("I/O 物理アドレスに対する返値の確認\n"));
#if 0   // ここの #if 0 はカーネルでの固定マップが削除されたら
        // テストとしてパスするようになるはず。
    {
        NN_LOG(NN_TEXT("I/O 物理アドレス -1 への Query に失敗するか\n"));
        const nn::dd::PhysicalAddress   padrs = TargetIoPhysicalAddress - 1;
        const size_t                    size  = TargetIoSize;
        uintptr_t vadrs = nn::dd::QueryIoMappingAddress(padrs, size);
        NN_LOG("PhysicalAddress = 0x%016llx  size=0x%08x\n", padrs, size);
        NN_LOG("VirtualAddress  = 0x%p", vadrs);
        PrintJudgement( vadrs == 0 );
        NN_LOG("\n");
    }
#endif
    {
        NN_LOG(NN_TEXT("I/O 物理アドレスへの Query に成功するか\n"));
        const nn::dd::PhysicalAddress   padrs = TargetIoPhysicalAddress;
        const size_t                    size  = TargetIoSize;
        uintptr_t vadrs = nn::dd::QueryIoMappingAddress(padrs, size);
        NN_LOG("PhysicalAddress = 0x%016llx  size=0x%08x\n", padrs, size);
        NN_LOG("VirtualAddress  = 0x%p", vadrs);
        PrintJudgement( vadrs != 0 );
        NN_LOG("\n");
    }
#if 0   // ここの #if 0 はカーネルでの固定マップが削除されたら
        // テストとしてパスするようになるはず。
    {
        NN_LOG(NN_TEXT("I/O 物理アドレス +1 への Query に失敗するか\n"));
        const nn::dd::PhysicalAddress   padrs = TargetIoPhysicalAddress + 1;
        const size_t                    size  = TargetIoSize;
        uintptr_t vadrs = nn::dd::QueryIoMappingAddress(padrs, size);
        NN_LOG("PhysicalAddress = 0x%016llx  size=0x%08x\n", padrs, size);
        NN_LOG("VirtualAddress  = 0x%p", vadrs);
        PrintJudgement( vadrs == 0 );
        NN_LOG("\n");
    }
#endif
#endif

    NN_LOG(NN_TEXT("I/O アドレス変換関連のテスト終了\n"));
}


#if 0
//  SIGLO-69191
//  アクセス可能なレジスタ範囲が、LSI 仕様に強く依存している。
//  本来はカーネルの svc 仕様および実装に追従し続ける必要があるが、
//  API の動作チェックとしては、一定の役割を終えているため、
//  このテスト自身を一旦無効化しておく。

#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
//----------------------------------------------------------------------------
// MC（メモリコントローラ）への I/O レジスタアクセス
//
// アクセス可能なMCレジスタのビット
const nn::Bit8 g_McAccessible[] =
{
    0x00, 0x00, 0x20, 0x00, 0xF0, 0xFF, 0xF7, 0x01, // 0x000 - 0x0fc
    0xCD, 0xFE, 0xC0, 0xFE, 0x00, 0x00, 0x00, 0x00, // 0x100 - 0x1fc
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6E, // 0x200 - 0x2fc
    0x30, 0x05, 0x06, 0xB0, 0x71, 0xC8, 0x43, 0x04, // 0x300 - 0x3fc
    0x80, 0x1F, 0x08, 0x80, 0x03, 0x00, 0x0E, 0x00, // 0x400 - 0x4fc
    0x08, 0x00, 0xE0, 0x00, 0x0E, 0x00, 0x00, 0x00, // 0x500 - 0x5fc
    0x00, 0x00, 0x00, 0x30, 0xF0, 0x03, 0x03, 0x30, // 0x600 - 0x6fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x700 - 0x7fc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x800 - 0x8fc
    0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x00, // 0x900 - 0x9fc
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xa00 - 0xafc
    0x00, 0x00, 0x00, 0x00, 0x0C, 0x00, 0xFE, 0x0F, // 0xb00 - 0xbfc
    0x01, 0x00, 0x80, 0x00, 0x00, 0x08, 0x00, 0x00, // 0xc00 - 0xcfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xd00 - 0xdfc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xe00 - 0xefc
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  // 0xf00 - 0xffc
};

TEST(ioRegisterAccess, test_ReadIoRegister)
{
    nn::Bit8* accessible = const_cast<nn::Bit8*>(g_McAccessible);
    nn::dd::PhysicalAddress address = 0x70019000;

    NN_LOG("\nTest of nn::dd:ReadIoRegister()\n\n");
    for (int i=0; i<0x1000; i+=0x20)
    {
        NN_LOG("0x%p:", address);
        nn::Bit8 pattern = *accessible++;
        for (int j=0; j<0x20; j+=0x4)
        {
            if (pattern & 0x1)
            {
                auto data = nn::dd::ReadIoRegister(address);
                NN_LOG(" 0x%08x", data);
            }
            else
            {
                NN_LOG(" ----------");
            }
            pattern >>= 1;
            address += 0x4;
        }
        NN_LOG("\n");
    }
    NN_LOG("\n");
}

TEST(ioRegisterAccess, test_WriteIoRegister)
{
    nn::Bit8* accessible = const_cast<nn::Bit8*>(g_McAccessible);
    nn::dd::PhysicalAddress address = 0x70019000;

    NN_LOG("\nTest of nn::dd:WriteIoRegister()\n\n");
    for (int i=0; i<0x1000; i+=0x20)
    {
        NN_LOG("0x%p:", address);
        nn::Bit8 pattern = *accessible++;
        for (int j=0; j<0x20; j+=0x4)
        {
            if (pattern & 0x1)
            {
                auto data = nn::dd::ReadIoRegister(address);
                nn::dd::WriteIoRegister(address, data);
                NN_LOG(" 0x%08x", data);
            }
            else
            {
                NN_LOG(" ----------");
            }
            pattern >>= 1;
            address += 0x4;
        }
        NN_LOG("\n");
    }
    NN_LOG("\n");
}

TEST(ioRegisterAccess, test_ReadModifyWriteIoRegister)
{
    nn::Bit8* accessible = const_cast<nn::Bit8*>(g_McAccessible);
    nn::dd::PhysicalAddress address = 0x70019000;

    NN_LOG("\nTest of nn::dd:ReadModifyWriteIoRegister()\n\n");
    for (int i=0; i<0x1000; i+=0x20)
    {
        NN_LOG("0x%p:", address);
        nn::Bit8 pattern = *accessible++;
        for (int j=0; j<0x20; j+=0x4)
        {
            if (pattern & 0x1)
            {
                auto data = nn::dd::ReadIoRegister(address);
                auto data1 = nn::dd::ReadModifyWriteIoRegister(address, data, 0x55555555);
                auto data2 = nn::dd::ReadModifyWriteIoRegister(address, data, 0xaaaaaaaa);
                auto data3 = nn::dd::ReadModifyWriteIoRegister(address, data, 0x00000000);
                auto data4 = nn::dd::ReadModifyWriteIoRegister(address, data, 0xffffffff);
                NN_LOG(" 0x%08x", data);

                // レジスタによっては値が変化するためチェックしない
                NN_UNUSED(data1);       //EXPECT_EQ(data, data1);
                NN_UNUSED(data2);       //EXPECT_EQ(data, data2);
                NN_UNUSED(data3);       //EXPECT_EQ(data, data3);
                EXPECT_EQ(0,    data4);
            }
            else
            {
                NN_LOG(" ----------");
            }
            pattern >>= 1;
            address += 0x4;
        }
        NN_LOG("\n");
    }
    NN_LOG("\n");
}
#endif
#endif

}}} // namespace nnt::dd::ioMapAddress

