﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  キャッシュ操作 API のテスト
//---------------------------------------------------------------------------

#include <cstring>

#include "../Common/test_Pragma.h"

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/nn_SdkText.h>

#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/dd/dd_Result.h>

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#endif

namespace nnt { namespace dd { namespace cacheApi {

const int bufferSize = 8 * 1024 * 1024;
char buffer[ bufferSize ];

// API が呼び出せるかどうかのみのテスト
TEST(CacheApi, CallTest)
{
    // 自プロセスのキャッシュ制御
    nn::dd::InvalidateDataCache( buffer, bufferSize );
    nn::dd::StoreDataCache( buffer, bufferSize );
    nn::dd::FlushDataCache( buffer, bufferSize );

    // 他プロセスのキャッシュ制御
    auto current = nn::dd::GetCurrentProcessHandle();
    auto address = reinterpret_cast<uint64_t>(buffer);
    auto size    = static_cast<uint64_t>(bufferSize);
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // Win 環境では常に ResultNotSupported() が返る
    auto result = nn::dd::InvalidateProcessDataCache( current, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultNotSupported() );

    result = nn::dd::StoreProcessDataCache( current, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultNotSupported() );

    result = nn::dd::FlushProcessDataCache( current, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultNotSupported() );
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    // Success 系
    auto result = nn::dd::InvalidateProcessDataCache( current, address, size );
    EXPECT_TRUE( result.IsSuccess() );

    result = nn::dd::StoreProcessDataCache( current, address, size );
    EXPECT_TRUE( result.IsSuccess() );

    result = nn::dd::FlushProcessDataCache( current, address, size );
    EXPECT_TRUE( result.IsSuccess() );

    // 無効なハンドルを渡す
    const auto invalid = nn::os::InvalidNativeHandle;
    result = nn::dd::InvalidateProcessDataCache( invalid, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultInvalidHandle() );

    result = nn::dd::StoreProcessDataCache( invalid, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultInvalidHandle() );

    result = nn::dd::FlushProcessDataCache( invalid, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultInvalidHandle() );

    // 無効なアドレス空間を指定する
    address = 0ull;
    size    = 128ull;
    result = nn::dd::InvalidateProcessDataCache( current, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultInvalidMemoryState() );

    result = nn::dd::StoreProcessDataCache( current, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultInvalidMemoryState() );

    result = nn::dd::FlushProcessDataCache( current, address, size );
    EXPECT_TRUE( result <= nn::dd::ResultInvalidMemoryState() );
#endif
}

// 各API でパフォーマンスを計測
TEST(CacheApi, PerformanceMeasurement)
{
    const int MemoryFillLoopNumber = 10;
    nn::os::Tick startTick;
    nn::os::Tick duration;

    // メモリ書き込みのみ
    startTick = nn::os::GetSystemTick();
    for (int i=0; i<MemoryFillLoopNumber; i++)
    {
        std::memset(buffer, 0x11, bufferSize);
    }
    duration = nn::os::GetSystemTick() - startTick;
    NN_LOG(NN_TEXT("メモリ書き込みのみの場合 %lld us\n"), nn::os::ConvertToTimeSpan( duration ).GetMicroSeconds());

    // Invalidate + メモリ書き込み
    startTick = nn::os::GetSystemTick();
    for (int i=0; i<MemoryFillLoopNumber; i++)
    {
        nn::dd::InvalidateDataCache( buffer, bufferSize );
        std::memset(buffer, 0x11, bufferSize);
    }
    duration = nn::os::GetSystemTick() - startTick;
    NN_LOG(NN_TEXT("Invalidate + メモリ書き込みの場合 %lld us\n"), nn::os::ConvertToTimeSpan( duration ).GetMicroSeconds());

    // メモリ書き込み + Store
    startTick = nn::os::GetSystemTick();
    for (int i=0; i<MemoryFillLoopNumber; i++)
    {
        std::memset(buffer, 0x11, bufferSize);
        nn::dd::StoreDataCache( buffer, bufferSize );
    }
    duration = nn::os::GetSystemTick() - startTick;
    NN_LOG(NN_TEXT("メモリ書き込み + Store の場合 %lld us\n"), nn::os::ConvertToTimeSpan( duration ).GetMicroSeconds());

    // メモリ書き込み + Flush
    startTick = nn::os::GetSystemTick();
    for (int i=0; i<MemoryFillLoopNumber; i++)
    {
        std::memset(buffer, 0x11, bufferSize);
        nn::dd::FlushDataCache( buffer, bufferSize );
    }
    duration = nn::os::GetSystemTick() - startTick;
    NN_LOG(NN_TEXT("メモリ書き込み + Flush の場合 %lld us\n"), nn::os::ConvertToTimeSpan( duration ).GetMicroSeconds());
}


#if defined( NN_BUILD_CONFIG_OS_HORIZON )

// svc の Flush 命令の引数を nn::dd::FlushDataCache に合わせる
void FlushDataCacheViaSvc(const void* buffer, size_t size)
{
    nn::svc::FlushProcessDataCache(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, reinterpret_cast<uintptr_t>( buffer ), size);
}

// svc の Store 命令の引数を nn::dd::StoreDataCache に合わせる
void StoreDataCacheViaSvc(const void* buffer, size_t size)
{
    nn::svc::StoreProcessDataCache(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, reinterpret_cast<uintptr_t>( buffer ), size);
}

// svc の EntireFlush 命令の引数を nn::dd::FlushDataCache に合わせる
void FlushEntireDataCacheViaSvc(const void* buffer, size_t size)
{
    NN_UNUSED(buffer);
    NN_UNUSED(size);
    nn::svc::FlushEntireDataCache();
}

// キャッシュ操作にかかる時間を計測
void PerformanceTestPerSize(void (*testFunc)(const void*, size_t), const char* testType)
{
    const int MemoryFillLoopNumber = 1000;
    nn::os::Tick startTick;
    nn::os::Tick duration;

    // User Flush
    for (size_t size=64; size <= bufferSize; size *= 2)
    {
        // 計測中に余計な領域が Flush されないようにキャッシュ全体を Flush しておく
        nn::svc::FlushEntireDataCache();

        // buffer を Flush しておく
        std::memset(buffer, 0x11, size);
        nn::dd::FlushDataCache( buffer, size );

        // すでに Flush 済みの領域を何度も Flush して時間を計測する
        startTick = nn::os::GetSystemTick();
        for (int i=0; i<MemoryFillLoopNumber; i++)
        {
            testFunc( buffer, size );
        }
        duration = nn::os::GetSystemTick() - startTick;
        NN_LOG(NN_TEXT("%s size=%d: %lld us\n"), testType, size, nn::os::ConvertToTimeSpan( duration ).GetMicroSeconds());
    }
}

// キャッシュ操作 API ごとにぞれそれパフォーマンスを計測
TEST(CacheApi, PerformanceMeasurementPerSize)
{
    PerformanceTestPerSize(nn::dd::FlushDataCache,     "User Flush");
    PerformanceTestPerSize(FlushDataCacheViaSvc,       "SVC Flush");
    PerformanceTestPerSize(nn::dd::StoreDataCache,     "User Store");
    PerformanceTestPerSize(StoreDataCacheViaSvc,       "SVC Store");
    PerformanceTestPerSize(FlushEntireDataCacheViaSvc, "SVC Entire Flush");
}

#endif

}}} // namespace nnt::dd::cacheApi
