﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/crypto/crypto_RsaPssSigner.h>
#include <nn/crypto/crypto_RsaPssVerifier.h>
#include <nn/crypto/crypto_RsaPssSha256Signer.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>

/*
Some command lines used to debug SIGLO-19745 :

N="4D9jmR0xTu2VZXUqoK5vHmndF+tVcg63cBYnYMzB6bHJMooAEzG/WckkXzTMEtXvR3NfMafG8nf0IFMg1HrOVTQqklQ6CTt1AJbGfmnUkao3Ds/9CDltO1iaYqV3SCEwt1OXaRk4Qy6oJNUPBoJ/ikXw1wpks1p2xBaJ1ZNIkdU="
echo "$N" | base64 -d | xxd -c 16 -i >>net.cpp

gcc test.cpp -lcrypto && ./a.exe >netPriv.pem

printf "obase=2;%d\n" 0x22889211 | bc

*/
#if 0
#include <openssl/bn.h>
#include <openssl/rsa.h>
#include <openssl/pem.h>

static void BN_dump(const char* what, const BIGNUM* bn)
{
    char* str = BN_bn2hex(bn);
    printf("%s : %s\n", what, str);
    OPENSSL_free(str);
}

int err(const char* msg)
{
    fprintf(stderr, "%s\n", msg);
    return 1;
}

int main(int argc, char* argv[])
{
    RSA* rsa = RSA_new();
    if (!rsa)
        return err("RSA_new() failed");
    rsa->n = BN_bin2bn(gKeyPublicN, sizeof(gKeyPublicN), NULL);
    rsa->e = BN_bin2bn(gKeyPublicE, sizeof(gKeyPublicE), NULL);
    rsa->d = BN_bin2bn(gKeyPrivateD, sizeof(gKeyPrivateD), NULL);
    rsa->p = BN_bin2bn(binP, sizeof(binP), NULL);
    rsa->q = BN_bin2bn(binQ, sizeof(binQ), NULL);
    rsa->dmp1 = BN_bin2bn(binDP, sizeof(binDP), NULL);
    rsa->dmq1 = BN_bin2bn(binDQ, sizeof(binDQ), NULL);
    rsa->iqmp = BN_bin2bn(binIQ, sizeof(binIQ), NULL);

    if (RSA_check_key(rsa) != 1)
        err("Key check failed");

    BIGNUM* bnOaep = BN_bin2bn(binOaep, sizeof(binOaep), NULL);
    BIGNUM* bnZero = BN_bin2bn(binZero, sizeof(binZero), NULL);
    BIGNUM* bnOne = BN_bin2bn(binOne, sizeof(binOne), NULL);

    struct bignum_ctx* ctx = BN_CTX_new();
    BN_CTX_start(ctx);
    struct bignum_st* bnRes = BN_CTX_get(ctx);
    BN_mod_exp(bnRes, bnOne, rsa->d, rsa->n, ctx);
    BN_dump("bnRes", bnRes);
    BN_dump("bnOaep", bnOaep);
    BN_dump("rsa->d", rsa->d);
    BN_dump("rsa->n", rsa->n);
    BN_CTX_end(ctx);
    BN_CTX_free(ctx);

    int ok = PEM_write_RSAPrivateKey(stdout, rsa, NULL, NULL,0, NULL,NULL);
    //int ok = PEM_write_RSA_PUBKEY(stdout, rsa);
    if (!ok)
        return err("Key export failed");
    return 0;
}

#endif

static const unsigned char gKeyPublicN[] =
{
    0xad, 0x19, 0x47, 0x6e, 0x9f, 0xa2, 0x4f, 0xda, 0xbd, 0xe4, 0xa8, 0x82, 0x58, 0x83, 0xe5, 0xa1,
    0x44, 0xbe, 0xd7, 0xf9, 0x67, 0x5c, 0xdc, 0x1f, 0x42, 0x69, 0xb1, 0x84, 0xce, 0xf1, 0xa0, 0x09,
    0xad, 0xef, 0x3f, 0xb1, 0x6b, 0x14, 0xa7, 0x8a, 0x88, 0x7c, 0xbd, 0x4d, 0xad, 0x3a, 0x4f, 0x49,
    0xde, 0xad, 0xdd, 0x05, 0xfd, 0xb1, 0x42, 0x0f, 0x17, 0x68, 0xe7, 0xfe, 0x3f, 0x7c, 0x30, 0x51,
    0x53, 0xfc, 0xde, 0xda, 0x55, 0x36, 0x67, 0x5b, 0x6b, 0xe8, 0xe3, 0xda, 0x36, 0x57, 0xaf, 0x08,
    0xd3, 0x30, 0xed, 0xd6, 0x02, 0xf8, 0xee, 0x44, 0xe4, 0x42, 0x19, 0xfb, 0x25, 0x36, 0x18, 0xb1,
    0x5e, 0x24, 0x34, 0x92, 0xbf, 0xa3, 0x14, 0x36, 0xa3, 0x6b, 0x2a, 0xb2, 0x3d, 0x08, 0x14, 0xc6,
    0x58, 0x2f, 0x10, 0x58, 0x5b, 0xab, 0x12, 0xab, 0xe0, 0x65, 0x0e, 0x43, 0x6f, 0x90, 0x00, 0x03,
    0x64, 0x83, 0x6d, 0x5b, 0xd0, 0xda, 0x62, 0x43, 0xa1, 0x2d, 0xde, 0xcf, 0x57, 0x34, 0x36, 0x5d,
    0xd9, 0x20, 0x2e, 0x04, 0x78, 0x1f, 0x7d, 0xb0, 0xaa, 0x5f, 0xed, 0x15, 0xc7, 0x11, 0xc5, 0xfc,
    0xf6, 0xc4, 0x1e, 0xa2, 0x57, 0xf7, 0x5a, 0x4e, 0x9b, 0x3a, 0x46, 0xa3, 0xdc, 0xc4, 0x57, 0xc7,
    0x7e, 0xcd, 0x67, 0x60, 0xf3, 0x40, 0x78, 0x59, 0x95, 0x15, 0x64, 0xd1, 0x13, 0x3d, 0x4d, 0xb2,
    0xdd, 0xbc, 0x68, 0x27, 0x5d, 0x6e, 0x4c, 0xaa, 0x6c, 0x26, 0x44, 0xde, 0x81, 0x9b, 0x90, 0xf5,
    0xf5, 0x40, 0xad, 0x29, 0x3e, 0x65, 0xe0, 0x26, 0x4f, 0x3d, 0x20, 0x9c, 0xac, 0x04, 0x6e, 0x64,
    0xe3, 0x26, 0x26, 0x15, 0x9e, 0x39, 0xd2, 0x65, 0x39, 0x9a, 0xb7, 0xbc, 0x57, 0x73, 0x7d, 0x09,
    0x5d, 0xb5, 0x76, 0x6e, 0x39, 0x06, 0xd9, 0xfd, 0x53, 0xce, 0x43, 0xd0, 0x2f, 0x95, 0x9c, 0x61
};
static const unsigned char gKeyPublicE[] =
{
    0x01,0x00,0x01
};

static const unsigned char gKeyPrivateD[] =
{
    0x08, 0x24, 0x97, 0x4a, 0xa9, 0xf1, 0xcd, 0xc5, 0xe1, 0x86, 0xff, 0x32, 0x5a, 0x8d, 0xa1, 0xe0,
    0xcb, 0x46, 0x1b, 0x0c, 0xdd, 0x82, 0xed, 0xcb, 0xfe, 0xa8, 0xd0, 0x34, 0x0a, 0x34, 0x2d, 0xae,
    0x26, 0xe3, 0x89, 0x60, 0x84, 0x44, 0x0d, 0x79, 0xa3, 0xe0, 0x12, 0x5b, 0xb4, 0x19, 0x79, 0x1f,
    0x65, 0x7d, 0x52, 0x57, 0x74, 0xce, 0xc0, 0xe7, 0xca, 0xd7, 0x21, 0x8a, 0xc1, 0x5b, 0xcb, 0x3e,
    0x18, 0x30, 0x5e, 0x1f, 0x71, 0x40, 0xc4, 0xed, 0xf5, 0x0e, 0x98, 0xa1, 0x6c, 0x4b, 0x05, 0xad,
    0xe9, 0x99, 0x74, 0xa6, 0xcb, 0xaf, 0x00, 0x2e, 0x77, 0x7a, 0x70, 0x94, 0x72, 0xf4, 0xaf, 0x86,
    0xb0, 0x82, 0x0d, 0x83, 0x15, 0xc6, 0xcb, 0x2d, 0xa2, 0xd4, 0x68, 0xc6, 0xc7, 0xbe, 0x38, 0x93,
    0xf8, 0x14, 0x06, 0xb5, 0xd1, 0x62, 0x23, 0xd7, 0xb8, 0x0f, 0xa8, 0xf3, 0x1d, 0x2d, 0x6e, 0x02,
    0xa7, 0x00, 0x6c, 0xc3, 0x17, 0xe3, 0xa9, 0x4c, 0x84, 0x58, 0xc0, 0x48, 0xce, 0x92, 0x16, 0x2a,
    0x8b, 0x8f, 0x73, 0xb5, 0x6b, 0x20, 0x6a, 0xcd, 0xa0, 0x77, 0x64, 0x81, 0xf4, 0x93, 0x96, 0x36,
    0xfc, 0xe8, 0x70, 0x2c, 0x23, 0xbb, 0xc5, 0xa4, 0x79, 0x1e, 0x4a, 0x70, 0x3f, 0xa8, 0x44, 0x12,
    0x4d, 0x77, 0x27, 0x63, 0x57, 0x07, 0xbf, 0x41, 0x79, 0x75, 0x88, 0xbb, 0x3a, 0x17, 0x4d, 0x95,
    0xd6, 0xdd, 0x5e, 0x48, 0xc6, 0x85, 0x0d, 0x6c, 0xe7, 0x87, 0x0f, 0x5d, 0x4a, 0xee, 0xd1, 0xd9,
    0x08, 0xc4, 0xbb, 0xc0, 0x5c, 0xc0, 0x6d, 0x53, 0x41, 0xf9, 0xfa, 0x8c, 0x10, 0x1f, 0xff, 0x28,
    0xfa, 0x56, 0x0c, 0xd9, 0x75, 0x5a, 0x68, 0xb9, 0x99, 0x9c, 0x9c, 0x97, 0xe8, 0x60, 0xf6, 0xc8,
    0x59, 0x40, 0x47, 0xfd, 0xbf, 0x34, 0xea, 0xab, 0x9d, 0xe2, 0x83, 0x2b, 0x22, 0x88, 0x92, 0x11
};

static const unsigned char gMessage[] = {
    0x4e, 0x43, 0x41, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0xcd, 0x06, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xbf, 0xee, 0xe9, 0x18, 0xc0, 0xfb, 0x75, 0xa7, 0xec, 0xb0, 0x8a, 0x87, 0xf6, 0xb9, 0x45, 0x7e,
    0xce, 0xaf, 0x6c, 0x9d, 0xf5, 0xd1, 0x3b, 0x31, 0x73, 0x39, 0x05, 0x2a, 0x10, 0x1f, 0x49, 0x56,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x89, 0xa4, 0x66, 0x1f, 0x01, 0x99, 0x8d, 0xab, 0xb4, 0x0f, 0x7e, 0x44, 0x73, 0x29, 0xc3, 0x05,
    0x9c, 0xf1, 0xf4, 0x44, 0xd9, 0x81, 0x7c, 0x1b, 0x5a, 0x8d, 0x42, 0xd0, 0xc5, 0xe6, 0xbe, 0x12,
    0x1c, 0xa7, 0xda, 0xe8, 0x31, 0xde, 0xe7, 0x7c, 0x37, 0x27, 0x3b, 0xd3, 0xa3, 0xaf, 0x0c, 0xec,
    0x09, 0xbc, 0x98, 0xa2, 0x70, 0x42, 0x47, 0x27, 0x66, 0x6e, 0x74, 0xcf, 0xcd, 0xc9, 0x20, 0x21,
    0xfe, 0x65, 0xa5, 0xea, 0xc8, 0xc3, 0x9b, 0xce, 0xf1, 0x02, 0xdf, 0xc2, 0x01, 0x07, 0xaf, 0xcc,
    0x6b, 0x49, 0x46, 0x8a, 0xfa, 0x55, 0x17, 0x18, 0x48, 0x3e, 0xd1, 0xc3, 0x9c, 0x26, 0x49, 0xf8,
    0x60, 0x45, 0x63, 0xd1, 0x7a, 0xcc, 0xfd, 0x27, 0x87, 0xa8, 0x2e, 0x1f, 0xca, 0x1d, 0xfd, 0x21,
    0x63, 0x7c, 0x3a, 0x60, 0x46, 0xb7, 0x4c, 0xdf, 0xbb, 0x51, 0xf6, 0x0b, 0xc4, 0xd8, 0xa7, 0xaa,
    0x7e, 0xb0, 0x3f, 0x01, 0x10, 0xcd, 0x4d, 0x18, 0x28, 0x3b, 0x5a, 0x49, 0x7c, 0x37, 0x95, 0x25,
    0xb3, 0xab, 0x71, 0x56, 0x38, 0xac, 0xd0, 0x06, 0xcc, 0xb5, 0xe5, 0x8b, 0x8c, 0x0f, 0x1a, 0x99,
    0xba, 0x78, 0xe6, 0xd3, 0x79, 0x59, 0x18, 0xfa, 0x6f, 0x53, 0x21, 0x51, 0x26, 0xb3, 0x5f, 0xe6,
    0x56, 0x68, 0x08, 0xa6, 0x0a, 0x65, 0x63, 0x08, 0x19, 0x5d, 0xc9, 0x2e, 0x4b, 0xd7, 0x14, 0x12,
    0xbd, 0x92, 0x60, 0xfc, 0x01, 0x8b, 0xe3, 0x77, 0x29, 0x23, 0xe0, 0x75, 0x60, 0xd5, 0x8d, 0x2c,
    0x6e, 0x71, 0x7e, 0x03, 0x48, 0xa8, 0x6b, 0x8b, 0xac, 0x1c, 0x17, 0xae, 0xb3, 0x0f, 0x42, 0x3d,
    0x1a, 0xcd, 0xd4, 0x7d, 0xae, 0x56, 0xd9, 0x93, 0xbd, 0x0d, 0xe4, 0x63, 0xa1, 0xa3, 0x1e, 0x14,
    0x78, 0x9e, 0xb8, 0x80, 0xfe, 0x46, 0x24, 0x3b, 0xa0, 0x4d, 0xa6, 0xea, 0x6e, 0x00, 0x9e, 0xda
};

static const unsigned char gSignature[] = {
    0x58, 0xb0, 0x8a, 0x78, 0x08, 0x20, 0xbf, 0xd6, 0x26, 0x25, 0x37, 0xd3, 0x51, 0x59, 0x30, 0x86,
    0x59, 0x4a, 0xb0, 0xe4, 0x44, 0xfa, 0x7d, 0x30, 0x98, 0x4f, 0x67, 0xd0, 0x98, 0xd0, 0xdb, 0x68,
    0x55, 0x1d, 0x2c, 0x8f, 0x9c, 0x93, 0x50, 0x2a, 0x1e, 0xaa, 0x0b, 0x5a, 0x63, 0x5b, 0x38, 0xee,
    0xa2, 0x11, 0xcf, 0xf4, 0xe0, 0xee, 0x7d, 0x34, 0x82, 0x86, 0x42, 0x7e, 0xae, 0x6b, 0xdb, 0x99,
    0xca, 0xf8, 0xda, 0x89, 0xb2, 0xc2, 0xae, 0x5f, 0xbc, 0xa8, 0x13, 0x34, 0x91, 0x59, 0x5c, 0x3c,
    0x3b, 0xec, 0x2c, 0x52, 0x6b, 0xcf, 0x69, 0xb1, 0x34, 0x80, 0xa3, 0xec, 0xeb, 0xb3, 0xf2, 0xe4,
    0x5a, 0x99, 0x96, 0x31, 0xa3, 0xa8, 0x3d, 0xca, 0x22, 0x0c, 0xbb, 0xc0, 0xf0, 0x4c, 0x39, 0xe8,
    0xcf, 0xba, 0x95, 0x2b, 0xc9, 0x83, 0x57, 0xee, 0xde, 0x09, 0x73, 0xc4, 0xce, 0x4f, 0x00, 0xf3,
    0xc3, 0xf1, 0x69, 0xc2, 0xd5, 0x66, 0x3d, 0xc6, 0x7d, 0xd1, 0x4c, 0x89, 0xf5, 0x48, 0x07, 0xb7,
    0x50, 0x0e, 0xfb, 0x10, 0xaa, 0x13, 0x5f, 0x3a, 0x9a, 0xc2, 0x9d, 0x49, 0xf4, 0xa4, 0xe4, 0x30,
    0x18, 0x0c, 0xfc, 0xbe, 0x30, 0x77, 0xb4, 0x78, 0xa4, 0xee, 0xd8, 0xc7, 0x97, 0xaf, 0x16, 0x6a,
    0x77, 0xee, 0xc9, 0x64, 0x74, 0xbe, 0xfd, 0x3d, 0xbe, 0x01, 0x78, 0x1d, 0x71, 0x63, 0x4e, 0x80,
    0xb2, 0x6e, 0xf9, 0x60, 0x17, 0x18, 0x5a, 0x1d, 0x76, 0xc5, 0x72, 0xdb, 0xd5, 0xdf, 0x3c, 0x55,
    0xdc, 0x35, 0x01, 0xf3, 0xce, 0xaf, 0xa3, 0x0b, 0xbc, 0x32, 0x7c, 0xcb, 0x74, 0x63, 0x54, 0x38,
    0x01, 0x0a, 0xf3, 0xe1, 0x91, 0x16, 0x4a, 0x70, 0x3a, 0x34, 0x3a, 0x27, 0x4b, 0x0f, 0xec, 0x67,
    0xb3, 0x47, 0x17, 0x35, 0x6b, 0xa2, 0x57, 0x16, 0xed, 0xcf, 0xcf, 0x43, 0x03, 0xbe, 0x2b, 0x15
};

static const unsigned char gOaep[] = {
    0x19, 0x23, 0xeb, 0xad, 0x2f, 0x61, 0x5e, 0x2b, 0xea, 0x1a, 0x32, 0x57, 0x52, 0xff, 0x19, 0x3c,
    0xd0, 0x41, 0x6a, 0xa2, 0x84, 0x87, 0xe8, 0x93, 0x14, 0x3c, 0x70, 0xae, 0x08, 0xb3, 0xe6, 0x11,
    0x1b, 0xea, 0x9b, 0x89, 0xc4, 0x68, 0x92, 0x47, 0x4a, 0x6b, 0x46, 0xee, 0x2a, 0xf4, 0xc3, 0xed,
    0x9b, 0x34, 0x2d, 0x6a, 0x8c, 0x51, 0xfd, 0x13, 0x1e, 0x3b, 0x5c, 0xb9, 0x27, 0xa2, 0x6a, 0x28,
    0x00, 0x2c, 0x79, 0xee, 0x32, 0x51, 0x69, 0xf4, 0xb2, 0x3b, 0xd0, 0x9c, 0x4c, 0x4b, 0x89, 0xc8,
    0x1c, 0x24, 0x26, 0x59, 0xcd, 0xa6, 0x9f, 0xda, 0x7b, 0x0a, 0x98, 0x69, 0x77, 0x96, 0xeb, 0x3b,
    0x03, 0x62, 0x14, 0x2f, 0x76, 0x78, 0xf2, 0x40, 0x2a, 0xf2, 0xc5, 0x7a, 0x11, 0x20, 0x6b, 0xa0,
    0x3e, 0x89, 0x1c, 0x4a, 0xb3, 0xf8, 0xa6, 0xfb, 0xa3, 0xea, 0x5f, 0xbc, 0x11, 0xe2, 0xed, 0xc8,
    0x34, 0x78, 0xea, 0xa5, 0x8a, 0x59, 0xd2, 0x0b, 0x21, 0x37, 0x41, 0xa2, 0x87, 0x54, 0x84, 0xb9,
    0xa4, 0x9a, 0x57, 0x4e, 0x5a, 0xf2, 0x51, 0x5b, 0x7e, 0x34, 0xd3, 0xea, 0xae, 0xb1, 0x51, 0x9d,
    0x06, 0x3e, 0x92, 0x87, 0xc8, 0x77, 0xd2, 0xdd, 0x8a, 0x44, 0xe4, 0xd9, 0xfe, 0xb0, 0xe8, 0x8c,
    0xf5, 0xd2, 0xcf, 0xde, 0x39, 0x73, 0x53, 0x81, 0x73, 0x97, 0x5a, 0x7d, 0xad, 0x33, 0xf9, 0x84,
    0x77, 0x07, 0x0e, 0x07, 0xf1, 0x5c, 0x15, 0xe6, 0xf8, 0x4e, 0xf3, 0xd9, 0xef, 0x05, 0xad, 0x12,
    0x3a, 0x89, 0xac, 0x17, 0xcc, 0x23, 0xe4, 0xd1, 0xf9, 0xe4, 0xdb, 0xcd, 0xc6, 0x86, 0x4d, 0x5c,
    0xa8, 0xb3, 0x5c, 0x2a, 0x2a, 0x1b, 0xc3, 0x90, 0x2b, 0xc2, 0x5f, 0x29, 0x0e, 0x61, 0xce, 0x73,
    0xa7, 0xfd, 0x6f, 0xd4, 0xb3, 0xf6, 0x2c, 0x61, 0x7c, 0x49, 0x0e, 0xfd, 0x35, 0x96, 0x3d, 0xa4
};

TEST(RsaTest, ModExp19745)
{
    nn::crypto::RsaCalculator<256, 256> privKey;
    nn::crypto::RsaCalculator<256, 3> pubKey;
    privKey.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD));
    pubKey.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE));
    char plain[256];
    //std::memset(plain, 1, sizeof(plain));
    std::memcpy( plain, gOaep, sizeof(plain) );
    char cipher[256];
    char plain2[256];
    privKey.ModExp(cipher, sizeof(cipher), plain, sizeof(plain));
    EXPECT_EQ(0, cipher[0]);
    pubKey.ModExp(plain2, sizeof(plain2), cipher, sizeof(cipher));
    EXPECT_EQ(0, std::memcmp(plain, plain2, sizeof(plain)));

    // 0 = 0^d mod n
    std::memset(plain, 0, sizeof(plain));
    privKey.ModExp(cipher, sizeof(cipher), plain, sizeof(plain));
    EXPECT_EQ(0, std::memcmp(cipher, plain, sizeof(plain)));
    // 1 = 1^d mod n
    plain[sizeof(plain) - 1] = 1;
    privKey.ModExp(cipher, sizeof(cipher), plain, sizeof(plain));
    // Need PR 10142 to be accepted before submitting support for that one
    //EXPECT_EQ(0, std::memcmp(cipher, plain, sizeof(plain)));
}

TEST(RsaTest, PssVerify19745)
{
    nn::crypto::Rsa2048PssSha256Verifier ver;
    ver.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPublicE, sizeof(gKeyPublicE) );
    ver.Update( gMessage, sizeof(gMessage) );
    EXPECT_TRUE( ver.Verify( gSignature, sizeof(gSignature) ) );

    EXPECT_TRUE( nn::crypto::VerifyRsa2048PssSha256( gSignature, sizeof(gSignature),
                                                     gKeyPublicN, sizeof(gKeyPublicN),
                                                     gKeyPublicE, sizeof(gKeyPublicE),
                                                     gMessage, sizeof(gMessage) ) );
}

// Same random as used by OpenSSL above, to get matching signatures
static const unsigned char gSalt[] = {
    0x52, 0xfc, 0x20, 0xde, 0x93, 0x04, 0x85, 0x7a, 0x46, 0x3e, 0x39, 0xc2, 0x64, 0xb1, 0xd3, 0x76,
    0x6b, 0x3b, 0x16, 0xc6, 0xaf, 0x75, 0x9d, 0x3a, 0x2b, 0x54, 0xcc, 0x00, 0x3a, 0xa6, 0x36, 0x14
};

TEST(RsaTest, PssSign19745)
{
    nn::crypto::Rsa2048PssSha256Signer gen;
    gen.Initialize( gKeyPublicN, sizeof(gKeyPublicN),
                    gKeyPrivateD, sizeof(gKeyPrivateD) );
    gen.Update( gMessage, sizeof(gMessage) );

    char signature[sizeof(gKeyPublicN)];

    std::memset( signature, 0, sizeof(signature) );
    gen.Sign( signature, sizeof(signature), gSalt, sizeof(gSalt) );
    EXPECT_EQ( 0, std::memcmp( signature, gSignature, sizeof(signature) ) );

    std::memset( signature, 0, sizeof(signature) );
    nn::crypto::SignRsa2048PssSha256( signature, sizeof(signature),
                                      gKeyPublicN, sizeof(gKeyPublicN),
                                      gKeyPrivateD, sizeof(gKeyPrivateD),
                                      gMessage, sizeof(gMessage),
                                      gSalt, sizeof(gSalt) );
    EXPECT_EQ( 0, std::memcmp( signature, gSignature, sizeof(signature) ) );
}
