﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ProfileLog.h"

#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>

static ProfileLog* GlobalLog = NULL;

ProfileLog::ProfileLog( const char* fileout )
    : m_outputFileOffset( -1 )
{
    InitializeOutput(fileout);
}

ProfileLog::~ProfileLog()
{
    FinalizeOutput();
}

ProfileLog* ProfileLog::InitializeGlobalLog(const char* fileout)
{
    if ((GlobalLog == NULL) && (fileout != NULL))
    {
        GlobalLog = new ProfileLog(fileout);
    }
    return GlobalLog;
}

void ProfileLog::FinalizeGlobalLog()
{
    if ( GlobalLog != NULL )
    {
        delete GlobalLog;
        GlobalLog = NULL;
    }
}

void ProfileLog::WriteGlobalLog(const char* format, ...)
{
    std::va_list args;
    va_start(args, format);
    WriteGlobalVLog(format, args);
    va_end(args);
}

void ProfileLog::WriteGlobalVLog(const char* format, std::va_list args)
{
    if ( GlobalLog != NULL )
    {
        GlobalLog->WriteVLog(format, args);
    } else {
        NN_VLOG(format, args);
    }
}

void ProfileLog::WriteLog(const char* format, ...)
{
    std::va_list args;
    va_start(args, format);
    WriteVLog(format, args);
    va_end(args);
}

void ProfileLog::WriteVLog(const char* format, std::va_list args)
{
    if (0 <= m_outputFileOffset)
    {
        WriteToFile(format, args);
    }
    else
    {
        WriteToConsole(format, args);
    }
}

void ProfileLog::InitializeOutput(const char* fileout)
{
    if (fileout != NULL)
    {
        const char* kDestRootPath = "C:/temp";
        const int kDestRootPathSize = 128;
        char path[kDestRootPathSize];
        nn::Result result = nn::fs::ResultPathAlreadyExists();

        for (uint32_t ik = 0; ik < 100 && nn::fs::ResultPathAlreadyExists::Includes(result); ++ik)
        {
            nn::util::SNPrintf(path, kDestRootPathSize, "%s/%02d_%s", kDestRootPath, ik, fileout);
            result = nn::fs::CreateFile(path, 0);
        }

        if ( result.IsSuccess() )
        {
            result = nn::fs::OpenFile(&m_outputFile, path,
                                      nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
            if ( result.IsSuccess() )
            {
                m_outputFileOffset = 0;
            }
        }

        if ( result.IsFailure() )
        {
            NN_LOG("Error in %s. Description %d Module %d\n", __FUNCTION__,
                   result.GetDescription(), result.GetModule());
        }
    }
}

void ProfileLog::FinalizeOutput()
{
    if (0 <= m_outputFileOffset)
    {
        nn::Result result;
        result = nn::fs::FlushFile(m_outputFile);
        //NN_LOG("m_outputFileOffset %lld %d %d\n",m_outputFileOffset, result.GetDescription(), result.GetModule());
        nn::fs::CloseFile(m_outputFile);
        m_outputFileOffset = -1;
    }
}

void ProfileLog::WriteToConsole(const char* format, ...)
{
    std::va_list args;
    va_start(args, format);
    WriteToConsole(format, args);
    va_end(args);
}

void ProfileLog::WriteToConsole(const char* format, std::va_list args)
{
    NN_VLOG(format, args);
}

void ProfileLog::WriteToFile(const char* format, ...)
{
    std::va_list args;
    va_start(args, format);
    WriteToFile(format, args);
    va_end(args);
}

void ProfileLog::WriteToFile(const char* format, std::va_list args)
{
    if (0 <= m_outputFileOffset)
    {
        nn::Result result;
        const size_t bufferSize = 256;
        char buffer[bufferSize];
        nn::fs::WriteOption op = {};
        int writeOut = 0;
        writeOut = nn::util::VSNPrintf(buffer,bufferSize,format, args);
        result = nn::fs::WriteFile(m_outputFile, m_outputFileOffset, buffer, writeOut, op);
        if ( result.IsFailure() )
        {
            NN_LOG("Error in %s. Description %d Module %d\n", __FUNCTION__,
                   result.GetDescription(), result.GetModule());
        }
        m_outputFileOffset += writeOut;
    }
}
