﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Sha1Generator.h>

#include "testCrypto_Util.h"

struct Sha1TestVector
{
    char       data[nn::crypto::Sha1Generator::BlockSize + 1];
    size_t     dataSize;
    nn::Bit8   expectedHash[nn::crypto::Sha1Generator::HashSize];
};

/* http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
   から抜粋したテストベクトルに RFC-3174 のテストベクトルを加えたもの */
const Sha1TestVector sha1TestVectors[] =
{
    // #0) 0 byte (null message) (入力メッセージがない)
    {
        "",
        0,
        {0xda, 0x39, 0xa3, 0xee, 0x5e, 0x6b, 0x4b, 0x0d, 0x32, 0x55, 0xbf, 0xef, 0x95, 0x60, 0x18, 0x90,
         0xaf, 0xd8, 0x07, 0x09},
    },

    // #1) 1 byte 0x36 (ブロック長より短いメッセージ1)
    {
        "\x36",
        1,
        {0xc1, 0xdf, 0xd9, 0x6e, 0xea, 0x8c, 0xc2, 0xb6, 0x27, 0x85, 0x27, 0x5b, 0xca, 0x38, 0xac, 0x26,
         0x12, 0x56, 0xe2, 0x78},
    },

    // #2) 4 bytes 0x549e959e (ブロック長より短いメッセージ2)
    {
        "\x54\x9e\x95\x9e",
        4,
        {0xb7, 0x8b, 0xae, 0x6d, 0x14, 0x33, 0x8f, 0xfc, 0xcf, 0xd5, 0xd5, 0xb5, 0x67, 0x4a, 0x27, 0x5f,
         0x6e, 0xf9, 0xc7, 0x17},
    },

    // #3) 55 bytes data (パディングを考慮するとちょうど1ブロックになるメッセージ)
    {
        "\xec\x6b\x4a\x88\x71\x3d\xf2\x7c\x0f\x2d\x02\xe7\x38\xb6\x9d\xb4\x3a\xbd\xa3\x92\x13\x17\x25\x9c\x86\x4c\x1c\x38\x6e\x9a\x5a\x3f"
        "\x53\x3d\xc0\x5f\x3b\xee\xb2\xbe\xc2\xaa\xc8\xe0\x6d\xb4\xc6\xcb\x3c\xdd\xcf\x69\x7e\x03\xd5",
        55,
        {0xa7, 0x27, 0x2e, 0x23, 0x08, 0x62, 0x2f, 0xf7, 0xa3, 0x39, 0x46, 0x0a, 0xdc, 0x61, 0xef, 0xd0,
         0xea, 0x8d, 0xab, 0xdc},
    },

    // #4) 56 bytes data (パディングを考慮するとちょうど1ブロックを超えるメッセージ)
    {
        "\x03\x21\x73\x6b\xeb\xa5\x78\xe9\x0a\xbc\x1a\x90\xaa\x56\x15\x7d\x87\x16\x18\xf6\xde\x0d\x76\x4c\xc8\xc9\x1e\x06\xc6\x8e\xcd\x3b"
        "\x9d\xe3\x82\x40\x64\x50\x33\x84\xdb\x67\xbe\xb7\xfe\x01\x22\x32\xda\xca\xef\x93\xa0\x00\xfb\xa7",
        56,
        {0xae, 0xf8, 0x43, 0xb8, 0x69, 0x16, 0xc1, 0x6f, 0x66, 0xc8, 0x4d, 0x83, 0xa6, 0x00, 0x5d, 0x23,
         0xfd, 0x00, 0x5c, 0x9e},
    },

    // #5) 57 bytes data (パディングを考慮すると1ブロックを超えるメッセージ)
    {
        "\xd0\xa2\x49\xa9\x7b\x5f\x14\x86\x72\x1a\x50\xd4\xc4\xab\x3f\x5d\x67\x4a\x0e\x29\x92\x5d\x5b\xf2\x67\x8e\xf6\xd8\xd5\x21\xe4\x56"
        "\xbd\x84\xaa\x75\x53\x28\xc8\x3f\xc8\x90\x83\x77\x26\xa8\xe7\x87\x7b\x57\x0d\xba\x39\x57\x9a\xab\xdd",
        57,
        {0xbe, 0x2c, 0xd6, 0xf3, 0x80, 0x96, 0x9b, 0xe5, 0x9c, 0xde, 0x2d, 0xff, 0x5e, 0x84, 0x8a, 0x44,
         0xe7, 0x88, 0x0b, 0xd6},
    },

    // #6) 64 bytes data (1ブロックサイズ分のメッセージ)
    {
        "\x45\x92\x7e\x32\xdd\xf8\x01\xca\xf3\x5e\x18\xe7\xb5\x07\x8b\x7f\x54\x35\x27\x82\x12\xec\x6b\xb9\x9d\xf8\x84\xf4\x9b\x32\x7c\x64"
        "\x86\xfe\xae\x46\xba\x18\x7d\xc1\xcc\x91\x45\x12\x1e\x14\x92\xe6\xb0\x6e\x90\x07\x39\x4d\xc3\x3b\x77\x48\xf8\x6a\xc3\x20\x7c\xfe",
        64,
        {0xa7, 0x0c, 0xfb, 0xfe, 0x75, 0x63, 0xdd, 0x0e, 0x66, 0x5c, 0x7c, 0x67, 0x15, 0xa9, 0x6a, 0x8d,
         0x75, 0x69, 0x50, 0xc0},
    },

    // #7) 640 bytes data (長めのメッセージ) (RFC-3174 より)
    {
        "01234567012345670123456701234567",
        640,
        {0xde, 0xa3, 0x56, 0xa2, 0xcd, 0xdd, 0x90, 0xc7, 0xa7, 0xec, 0xed, 0xc5, 0xeb, 0xb5, 0x63, 0x93,
         0x4f, 0x46, 0x04, 0x52},
    },

    // #8) 1000000 bytes of 'a' (比較的長めのメッセージ) (RFC-3174 より)
    {
        "a",
        1000000,
        {0x34, 0xaa, 0x97, 0x3c, 0xd4, 0xc4, 0xda, 0xa4, 0xf6, 0x1e, 0xeb, 0x2b, 0xdb, 0xad, 0x27, 0x31,
         0x65, 0x34, 0x01, 0x6f},
    },
};

const int TestVectorCount = sizeof(sha1TestVectors) / sizeof(sha1TestVectors[0]);

void Sha1BasicTest(const Sha1TestVector& testVector)
{
    nn::crypto::Sha1Generator sha1;
    nn::Bit8                  hash[nn::crypto::Sha1Generator::HashSize];

    sha1.Initialize();
    if (testVector.dataSize <= nn::crypto::Sha1Generator::BlockSize)
    {
        // 入力がブロックサイズより小さい場合は一括で計算
        sha1.Update(testVector.data, testVector.dataSize);
    }
    else
    {
        // 入力が大きい場合は繰り返し単位ごとに分割して計算
        size_t remaining = testVector.dataSize;
        while (remaining != 0 )
        {
            size_t calculationSize = std::min(std::strlen(testVector.data), remaining);
            calculationSize = (calculationSize == 0) ? 1: calculationSize; // ヌル文字列の場合は1を入力
            sha1.Update(testVector.data, calculationSize);
            remaining -= calculationSize;
        }
    }
    sha1.GetHash(hash, sizeof(hash));

    EXPECT_ARRAY_EQ(hash, testVector.expectedHash, nn::crypto::Sha1Generator::HashSize);
}

void Sha1FunctionTest(const Sha1TestVector& testVector)
{
    nn::Bit8  hash[nn::crypto::Sha1Generator::HashSize];

    if (testVector.dataSize <= nn::crypto::Sha1Generator::BlockSize)
    {
        nn::crypto::GenerateSha1Hash(hash,
                                     sizeof(hash),
                                     testVector.data,
                                     testVector.dataSize);

        EXPECT_ARRAY_EQ(hash, testVector.expectedHash, nn::crypto::Sha1Generator::HashSize);
    }
}

/**
  @brief   Sha1Generator クラスによるハッシュ計算をテストします。

  @details
  NIST が公表しているテストベクトルを用いて、
  正しいハッシュ値が計算されることを確認します。
 */
TEST(Sha1Test, Basic)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        Sha1BasicTest(sha1TestVectors[i]);
    }
}

/**
  @brief   GenerateSha1Hash 関数によるハッシュ計算をテストします。

  @details
  NIST が公表しているテストベクトルを用いて、
  正しいハッシュ値が計算されることを確認します。
 */
TEST(Sha1Test, FunctionInterface)
{
    // 短いテストベクトルについてのみテストする
    for (int i = 0; i < TestVectorCount; i++)
    {
        Sha1FunctionTest(sha1TestVectors[i]);
    }
}

/**
  @brief   Sha1Generator クラスの状態遷移をテストします。
 */
TEST(Sha1Test, StateTransition)
{
    nn::crypto::Sha1Generator sha1;
    nn::Bit8                  hash[nn::crypto::Sha1Generator::HashSize];
    nn::Bit8                  hash2[nn::crypto::Sha1Generator::HashSize];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 初期化せずに Update が呼ばれたら NG
    EXPECT_DEATH_IF_SUPPORTED(sha1.Update(sha1TestVectors[0].data, sha1TestVectors[0].dataSize), "");

    // 初期化せずに GetHash が呼ばれても NG
    EXPECT_DEATH_IF_SUPPORTED(sha1.GetHash(hash, sizeof(hash)), "");
#endif

    // 初期化
    sha1.Initialize();

    // 初期化の後に GetHash が呼ばれても大丈夫
    sha1.GetHash(hash, sizeof(hash));

    // GetHash は連続で呼んでも大丈夫で、同じ値が出力されるはず
    sha1.GetHash(hash2, sizeof(hash2));

    EXPECT_ARRAY_EQ(hash, hash2, nn::crypto::Sha1Generator::HashSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // GetHash の後に Update が呼ばれたら NG (Release ビルドでは ASSERT が無効のため停止しない)
    EXPECT_DEATH_IF_SUPPORTED(sha1.Update(sha1TestVectors[0].data, sha1TestVectors[0].dataSize), "");
#endif
}

/**
  @brief   非 32bit アラインメントの入出力バッファに対する挙動をテストします。
 */
TEST(Sha1Test, UnalignedIoBuffer)
{
    nn::crypto::Sha1Generator sha1;
    nn::Bit8                  hash[nn::crypto::Sha1Generator::HashSize + sizeof(nn::Bit64)] = {0};
    nn::Bit8                  data[nn::crypto::Sha1Generator::BlockSize + sizeof(nn::Bit64)] = {0};

    // http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/SHA1.pdf にある、
    // 処理する位置・順序がずれたら計算結果が変わってしまうようなテストベクトルを使用する
    Sha1TestVector testVector =
    {
        "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        56,
        {0x84, 0x98, 0x3e, 0x44, 0x1c, 0x3b, 0xd2, 0x6e, 0xba, 0xae, 0x4a, 0xa1, 0xf9, 0x51, 0x29, 0xe5,
         0xe5, 0x46, 0x70, 0xf1},
    };

    // 非 32bit アラインメントの入力バッファに対して問題ないことをテストする
    for (int i = 0; i < static_cast<int>(sizeof(nn::Bit64)); i++)
    {
        nn::Bit8* head = data + i;

        std::memcpy(head, testVector.data, testVector.dataSize);
        sha1.Initialize();
        sha1.Update(head, testVector.dataSize);
        sha1.GetHash(hash, nn::crypto::Sha1Generator::HashSize);

        EXPECT_ARRAY_EQ(hash, testVector.expectedHash, nn::crypto::Sha1Generator::HashSize);
    }

    // 非 32bit アラインメントの出力バッファに対して問題ないことをテストする
    for (int i = 0; i < static_cast<int>(sizeof(nn::Bit64)); i++)
    {
        nn::Bit8* head = hash + i;

        sha1.Initialize();
        sha1.Update(testVector.data, testVector.dataSize);
        sha1.GetHash(head, nn::crypto::Sha1Generator::HashSize);

        EXPECT_ARRAY_EQ(head, testVector.expectedHash, nn::crypto::Sha1Generator::HashSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
TEST(Sha1Test, Destructor)
{
    nn::crypto::Sha1Generator sha;
    sha.Initialize();

    // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
    EXPECT_ARRAY_NONZERO(&sha, sizeof(sha));
    sha.~Sha1Generator();
    EXPECT_ARRAY_ZERO(&sha, sizeof(sha));
}
