﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_CcmEncryptor.h>
#include <nn/crypto/crypto_CcmDecryptor.h>
#include <nn/crypto/crypto_Aes128CcmEncryptor.h>
#include <nn/crypto/crypto_Aes128CcmDecryptor.h>

#include "testCrypto_Util.h"

struct AesCcmTestVector
{
    nn::Bit8  key[32];
    size_t    keySize;
    nn::Bit8  iv[16];
    size_t    ivSize;
    nn::Bit8  aad[256];
    size_t    aadSize;
    nn::Bit8  plainText[160];
    size_t    textSize;
    nn::Bit8  cipherText[160];
    nn::Bit8  mac[16];
    size_t    macSize;
};

/* NIST SP800-38C に掲載されているテストベクトルおよび
   http://csrc.nist.gov/groups/STM/cavp/documents/mac/ccmtestvectors.zip
   に掲載されているテストベクトルから抜粋 */
const AesCcmTestVector aesCcmTestVectors[] =
{
    // #1) Klen = 16, Tlen = 4, Nlen = 7, Alen = 8, and Plen = 4 (平文も AAD もブロックサイズ未満)
    {
        {0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f},
        16,
        {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16},
        7,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07},
        8,
        {0x20, 0x21, 0x22, 0x23},
        4,
        {0x71, 0x62, 0x01, 0x5b},
        {0x4d, 0xac, 0x25, 0x5d},
        4,
    },

    // #2) Klen = 16, Tlen = 6, Nlen = 8, Alen = 16, and Plen = 16 (平文も AAD もちょうどブロックサイズ)
    {
        {0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f},
        16,
        {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17},
        8,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        16,
        {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f},
        16,
        {0xd2, 0xa1, 0xf0, 0xe0, 0x51, 0xea, 0x5f, 0x62, 0x08, 0x1a, 0x77, 0x92, 0x07, 0x3d, 0x59, 0x3d},
        {0x1f, 0xc6, 0x4f, 0xbf, 0xac, 0xcd},
        6,
    },

    // #3) Klen = 16, Tlen = 8, Nlen = 12, Alen = 20, and Plen = 24 (平文も AAD もブロックサイズ以上)
    {
        {0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f},
        16,
        {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b},
        12,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13},
        20,
        {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37},
        24,
        {0xe3, 0xb2, 0x01, 0xa9, 0xf5, 0xb7, 0x1a, 0x7a, 0x9b, 0x1c, 0xea, 0xec, 0xcd, 0x97, 0xe7, 0x0b,
         0x61, 0x76, 0xaa, 0xd9, 0xa4, 0x42, 0x8a, 0xa5},
        {0x48, 0x43, 0x92, 0xfb, 0xc1, 0xb0, 0x99, 0x51},
        8,
    },

    // #4) Klen = 16, Tlen = 14, Nlen = 13, Alen = 65536, and Plen = 32 (AAD が 16KB を超える)
    {
        {0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f},
        16,
        {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c},
        13,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
         0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
         0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
         0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
         0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
         0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
         0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
         0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
         0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
         0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        65536,
        {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f},
        32,
        {0x69, 0x91, 0x5d, 0xad, 0x1e, 0x84, 0xc6, 0x37, 0x6a, 0x68, 0xc2, 0x96, 0x7e, 0x4d, 0xab, 0x61,
         0x5a, 0xe0, 0xfd, 0x1f, 0xae, 0xc4, 0x4c, 0xc4, 0x84, 0x82, 0x85, 0x29, 0x46, 0x3c, 0xcf, 0x72},
        {0xb4, 0xac, 0x6b, 0xec, 0x93, 0xe8, 0x59, 0x8e, 0x7f, 0x0d, 0xad, 0xbc, 0xea, 0x5b},
        14,
    },

    // #5) Klen = 16, Tlen = 16, Nlen = 13, Alen = 0, and Plen = 24 (平文の入力のみ)
    {
        {0xd2, 0x4a, 0x3d, 0x3d, 0xde, 0x8c, 0x84, 0x83, 0x02, 0x80, 0xcb, 0x87, 0xab, 0xad, 0x0b, 0xb3},
        16,
        {0xf1, 0x10, 0x00, 0x35, 0xbb, 0x24, 0xa8, 0xd2, 0x60, 0x04, 0xe0, 0xe2, 0x4b},
        13,
        {0},
        0,
        {0x7c, 0x86, 0x13, 0x5e, 0xd9, 0xc2, 0xa5, 0x15, 0xaa, 0xae, 0x0e, 0x9a, 0x20, 0x81, 0x33, 0x89,
         0x72, 0x69, 0x22, 0x0f, 0x30, 0x87, 0x00, 0x06},
        24,
        {0x1f, 0xae, 0xb0, 0xee, 0x2c, 0xa2, 0xcd, 0x52, 0xf0, 0xaa, 0x39, 0x66, 0x57, 0x83, 0x44, 0xf2,
         0x4e, 0x69, 0xb7, 0x42, 0xc4, 0xab, 0x37, 0xab},
        {0x11, 0x23, 0x30, 0x12, 0x19, 0xc7, 0x05, 0x99, 0xb7, 0xc3, 0x73, 0xad, 0x4b, 0x3a, 0xd6, 0x7b},
        16,
    },

    // #6) Klen = 16, Tlen = 16, Nlen = 13, Alen = 32, and Plen = 0 (AAD の入力のみ)
    {
        {0x2e, 0xbf, 0x60, 0xf0, 0x96, 0x90, 0x13, 0xa5, 0x4a, 0x3d, 0xed, 0xb1, 0x9d, 0x20, 0xf6, 0xc8},
        16,
        {0x1d, 0xe8, 0xc5, 0xe2, 0x1f, 0x9d, 0xb3, 0x31, 0x23, 0xff, 0x87, 0x0a, 0xdd},
        13,
        {0xe1, 0xde, 0x6c, 0x61, 0x19, 0xd7, 0xdb, 0x47, 0x11, 0x36, 0x28, 0x5d, 0x10, 0xb4, 0x7a, 0x45,
         0x02, 0x21, 0xb1, 0x69, 0x78, 0x56, 0x91, 0x90, 0xef, 0x6a, 0x22, 0xb0, 0x55, 0x29, 0x56, 0x03},
        32,
        {0},
        0,
        {0},
        {0x0e, 0xad, 0x29, 0xef, 0x20, 0x5f, 0xbb, 0x86, 0xd1, 0x1a, 0xbe, 0x5e, 0xd7, 0x04, 0xb8, 0x80},
        16,
    },

    // #7) Klen = 24, Tlen = 16, Nlen = 7, Alen = 24, and Plen = 24 (192bit 鍵)
    {
        {0xce, 0xb0, 0x09, 0xae, 0xa4, 0x45, 0x44, 0x51, 0xfe, 0xad, 0xf0, 0xe6, 0xb3, 0x6f, 0x45, 0x55,
         0x5d, 0xd0, 0x47, 0x23, 0xba, 0xa4, 0x48, 0xe8},
        24,
        {0x76, 0x40, 0x43, 0xc4, 0x94, 0x60, 0xb7},
        7,
        {0x6e, 0x80, 0xdd, 0x7f, 0x1b, 0xad, 0xf3, 0xa1, 0xc9, 0xab, 0x25, 0xc7, 0x5f, 0x10, 0xbd, 0xe7,
         0x8c, 0x23, 0xfa, 0x0e, 0xb8, 0xf9, 0xaa, 0xa5, 0x3a, 0xde, 0xfb, 0xf4, 0xcb, 0xf7, 0x8f, 0xe4},
        32,
        {0xc8, 0xd2, 0x75, 0xf9, 0x19, 0xe1, 0x7d, 0x7f, 0xe6, 0x9c, 0x2a, 0x1f, 0x58, 0x93, 0x9d, 0xfe,
         0x4d, 0x40, 0x37, 0x91, 0xb5, 0xdf, 0x13, 0x10},
        24,
        {0x8a, 0x0f, 0x3d, 0x82, 0x29, 0xe4, 0x8e, 0x74, 0x87, 0xfd, 0x95, 0xa2, 0x8a, 0xd3, 0x92, 0xc8,
         0x0b, 0x36, 0x81, 0xd4, 0xfb, 0xc7, 0xbb, 0xfd},
        {0x2d, 0xd6, 0xef, 0x1c, 0x45, 0xd4, 0xcc, 0xb7, 0x23, 0xdc, 0x07, 0x44, 0x14, 0xdb, 0x50, 0x6d},
        16,
    },

    // #8) Klen = 32, Tlen = 16, Nlen = 7, Alen = 24, and Plen = 24 (256bit 鍵)
    {
        {0x55, 0x35, 0x21, 0xa7, 0x65, 0xab, 0x0c, 0x3f, 0xd2, 0x03, 0x65, 0x4e, 0x99, 0x16, 0x33, 0x0e,
         0x18, 0x9b, 0xdf, 0x95, 0x1f, 0xee, 0xe9, 0xb4, 0x4b, 0x10, 0xda, 0x20, 0x8f, 0xee, 0x7a, 0xcf},
        32,
        {0xaa, 0xa2, 0x3f, 0x10, 0x16, 0x47, 0xd8},
        7,
        {0xa3, 0x55, 0xd4, 0xc6, 0x11, 0x81, 0x2e, 0x5f, 0x92, 0x58, 0xd7, 0x18, 0x8b, 0x3d, 0xf8, 0x85,
         0x14, 0x77, 0x09, 0x4f, 0xfc, 0x2a, 0xf2, 0xcf, 0x0c, 0x86, 0x70, 0xdb, 0x90, 0x3f, 0xbb, 0xe0},
        32,
        {0x64, 0x4e, 0xb3, 0x4b, 0x9a, 0x12, 0x6e, 0x43, 0x7b, 0x5e, 0x01, 0x5e, 0xea, 0x14, 0x1c, 0xa1,
         0xa8, 0x80, 0x20, 0xf2, 0xd5, 0xd6, 0xcc, 0x2c},
        24,
        {0x27, 0xed, 0x90, 0x66, 0x81, 0x74, 0xeb, 0xf8, 0x24, 0x1a, 0x3c, 0x74, 0xb3, 0x5e, 0x12, 0x46,
         0xb6, 0x61, 0x7e, 0x41, 0x23, 0x57, 0x8f, 0x15},
        {0x3b, 0xdb, 0x67, 0x06, 0x2a, 0x13, 0xef, 0x4e, 0x98, 0x6f, 0x5b, 0xb3, 0xd0, 0xbb, 0x43, 0x07},
        16,
    },
};

const int TestVectorCount = sizeof(aesCcmTestVectors) / sizeof(aesCcmTestVectors[0]);

template <typename T>
void AuthenticatedEncryptionTest(const AesCcmTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::CcmEncryptor<T> aesCcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::CcmEncryptor<T>::MaxMacSize];

    EXPECT_GE(sizeof(output), nn::crypto::CcmEncryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesCcm.Initialize(&aes, testVector.iv, testVector.ivSize, testVector.aadSize, testVector.textSize, testVector.macSize);

    if (testVector.aadSize > 0)
    {
        // データが小さい場合はそのまま、大きい場合は分割して入力
        size_t remaining = testVector.aadSize;
        while (remaining > 0)
        {
            size_t inputSize = std::min(sizeof(testVector.aad), remaining);
            aesCcm.UpdateAad(testVector.aad, inputSize);
            remaining -= inputSize;
        }
    }

    if (testVector.textSize > 0)
    {
        if (isInPlace)
        {
            std::memcpy(output, testVector.plainText, testVector.textSize);
            aesCcm.Update(output, sizeof(output), output, testVector.textSize);
        }
        else
        {
            aesCcm.Update(output, sizeof(output), testVector.plainText, testVector.textSize);
        }
    }

    aesCcm.GetMac(mac, testVector.macSize);

    // 暗号化の結果が正しいことの確認
    EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);

    // MAC の値が正しいことの確認
    EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
}

template <typename T>
void AuthenticatedDecryptionTest(const AesCcmTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::CcmDecryptor<T> aesCcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::CcmDecryptor<T>::MaxMacSize];

    EXPECT_GE(sizeof(mac), nn::crypto::CcmDecryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesCcm.Initialize(&aes, testVector.iv, testVector.ivSize, testVector.aadSize, testVector.textSize, testVector.macSize);

    if (testVector.aadSize > 0)
    {
        // データが小さい場合はそのまま、大きい場合は分割して入力
        size_t remaining = testVector.aadSize;
        while (remaining > 0)
        {
            size_t inputSize = std::min(sizeof(testVector.aad), remaining);
            aesCcm.UpdateAad(testVector.aad, inputSize);
            remaining -= inputSize;
        }
    }

    if (testVector.textSize > 0)
    {
        if (isInPlace)
        {
            std::memcpy(output, testVector.cipherText, testVector.textSize);
            aesCcm.Update(output, sizeof(output), output, testVector.textSize);
        }
        else
        {
            aesCcm.Update(output, sizeof(output), testVector.cipherText, testVector.textSize);
        }
    }

    aesCcm.GetMac(mac, testVector.macSize);

    // 復号化の結果が正しいことの確認
    EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);

    // MAC の値が正しいことの確認
    EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
}

template <typename T>
void AuthenticatedEncryptDataBufferingTest(const AesCcmTestVector& testVector)
{
    T                           aes;
    nn::crypto::CcmEncryptor<T> aesCcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::CcmDecryptor<T>::MaxMacSize];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        // 繰り返し入力が必要な特殊なテストベクトルはこのテストでは割愛
        if (testVector.aadSize != sizeof(testVector.aad))
        {
            continue;
        }

        aesCcm.Initialize(&aes, testVector.iv, testVector.ivSize, testVector.aadSize, testVector.textSize, testVector.macSize);

        // AAD を分割入力
        const nn::Bit8* pInput = testVector.aad;
        size_t remaining = testVector.aadSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            aesCcm.UpdateAad(pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
        }

        // 平文を分割入力
        pInput = testVector.plainText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesCcm.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        // 暗号化の結果が正しいことの確認
        EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);

        aesCcm.GetMac(mac, testVector.macSize);

        // MAC の値が正しいことの確認
        EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
    }
}

template <typename T>
void AuthenticatedDecryptDataBufferingTest(const AesCcmTestVector& testVector)
{
    T                           aes;
    nn::crypto::CcmDecryptor<T> aesCcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::CcmDecryptor<T>::MaxMacSize];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        // 繰り返し入力が必要な特殊なテストベクトルはこのテストでは割愛
        if (testVector.aadSize != sizeof(testVector.aad))
        {
            continue;
        }

        aesCcm.Initialize(&aes, testVector.iv, testVector.ivSize, testVector.aadSize, testVector.textSize, testVector.macSize);

        // AAD を分割入力
        const nn::Bit8* pInput = testVector.aad;
        size_t remaining = testVector.aadSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            aesCcm.UpdateAad(pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
        }

        // 暗号文を分割入力
        pInput = testVector.cipherText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesCcm.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        // 復号化の結果が正しいことの確認
        EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);

        aesCcm.GetMac(mac, testVector.macSize);

        // MAC の値が正しいことの確認
        EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
    }
}

/**
  @brief   AES-CCM による認証付き暗号化をテストします。

  @details
  テストベクトルを用いて正しく暗号化と MAC 計算ができることをテストします。
 */
TEST(AesCcmTest, AuthenticatedEncryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor128>(aesCcmTestVectors[i], false);
            break;
        case 24:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor192>(aesCcmTestVectors[i], false);
            break;
        case 32:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor256>(aesCcmTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による認証付き復号化をテストします。

  @details
  テストベクトルを用いて正しく復号化と MAC 計算ができることをテストします。
 */
TEST(AesCcmTest, AuthenticatedDecryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor128>(aesCcmTestVectors[i], false);
            break;
        case 24:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor192>(aesCcmTestVectors[i], false);
            break;
        case 32:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor256>(aesCcmTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による同一入出力バッファ上での認証付き暗号化をテストします。

  @details
  テストベクトルを用いて正しく暗号化と MAC 計算ができることをテストします。
 */
TEST(AesCcmTest, InPlaceAuthenticatedEncryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor128>(aesCcmTestVectors[i], true);
            break;
        case 24:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor192>(aesCcmTestVectors[i], true);
            break;
        case 32:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor256>(aesCcmTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による同一入出力バッファ上での認証付き復号化をテストします。

  @details
  テストベクトルを用いて正しく復号化と MAC 計算ができることをテストします。
 */
TEST(AesCcmTest, InPlaceAuthenticatedDecryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor128>(aesCcmTestVectors[i], true);
            break;
        case 24:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor192>(aesCcmTestVectors[i], true);
            break;
        case 32:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor256>(aesCcmTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による暗号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく暗号化できることをテストします。
 */
TEST(AesCcmTest, AuthenticatedEncryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedEncryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesCcmTestVectors[i]);
            break;
        case 24:
            AuthenticatedEncryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesCcmTestVectors[i]);
            break;
        case 32:
            AuthenticatedEncryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesCcmTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による復号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく復号化できることをテストします。
 */
TEST(AesCcmTest, AuthenticatedDecryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedDecryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesCcmTestVectors[i]);
            break;
        case 24:
            AuthenticatedDecryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesCcmTestVectors[i]);
            break;
        case 32:
            AuthenticatedDecryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesCcmTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   通常の入力パターンでの CcmEncryptor クラスの状態遷移をテストします。
 */
TEST(AesCcmTest, StateTransition)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor128> aesCcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];
    nn::Bit8                                              mac2[16];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 初期化前は Initialize 以外の処理は呼べない
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Update(output, sizeof(output),
                                            aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesCcm.GetMac(mac, sizeof(mac)), "");
#endif

    // 初期化
    aes.Initialize(aesCcmTestVectors[0].key, aesCcmTestVectors[0].keySize);
    aesCcm.Initialize(&aes, aesCcmTestVectors[0].iv, aesCcmTestVectors[0].ivSize,
                      aesCcmTestVectors[0].aadSize, aesCcmTestVectors[0].textSize, aesCcmTestVectors[0].macSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // aadSize > 0 のとき、AAD の入力前に平文を入力したら NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Update(output, sizeof(output),
                                            aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize), "");
#endif

    // AAD の入力
    aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 所定のサイズの入力後には状態が変わっているので NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize), "");
#endif

    // AAD の入力後の平文入力は OK
    aesCcm.Update(output, sizeof(output),
                  aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // AAD と同様に所定のサイズの入力後には状態が変わっているので NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Update(output, sizeof(output),
                                            aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize), "");
#endif

    // AAD、平文入力後の MAC の取得は可能
    aesCcm.GetMac(mac, sizeof(mac));

    // MAC の取得は連続で呼んでも大丈夫で、同じ値が出力されるはず
    aesCcm.GetMac(mac2, sizeof(mac2));

    EXPECT_ARRAY_EQ(mac, mac2, aesCcmTestVectors[0].macSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // GetMac の後は UpdateAad も Update も呼べない
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Update(output, sizeof(output),
                                            aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize), "");
#endif
}

/**
  @brief   AAD の入力がない場合の CcmEncryptor クラスの状態遷移をテストします。
           AesCcmTest.StateTransition で確認できているシーケンスは確認しません。
 */
TEST(AesCcmTest, StateTransition2)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor128> aesCcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];

    // 初期化(テストベクトル #5 は AAD の入力がないパターン)
    aes.Initialize(aesCcmTestVectors[4].key, aesCcmTestVectors[4].keySize);
    aesCcm.Initialize(&aes, aesCcmTestVectors[4].iv, aesCcmTestVectors[4].ivSize,
                      aesCcmTestVectors[4].aadSize, aesCcmTestVectors[4].textSize, aesCcmTestVectors[4].macSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // aadSize == 0 のときは UpdateAad は呼べない
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.UpdateAad(aesCcmTestVectors[4].aad, aesCcmTestVectors[4].aadSize), "");
#endif

    // 平文の入力と MAC の取得
    aesCcm.Update(output, sizeof(output),
                  aesCcmTestVectors[4].plainText ,aesCcmTestVectors[4].textSize);

    aesCcm.GetMac(mac, sizeof(mac));
}

/**
  @brief   平文 の入力がない場合の CcmEncryptor クラスの状態遷移をテストします。
           AesCcmTest.StateTransition で確認できているシーケンスは確認しません。
 */
TEST(AesCcmTest, StateTransition3)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor128> aesCcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];

    // 初期化(テストベクトル #6 は平文の入力がないパターン)
    aes.Initialize(aesCcmTestVectors[5].key, aesCcmTestVectors[5].keySize);
    aesCcm.Initialize(&aes, aesCcmTestVectors[5].iv, aesCcmTestVectors[5].ivSize,
                      aesCcmTestVectors[5].aadSize, aesCcmTestVectors[5].textSize, aesCcmTestVectors[5].macSize);

    // AAD の入力
    aesCcm.UpdateAad(aesCcmTestVectors[5].aad, aesCcmTestVectors[5].aadSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // dataSize == 0 のときは平文の入力はできない
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Update(output, sizeof(output),
                  aesCcmTestVectors[5].plainText ,aesCcmTestVectors[5].textSize), "");
#else
    NN_UNUSED(output);
#endif

    // MAC の取得
    aesCcm.GetMac(mac, sizeof(mac));
}

/**
  @brief   Initialize() 時の事前条件チェックをテストします。
 */
TEST(AesCcmTest, PreconditionCheck)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor128> aesCcm;
    nn::Bit8                                              key[16] = {0};
    nn::Bit8                                              nonce[nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor128>::MaxNonceSize] = {0};

    aes.Initialize(key, 16);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 小さすぎる nonceSize は NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 6, 0, 0, 16), "");

    // 大きすぎる nonceSize も NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 14, 0, 0, 16), "");

    // 負の aadSize は NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 7, -1, 0, 16), "");

    // 負の dataSize も NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 7, 0, -1, 16), "");

    // 小さすぎる macSize は NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 7, 0, 0, 2), "");

    // 大きすぎる macSize も NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 7, 0, 0, 18), "");

    // 範囲内でも奇数の macSize も NG
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 7, 0, 0, 9), "");
#endif

    // nonceSize == 13 のとき、dataSize として指定できるのは 16bit まで
    aesCcm.Initialize(&aes, nonce, 13, 0, 0xFFFF, 16);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // nonceSize == 13 のとき、dataSize として指定できるのは 16bit まで
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 13, 0, 0x10000, 16), "");
#endif

    // nonceSize を 1 つ減らすと次がサポートできる
    aesCcm.Initialize(&aes, nonce, 12, 0, 0x10000, 16);

    // nonceSize == 12 のとき、dataSize として指定できるのは 24bit まで
    aesCcm.Initialize(&aes, nonce, 12, 0, 0xFFFFFF, 16);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // nonceSize == 12 のとき、dataSize として指定できるのは 24bit まで
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 12, 0, 0x1000000, 16), "");
#endif

    // ...

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // nonceSize == 8 のときはまだ最大値は扱えない
    EXPECT_DEATH_IF_SUPPORTED(aesCcm.Initialize(&aes, nonce, 8, 0, INT64_MAX, 16), "");
#endif

    // nonceSize == 7 のとき、dataSize は最大値まで指定可能になる
    aesCcm.Initialize(&aes, nonce, 7, 0, INT64_MAX, 16);
}

/*
  @brief   Aes128CcmEncryptor/Decryptor による暗号化・復号化をテストします。
 */
TEST(AesCcmTest, UtilityClassInterface)
{
    nn::Bit8 output[160];
    nn::Bit8 mac[nn::crypto::Aes128CcmEncryptor::MaxMacSize];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128CcmEncryptor aes128CcmEnc;

        aes128CcmEnc.Initialize(aesCcmTestVectors[i].key, aesCcmTestVectors[i].keySize,
                                aesCcmTestVectors[i].iv, aesCcmTestVectors[i].ivSize,
                                aesCcmTestVectors[i].aadSize, aesCcmTestVectors[i].textSize, aesCcmTestVectors[i].macSize);

        if (aesCcmTestVectors[i].aadSize > 0)
        {
            // データが小さい場合はそのまま、大きい場合は分割して入力
            size_t remaining = aesCcmTestVectors[i].aadSize;
            while (remaining > 0)
            {
                size_t inputSize = std::min(sizeof(aesCcmTestVectors[i].aad), remaining);
                aes128CcmEnc.UpdateAad(aesCcmTestVectors[i].aad, inputSize);
                remaining -= inputSize;
            }
        }

        if (aesCcmTestVectors[i].textSize > 0)
        {
            aes128CcmEnc.Update(output, sizeof(output),
                                aesCcmTestVectors[i].plainText, aesCcmTestVectors[i].textSize);
        }

        aes128CcmEnc.GetMac(mac, sizeof(mac));

        // 暗号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[i].cipherText, aesCcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[i].mac, aesCcmTestVectors[i].macSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128CcmDecryptor aes128CcmDec;

        aes128CcmDec.Initialize(aesCcmTestVectors[i].key, aesCcmTestVectors[i].keySize,
                                aesCcmTestVectors[i].iv, aesCcmTestVectors[i].ivSize,
                                aesCcmTestVectors[i].aadSize, aesCcmTestVectors[i].textSize, aesCcmTestVectors[i].macSize);

        if (aesCcmTestVectors[i].aadSize > 0)
        {
            // データが小さい場合はそのまま、大きい場合は分割して入力
            size_t remaining = aesCcmTestVectors[i].aadSize;
            while (remaining > 0)
            {
                size_t inputSize = std::min(sizeof(aesCcmTestVectors[i].aad), remaining);
                aes128CcmDec.UpdateAad(aesCcmTestVectors[i].aad, inputSize);
                remaining -= inputSize;
            }
        }

        if (aesCcmTestVectors[i].textSize > 0)
        {
            aes128CcmDec.Update(output, sizeof(output),
                                aesCcmTestVectors[i].cipherText, aesCcmTestVectors[i].textSize);
        }

        aes128CcmDec.GetMac(mac, sizeof(mac));

        // 復号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[i].plainText, aesCcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[i].mac, aesCcmTestVectors[i].macSize);
    }
}

/**
  @brief   ユーティリティ関数による暗号化・復号化をテストします。
 */
TEST(AesCcmTest, UtilityFunctionInterface)
{
    nn::Bit8 output[160];
    nn::Bit8 mac[nn::crypto::Aes128CcmEncryptor::MaxMacSize];
    static nn::Bit8 aad[65536]; // 大きい AAD も一括でテストできる大きさのバッファ

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        // AAD が小さい場合はそのまま、大きい場合は分割してバッファにコピー
        size_t remaining = aesCcmTestVectors[i].aadSize;
        size_t copied = 0;
        while (remaining > 0)
        {
            size_t inputSize = std::min(sizeof(aesCcmTestVectors[i].aad), remaining);
            std::memcpy(aad + copied, aesCcmTestVectors[i].aad, inputSize);
            remaining -= inputSize;
            copied += inputSize;
        }

        nn::crypto::EncryptAes128Ccm(output, sizeof(output),
                                     mac, sizeof(mac),
                                     aesCcmTestVectors[i].key, aesCcmTestVectors[i].keySize,
                                     aesCcmTestVectors[i].iv, aesCcmTestVectors[i].ivSize,
                                     aesCcmTestVectors[i].plainText, aesCcmTestVectors[i].textSize,
                                     aad, aesCcmTestVectors[i].aadSize,
                                     aesCcmTestVectors[i].macSize);

        // 暗号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[i].cipherText, aesCcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[i].mac, aesCcmTestVectors[i].macSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        // AAD が小さい場合はそのまま、大きい場合は分割してバッファにコピー
        size_t remaining = aesCcmTestVectors[i].aadSize;
        size_t copied = 0;
        while (remaining > 0)
        {
            size_t inputSize = std::min(sizeof(aesCcmTestVectors[i].aad), remaining);
            std::memcpy(aad + copied, aesCcmTestVectors[i].aad, inputSize);
            remaining -= inputSize;
            copied += inputSize;
        }

        nn::crypto::DecryptAes128Ccm(output, sizeof(output),
                                     mac, sizeof(mac),
                                     aesCcmTestVectors[i].key, aesCcmTestVectors[i].keySize,
                                     aesCcmTestVectors[i].iv, aesCcmTestVectors[i].ivSize,
                                     aesCcmTestVectors[i].cipherText, aesCcmTestVectors[i].textSize,
                                     aad, aesCcmTestVectors[i].aadSize,
                                     aesCcmTestVectors[i].macSize);

        // 復号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[i].plainText, aesCcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[i].mac, aesCcmTestVectors[i].macSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
template <size_t KeySize>
void CcmDestructorTest()
{
    // Dummy key and iv
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };
    const uint8_t iv[16] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    };

    // Encryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, mode.MaxNonceSize, 0, 16, mode.MaxMacSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CcmEncryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードのほうだけクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }

    // Decryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CcmDecryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, mode.MaxNonceSize, 0, 16, mode.MaxMacSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CcmDecryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードがクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }
}

TEST(AesCcmTest, Destructor)
{
    CcmDestructorTest<16>();
    CcmDestructorTest<24>();
    CcmDestructorTest<32>();
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesCcmTest, ZeroByteEncryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CcmEncryptor<nn::crypto::AesEncryptor128> aesCcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];

    // 0 to AAD
    {
        // 初期化
        aes.Initialize(aesCcmTestVectors[0].key, aesCcmTestVectors[0].keySize);
        aesCcm.Initialize(&aes, aesCcmTestVectors[0].iv, aesCcmTestVectors[0].ivSize,
                          aesCcmTestVectors[0].aadSize, aesCcmTestVectors[0].textSize, aesCcmTestVectors[0].macSize);

        // AAD の入力前に 0 バイトの Update を呼ぶ
        aesCcm.UpdateAad(nullptr, 0);
        aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize);

        // 平文の入力
        aesCcm.Update(output, sizeof(output),
                      aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize);

        // MAC の取得
        aesCcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[0].cipherText, aesCcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[0].mac, aesCcmTestVectors[0].macSize);
    }

    // 0 to Encryption
    {
        // 初期化
        aes.Initialize(aesCcmTestVectors[0].key, aesCcmTestVectors[0].keySize);
        aesCcm.Initialize(&aes, aesCcmTestVectors[0].iv, aesCcmTestVectors[0].ivSize,
                          aesCcmTestVectors[0].aadSize, aesCcmTestVectors[0].textSize, aesCcmTestVectors[0].macSize);

        // AAD の入力
        aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize);

        // 平文の入力前に 0 バイトの Update を呼ぶ
        EXPECT_EQ(aesCcm.Update(output, sizeof(output), nullptr, 0), 0);
        aesCcm.Update(output, sizeof(output),
                      aesCcmTestVectors[0].plainText ,aesCcmTestVectors[0].textSize);

        // MAC の取得
        aesCcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[0].cipherText, aesCcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[0].mac, aesCcmTestVectors[0].macSize);
    }
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesCcmTest, ZeroByteDecryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CcmDecryptor<nn::crypto::AesEncryptor128> aesCcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];

    // 0 to AAD
    {
        // 初期化
        aes.Initialize(aesCcmTestVectors[0].key, aesCcmTestVectors[0].keySize);
        aesCcm.Initialize(&aes, aesCcmTestVectors[0].iv, aesCcmTestVectors[0].ivSize,
                          aesCcmTestVectors[0].aadSize, aesCcmTestVectors[0].textSize, aesCcmTestVectors[0].macSize);

        // AAD の入力前に 0 バイトの Update を呼ぶ
        aesCcm.UpdateAad(nullptr, 0);
        aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize);

        // 平文の入力
        aesCcm.Update(output, sizeof(output),
                      aesCcmTestVectors[0].cipherText, aesCcmTestVectors[0].textSize);

        // MAC の取得
        aesCcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[0].plainText, aesCcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[0].mac, aesCcmTestVectors[0].macSize);
    }

    // 0 to Decryption
    {
        // 初期化
        aes.Initialize(aesCcmTestVectors[0].key, aesCcmTestVectors[0].keySize);
        aesCcm.Initialize(&aes, aesCcmTestVectors[0].iv, aesCcmTestVectors[0].ivSize,
                          aesCcmTestVectors[0].aadSize, aesCcmTestVectors[0].textSize, aesCcmTestVectors[0].macSize);

        // AAD の入力
        aesCcm.UpdateAad(aesCcmTestVectors[0].aad, aesCcmTestVectors[0].aadSize);

        // 平文の入力前に 0 バイトの Update を呼ぶ
        EXPECT_EQ(aesCcm.Update(output, sizeof(output), nullptr, 0), 0);
        aesCcm.Update(output, sizeof(output),
                      aesCcmTestVectors[0].cipherText, aesCcmTestVectors[0].textSize);

        // MAC の取得
        aesCcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesCcmTestVectors[0].plainText, aesCcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesCcmTestVectors[0].mac, aesCcmTestVectors[0].macSize);
    }
}
