﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace TestUtility
{
    public class TestPath
    {
        public TestPath(TestContext testContext)
        {
            Assert.IsNotNull(testContext);
            this.testContext = testContext;
        }

        private const string SigloRootMarkFileName = "SigloRootMark";
        private const string TestRootPath = "Tests";
        private const string ToolTestRootPath = "Tests/Tools";
        private const string ResourceDirectoryName = "Resources";

        private TestContext testContext;

        public string GetSigloRoot()
        {
            return FindSigloRoot(System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location));
        }

        public string FindSigloRoot(string path)
        {
            string startPath = path;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                string sigloRootMarkFilePath = System.IO.Path.Combine(currentDirectory, SigloRootMarkFileName);
                if (System.IO.File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = System.IO.Path.GetFullPath(System.IO.Path.Combine(currentDirectory, ".."));
            }

            throw new InvalidDataException(string.Format("SigloRootMark を見つけられませんでした。\nPath=\n{0}", startPath));
        }

        public string GetTestRoot()
        {
            return Path.Combine(GetSigloRoot(), TestRootPath);
        }

        /// <summary>
        /// テスト出力ディレクトリの取得
        ///
        /// 今は TestContext.TestRunDirectory を返すだけの実装です。
        /// 将来的に適切な出力ファイルの構成を整える可能性があります。
        /// </summary>
        /// <returns></returns>
        public string GetOutputRoot()
        {
            return testContext.TestRunDirectory;
        }

        public string GetToolTestResourceDirectory()
        {
            return Path.Combine(GetSigloRoot(), ToolTestRootPath, ResourceDirectoryName);
        }

        public static string GetNewTemporaryDirectoryPath(string baseName)
        {
            var now = DateTime.Now;
            var path = string.Format(
                "{0}_{1}{2:D3}\\",
                Path.Combine(Path.GetTempPath(), baseName),
                now.ToString("yyMMdd"),
                new Random().Next(1000));
            return path;
        }

        public static string CreateTemporaryDirectory(string baseName)
        {
            var path = GetNewTemporaryDirectoryPath(baseName);

            DeleteDirectoryIfExisted(path);
            Directory.CreateDirectory(path);

            return path;
        }

        public static void DeleteDirectoryIfExisted(string path)
        {
            if (Directory.Exists(path))
            {
                Directory.Delete(path, true);
            }
            // Directory.Delete は非同期なので、削除されるまで待つ
            const int WaitForDeleteDirectoryTimeOutMilliSec = 1000 * 3;
            const int WaitForDeleteDirectoryWaitUnitMilliSec = 100;
            for (int waitMilliSec = 0; waitMilliSec < WaitForDeleteDirectoryTimeOutMilliSec; waitMilliSec += WaitForDeleteDirectoryWaitUnitMilliSec)
            {
                if (Directory.Exists(path) == false)
                {
                    break;
                }
                System.Threading.Thread.Sleep(WaitForDeleteDirectoryWaitUnitMilliSec);
            }
        }
    }
}
