﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Fx3Control/Fx3Control.h"

namespace nnt {
namespace usb {
namespace cdmsc {

static const uint32_t KeepAlivePeriodInMs = 2 * 60 * 1000;
extern bool g_DownloadFirmware;

TEST(Fx3MscTestSuite, KeepAlive14_0)
{
    int expectedDummyReadsCount = 2;
    int keepAliveDurationInMs = KeepAlivePeriodInMs * expectedDummyReadsCount;

    nn::cdmsc::UnitProfile profile;
    uint16_t logType = 0;
    int logIndex = 0;
    int countReadCommand = 0;
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    // clear command log
    logType = Fx3ClearCommandLog << 8;

    Fx3SendVendorSpecificRequest(
        nullptr,
        0x40,
        Fx3Request_ClearLog,
        logType,
        0x00,
        0x00
    );

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(keepAliveDurationInMs + 1000));

    // Get Command log and check for Read10 commands
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );

    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_Read10)
        {
            countReadCommand++;
        }
    }

    EXPECT_EQ(expectedDummyReadsCount, countReadCommand);
}


TEST(Fx3MscTestSuite, ThirteenTestCases_2_3)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };
    bool hasMassStorageReset = false;
    int logIndex = 0;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = Fx3MscCommand_TestUnitReady << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0x02;
    testResponse.u.FailAndSetStatusResidue.Residue = 0;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    // get command logs and check MassStorageReset was called
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );
    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_MassStorageReset)
        {
            hasMassStorageReset = true;
        }
    }

    EXPECT_TRUE(hasMassStorageReset);

}


const int DataSize = 2048;


TEST(Fx3MscTestSuite, ThirteenTestCases_4)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t readData[DataSize] = { 0 };
    uint32_t block = 1;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = Fx3MscCommand_Read10 << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0;
    testResponse.u.FailAndSetStatusResidue.Residue = 2048;

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    block = 2048 / profile.blockSize;

    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultIncompleteTransfer, nn::cdmsc::Read(readData, profile.handle, 0, block));
}


TEST(Fx3MscTestSuite, ThirteenTestCases_5)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t readData[DataSize] = { 0 };
    uint32_t block = 1;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);


    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    // Wait for dummy read from enumeration process to finish
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    commandInfo = Fx3MscCommand_Read10 << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0x00;
    testResponse.u.FailAndSetStatusResidue.Residue = 1024;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    if (DataSize > profile.blockSize)
    {
        block = DataSize / profile.blockSize;
    }

    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultIncompleteTransfer, nn::cdmsc::Read(readData, profile.handle, 0, block));
}


TEST(Fx3MscTestSuite, ThirteenTestCases_7_8)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };
    bool hasMassStorageReset = false;
    int logIndex = 0;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = Fx3MscCommand_Read10 << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0x02;
    testResponse.u.FailAndSetStatusResidue.Residue = 0;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    // get command logs and check MassStorageReset was called
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );
    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_MassStorageReset)
        {
            hasMassStorageReset = true;
            break;
        }
    }

    EXPECT_TRUE(hasMassStorageReset);
}

TEST(Fx3MscTestSuite, ThirteenTestCases_9)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t writeData[DataSize] = { 0 };
    uint32_t block = 1;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = Fx3MscCommand_Write10 << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0x00;
    testResponse.u.FailAndSetStatusResidue.Residue = 2048;

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    // Wait for dummy read from enumeration process to finish
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    block = 2048 / profile.blockSize;

    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultIncompleteTransfer, nn::cdmsc::Write(writeData, profile.handle, 0, block));
}

TEST(Fx3MscTestSuite, ThirteenTestCases_11)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t writeData[DataSize] = { 0 };
    uint32_t block = 1;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = Fx3MscCommand_Write10 << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0x00;
    testResponse.u.FailAndSetStatusResidue.Residue = 1024;

    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    // Wait for dummy read from enumeration process to finish
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );

    if (DataSize > profile.blockSize)
    {
        block = DataSize / profile.blockSize;
    }

    NNT_EXPECT_RESULT_FAILURE(nn::cdmsc::ResultIncompleteTransfer, nn::cdmsc::Write(writeData, profile.handle, 0, block));
}


TEST(Fx3MscTestSuite, ThirteenTestCases_10_13)
{
    nn::cdmsc::UnitProfile profile;
    uint16_t commandInfo = 0;
    NN_USB_DMA_ALIGN Fx3TestResponse testResponse = { 0 };
    NN_USB_DMA_ALIGN uint8_t writeData[DataSize] = { 0 };
    NN_USB_DMA_ALIGN uint8_t readLogData[Fx3CommandLogLength] = { 0 };
    uint32_t block = 1;
    bool hasMassStorageReset = false;
    int logIndex = 0;

    Fx3Reset(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);

    commandInfo = Fx3MscCommand_Write10 << 8;

    testResponse.Type = Fx3MscTestResponseType_FailSetStatusResidue;
    testResponse.u.FailAndSetStatusResidue.Status = 0x02;
    testResponse.u.FailAndSetStatusResidue.Residue = 0;

    Fx3SendVendorSpecificRequest(
        (uint8_t*)&testResponse,
        0x40,
        Fx3Request_SetTestResponse,
        commandInfo,
        0,
        sizeof(testResponse)
    );
    Fx3ActivateMassStorageMode(WAIT_SECONDS_FOR_ATTACH);
    NNT_EXPECT_RESULT_SUCCESS(Fx3Probe(&profile));

    if (DataSize > profile.blockSize)
    {
        block = DataSize / profile.blockSize;
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::cdmsc::Write(writeData, profile.handle, 0, block));

    // get command logs and check MassStorageReset was called
    Fx3SendVendorSpecificRequest(
        readLogData,
        0xC0,
        Fx3Request_GetCommandLog,
        0x00,
        0x00,
        Fx3CommandLogLength
    );
    for (logIndex = 0; logIndex < Fx3CommandLogLength; logIndex++)
    {
        if (readLogData[logIndex] == Fx3MscCommand_MassStorageReset)
        {
            hasMassStorageReset = true;
        }
    }

    EXPECT_TRUE(hasMassStorageReset);
}

} // cdmsc
} // usb
} // nnt
