﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <atomic>
#include <limits>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/ovln/ovln_ReceiverForOverlay.h>
#include <nn/ovln/format/ovln_CapsrvMessage.h>


struct ScreenShotNotificationCount
{
    int countSuccess;
    int countFailure;
    int countProhibited;
    int countThumbnail;
};

NN_ALIGNAS(4096) char g_BurstThreadStack[4096];
NN_ALIGNAS(4096) char g_CountThreadStack[4096];

nn::os::ThreadType g_BurstThread;
nn::os::ThreadType g_CountThread;

nn::os::BarrierType g_Barrier;

std::atomic_bool g_StopFlag;
ScreenShotNotificationCount g_Counts = {};



void BurstThreadFunction(void*) NN_NOEXCEPT
{
    nn::TimeSpan fireInterval = nn::TimeSpan::FromMilliSeconds(5);

    nn::os::AwaitBarrier(&g_Barrier);

    uint64_t seqNo = std::numeric_limits<uint64_t>::max();
    nn::ncm::ProgramId programId = {};
    nn::applet::AppletResourceUserId aruid = {};
    nn::capsrv::ScreenShotAttribute attribute;
    attribute.SetDefault();
    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(500);
    while(!g_StopFlag)
    {
        nn::os::SleepThread(fireInterval);
        nn::capsrv::RequestTakingScreenShot(seqNo, programId, aruid, attribute, {}, timeout);
        programId.value++;
    }
}

void CountThreadFunction(void*) NN_NOEXCEPT
{
    nn::ovln::ReceiverForOverlayType receiver;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::ovln::InitializeReceiver(&receiver)
    );

    ScreenShotNotificationCount counts = {};

    nn::os::AwaitBarrier(&g_Barrier);

    while(!g_StopFlag)
    {
        nn::ovln::Message message;
        nn::ovln::Receive(&message, &receiver);

        if(message.tag == nn::ovln::format::ScreenShotSuccessDataTag)
        {
            counts.countSuccess++;
        }
        else if(message.tag == nn::ovln::format::ScreenShotThumbnailIsReadyDataTag)
        {
            counts.countThumbnail++;
        }
        else if(message.tag == nn::ovln::format::ScreenShotIsNotPermittedDataTag)
        {
            counts.countProhibited++;
        }
        else if(message.tag == nn::ovln::format::ScreenShotFailureDataTag)
        {
            counts.countFailure++;
        }
    }

    g_Counts = counts;

    nn::ovln::FinalizeReceiver(&receiver);
}

extern "C" void nninitStartup()
{
}

extern "C" void nnMain()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::capsrv::InitializeScreenShotControl()
    );
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::ovln::InitializeReceiverLibraryForOverlay()
    );

    nn::TimeSpan duration = nn::TimeSpan::FromSeconds(60);
    g_StopFlag = false;

    nn::os::InitializeBarrier(&g_Barrier, 3);
    nn::os::CreateThread(&g_BurstThread, BurstThreadFunction, nullptr, g_BurstThreadStack, sizeof(g_BurstThreadStack), nn::os::DefaultThreadPriority, 1);
    nn::os::CreateThread(&g_CountThread, CountThreadFunction, nullptr, g_CountThreadStack, sizeof(g_CountThreadStack), nn::os::DefaultThreadPriority, 2);

    nn::os::StartThread(&g_BurstThread);
    nn::os::StartThread(&g_CountThread);

    nn::os::AwaitBarrier(&g_Barrier);

    nn::os::SleepThread(duration);

    g_StopFlag = true;


    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    {
        // 念の為 ovln にもう 1 個通知を入れておく
        nn::ncm::ProgramId programId = {};
        nn::capsrv::NotifyTakingScreenShotRefused(programId);
    }


    nn::os::WaitThread(&g_BurstThread);
    nn::os::WaitThread(&g_CountThread);
    nn::os::DestroyThread(&g_BurstThread);
    nn::os::DestroyThread(&g_CountThread);

    NN_LOG("Counts:\n");
    NN_LOG("  Success   : %d (%.2f/sec)\n", g_Counts.countSuccess   , static_cast<float>(g_Counts.countSuccess   ) / static_cast<float>(duration.GetSeconds()));
    NN_LOG("  Failure   : %d (%.2f/sec)\n", g_Counts.countFailure   , static_cast<float>(g_Counts.countFailure   ) / static_cast<float>(duration.GetSeconds()));
    NN_LOG("  Prohibited: %d (%.2f/sec)\n", g_Counts.countProhibited, static_cast<float>(g_Counts.countProhibited) / static_cast<float>(duration.GetSeconds()));
    NN_LOG("  Thumbnail : %d (%.2f/sec)\n", g_Counts.countThumbnail , static_cast<float>(g_Counts.countThumbnail ) / static_cast<float>(duration.GetSeconds()));

    nn::ovln::FinalizeReceicerLibraryForOverlay();
    nn::capsrv::FinalizeScreenShotControl();
}
