﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 以下の API のテストを行います
// - LoadAlbumScreenShotFile

#include <algorithm>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>
#include <nn/util/util_StringUtil.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_TestFileUtility.h"
#include "testCapsrv_StartupTestCase.h"

TEST(AlbumAccessApi, LoadAlbumScreenShotFile)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 1024, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .jpg, 1024, 12346),
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .jpg, 1024, 12347),
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .jpg, 1024, 12348),
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .jpg, 1024, 12349),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    const nn::capsrv::AlbumStorageType StorageList[] = { nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumStorage_Sd };
    static const int StorageCount = sizeof(StorageList) / sizeof(StorageList[0]);

    // ファイル作成
    for(int i = 0; i < TotalFileCount; i++)
    {
        FileList[i].CreateJpeg();
    }

    // 作った全ファイルを検査。
    for(int s = 0; s < StorageCount; s++)
    {
        nn::capsrv::AlbumStorageType storage = StorageList[s];
        nn::capsrv::AlbumEntry albumEntries[TotalFileCount];
        if(!nn::capsrv::IsAlbumMounted(storage))
        {
            continue;
        }
        int count = 0;
        nn::Result listResult = nn::capsrv::GetAlbumFileList(&count, albumEntries, TotalFileCount, storage);
        EXPECT_TRUE(listResult.IsSuccess());

        for(int k = 0; k < count; k++)
        {
            const nn::capsrv::AlbumEntry& e = albumEntries[k];
            // 対応する FileInfo を探す
            const nnt::capsrv::FileInfo* pFileInfo = nullptr;
            for(int i = 0; i < TotalFileCount; i++)
            {
                if(FileList[i].storage == storage && FileList[i].IsMatchTime(e))
                {
                    pFileInfo = &FileList[i];
                    break;
                }
            }
            EXPECT_TRUE(pFileInfo != nullptr);

            std::vector<uint8_t> data;
            data.resize(static_cast<size_t>(e.size));

            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotFile(&readSize, data.data(), data.size(), &e.fileId));
            EXPECT_EQ(e.size, static_cast<size_t>(readSize));

            EXPECT_LE(pFileInfo->CheckDataJpeg(data), 0.25); // やってみたら 0.25 くらいだった
        }
    }

    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}

TEST(AlbumAccessApi, LoadAlbumFile_FileSizeLimit)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot    , 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot + 1, 12345),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    // ファイル作成
    int createdFileCount = 0;
    for(int i = 0; i < TotalFileCount; i++)
    {
        if(FileList[i].Create())
        {
            createdFileCount++;
        }
    }

    std::vector<char> buffer;
    buffer.resize(2 * nn::capsrv::AlbumFileSizeLimit_ScreenShot);

    NN_LOG("Check file size limitation for LoadAlbumScreenShotFile()\n");
    {
        {
            auto& fileInfo = FileList[0];
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId = fileInfo.appId;
            fileId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            fileId.storage = fileInfo.storage;
            fileId.time = fileInfo.time;

            size_t size;
            // ファイルとして認識はするものの、内容は乱数なので読み込みは失敗する
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nn::capsrv::LoadAlbumScreenShotFile(&size, buffer.data(), buffer.size(), &fileId)
            );
        }
        {
            auto& fileInfo = FileList[1];
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId = fileInfo.appId;
            fileId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            fileId.storage = fileInfo.storage;
            fileId.time = fileInfo.time;

            size_t size;
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumFileNotFound,
                nn::capsrv::LoadAlbumScreenShotFile(&size, buffer.data(), buffer.size(), &fileId)
            );
        }
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, LoadAlbumFile_FileDataVerification)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 0 /*unused*/, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 02, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 0 /*unused*/, 12345), // 日付違い
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x123456789ABCDEF0, .jpg, 0 /*unused*/, 12345), // AppID 違い
        NNT_CAPSRV_FILEINFO(NA, 2016, 03, 03, 05, 06, 07, 01, 0x123456789ABCDEF0, .jpg, 0 /*unused*/, 12345), // Bad Signature
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 01, 01, 01, 01, 0x123456789ABCDEF0, .jpg, 0 /*unused*/, 12345), // empty
    };
    const int GoodFileIndex             = 0;
    const int BadDateTimeFileIndex      = 1;
    const int BadApplicationIdFileIndex = 2;
    const int BadSignatureFileIndex     = 3;
    const int EmpyFileIndex             = 4;

    // 基準のファイルを作成
    EXPECT_TRUE(FileList[GoodFileIndex].CreateJpeg());

    // 作ったファイルを読み込み
    NN_LOG("Checking valid file\n");
    std::vector<uint8_t> data0;
    data0.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
    {
        size_t readSize = 0;
        auto fileId = FileList[GoodFileIndex].GetAlbumFileId();
        EXPECT_TRUE(nn::capsrv::LoadAlbumScreenShotFile(&readSize, data0.data(), data0.size(), &fileId).IsSuccess());
        data0.resize(readSize);
    }

    std::vector<uint8_t> readBuffer;
    readBuffer.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);

    // 日付違いが読めないことを確認
    NN_LOG("Checking inconsistent DateTime\n");
    {
        auto& fileInfo = FileList[BadDateTimeFileIndex];
        // ファイルを保存
        auto filepath = fileInfo.GetDirectFilePath();
        nnt::capsrv::DirectAlbumAccessor::SaveFile(data0, filepath.c_str());
        auto fileId = fileInfo.GetAlbumFileId();
        // 読もうとしてみる
        size_t readSize = 0;
        auto result = nn::capsrv::LoadAlbumScreenShotFile(&readSize, readBuffer.data(), readBuffer.size(), &fileId);
        // ファイルデータの不正ではじかれるべき
        EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
    }

    // アプリ ID 違いが読めないことを確認
    NN_LOG("Checking inconsistent ApplicationId\n");
    {
        auto& fileInfo = FileList[BadApplicationIdFileIndex];
        // ファイルを保存
        auto filepath = fileInfo.GetDirectFilePath();
        nnt::capsrv::DirectAlbumAccessor::SaveFile(data0, filepath.c_str());
        auto fileId = fileInfo.GetAlbumFileId();
        // 読もうとしてみる
        size_t readSize = 0;
        auto result = nn::capsrv::LoadAlbumScreenShotFile(&readSize, readBuffer.data(), readBuffer.size(), &fileId);
        // ファイルデータの不正ではじかれるべき
        EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
    }

    // 不正な署名のファイルを読めないことを確認
    NN_LOG("Checking inconsistent Signature\n");
    {
        auto& fileInfo = FileList[BadSignatureFileIndex];
        EXPECT_TRUE(fileInfo.CreateJpeg(nnt::capsrv::TestFileSignaturePattern_HeadBitFlip));
        auto fileId = fileInfo.GetAlbumFileId();
        size_t readSize = 0;
        auto result = nn::capsrv::LoadAlbumScreenShotFile(&readSize, readBuffer.data(), readBuffer.size(), &fileId);
        // ファイルデータの不正ではじかれるべき
        EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
    }

    // 空のファイルを読めないことを確認
    NN_LOG("Checking empty file\n");
    {
        auto& fileInfo = FileList[EmpyFileIndex];
        // ファイルを保存
        auto filepath = fileInfo.GetDirectFilePath();
        std::vector<uint8_t> emptyData;
        nnt::capsrv::DirectAlbumAccessor::SaveFile(emptyData, filepath.c_str());
        auto fileId = fileInfo.GetAlbumFileId();
        // 読もうとしてみる
        size_t readSize = 0;
        auto result = nn::capsrv::LoadAlbumScreenShotFile(&readSize, readBuffer.data(), readBuffer.size(), &fileId);
        // ファイルデータの不正ではじかれるべき
        EXPECT_TRUE(nn::capsrv::ResultAlbumInvalidFileData::Includes(result));
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();

}
