﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <vector>
#include <random>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nnt.h>

//#define NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER(...) NN_LOG(__VA_ARGS__)

#ifndef NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER
#define NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER(...)
#endif


namespace nnt{ namespace capsrv{ namespace detail{

    class TestMovieStreamWrapperBase
    {
    public:
        ~TestMovieStreamWrapperBase()
        {
        }

    protected:
        virtual nn::Result WriteImpl(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT = 0;
        virtual nn::Result ReadImpl(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT = 0;
        virtual nn::Result GetSizeImpl(int64_t* pOutValue) NN_NOEXCEPT = 0;
        virtual nn::Result SetSizeImpl(int64_t size) NN_NOEXCEPT = 0;
        virtual nn::Result FlushImpl() NN_NOEXCEPT = 0;
        virtual const uint8_t* GetInternalStorageImpl(size_t* pOutSize) NN_NOEXCEPT = 0;

    public:
        nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
        {
            NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER("write\n");
            if(size > 0)
            {
                this->shadow.resize(std::max(this->shadow.size(), static_cast<size_t>(offset) + size), 0);
                std::memcpy(this->shadow.data() + offset, buffer, size);
            }
            return WriteImpl(offset, buffer, size);
        }

        nn::Result Read(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT
        {
            NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER("read\n");
            return ReadImpl(pOutReadSize, buffer, size, offset);
        }

        nn::Result Resize(int64_t size) NN_NOEXCEPT
        {
            NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER("resize\n");
            this->shadow.resize(static_cast<size_t>(size), 0);
            return SetSizeImpl(size);
        }

        nn::Result GetSize(int64_t* pOutValue) NN_NOEXCEPT
        {
            NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER("getSize\n");
            return GetSizeImpl(pOutValue);
        }

        nn::Result Flush() NN_NOEXCEPT
        {
            NNT_CAPSRV_LOG_TESTMOVIESTREAMWRAPPER("flush\n");
            return FlushImpl();
        }

        nn::Result WriteRandom(int64_t offset, int64_t size, std::mt19937& rand) NN_NOEXCEPT
        {
            std::vector<uint8_t> tmp(static_cast<size_t>(size), 0);
            std::uniform_int_distribution<int> dist;
            for(size_t i = 0; i < tmp.size(); i++)
            {
                tmp[i] = static_cast<uint8_t>(dist(rand));
            }
            NN_RESULT_DO(Write(offset, tmp.data(), tmp.size()));
            NN_RESULT_SUCCESS;
        }

        bool CheckRead(int64_t offset, int64_t size) NN_NOEXCEPT
        {
            std::vector<uint8_t> tmp(static_cast<size_t>(size));
            size_t readSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(Read(&readSize, tmp.data(), tmp.size(), offset));
            NN_ABORT_UNLESS_EQUAL(readSize, static_cast<size_t>(size));
            size_t preSize = this->shadow.size();
            this->shadow.resize(std::max(this->shadow.size(), static_cast<size_t>(offset + size)), 0);

            int64_t diffCount = 0;
            for(int64_t pos = 0; pos < size; pos++)
            {
                if(this->shadow[static_cast<size_t>(offset + pos)] != tmp[static_cast<size_t>(pos)])
                {
                    if(diffCount == 0)
                    {
                        NN_LOG("difference detected at pos = %lld (%lld)\n", pos, offset + pos);
                    }
                    diffCount++;
                }
            }
            if(diffCount)
            {
                NN_LOG("diffCount == %lld\n", diffCount);
            }

            bool result = diffCount == 0;
            this->shadow.resize(preSize);
            return result;
        }

        bool CheckStorage() NN_NOEXCEPT
        {
            int64_t fileSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(GetSize(&fileSize));
            size_t shadowSize = this->shadow.size();
            EXPECT_EQ(shadowSize, fileSize);

            std::vector<uint8_t> data(static_cast<size_t>(fileSize));
            const uint8_t* pStorageData = nullptr;
            size_t storageSize = 0;
            pStorageData = GetInternalStorageImpl(&storageSize);
            if(!pStorageData)
            {
                data.resize(static_cast<size_t>(fileSize));
                size_t readSize = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(Read(&readSize, data.data(), data.size(), 0));
                pStorageData = data.data();
                storageSize = data.size();
            }
            NN_ASSERT(storageSize >= static_cast<size_t>(fileSize));

            int64_t diffCount = 0;
            for(int64_t pos = 0; pos < fileSize; pos++)
            {
                if(this->shadow[static_cast<size_t>(pos)] != pStorageData[static_cast<size_t>(pos)])
                {
                    if(diffCount == 0)
                    {
                        NN_LOG("difference detected at pos = %lld\n", pos);
                    }
                    diffCount++;
                }
            }
            if(diffCount)
            {
                NN_LOG("diffCount == %lld\n", diffCount);
            }

            return diffCount == 0;
        }

    public:
        std::vector<uint8_t> shadow;
    };

}}}
