﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_WorkerThreadPool.h"

#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

namespace nnt{ namespace capsrv{ namespace threading{

    const int WorkerThreadPool::ThreadCountMax;

    WorkerThreadPool::WorkerThreadPool() NN_NOEXCEPT
        : m_ThreadCount(0)
        , m_pStackMemory(nullptr)
    {
    }

    void WorkerThreadPool::Initialize(int threadCount, size_t stackSize, int workerThreadPriority) NN_NOEXCEPT
    {
        NN_ASSERT(m_pStackMemory == nullptr);
        NN_ASSERT_MINMAX(threadCount, 0, ThreadCountMax);
        NN_ASSERT(stackSize % nn::os::MemoryPageSize == 0);

        auto pStackMemory = reinterpret_cast<char*>(aligned_alloc(nn::os::MemoryPageSize, stackSize * threadCount));
        NN_ASSERT_NOT_NULL(pStackMemory);

        for(int i = 0; i < threadCount; i++)
        {
            m_ContextList[i].m_ThreadNumber = i;
            m_ContextList[i].m_RequestQueue.Initialize();
            m_ContextList[i].m_ReplyQueue.Initialize();
        }

        for(int i = 0; i < threadCount; i++)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                &m_ThreadList[i],
                WorkerThreadFunction,
                &m_ContextList[i],
                pStackMemory + stackSize * i,
                stackSize,
                workerThreadPriority,
                CoreNumberMin + (i % (CoreNumberMax - CoreNumberMin + 1))
            ));
        }

        m_ThreadCount = threadCount;
        m_ActiveThreadCount = threadCount;
        m_pStackMemory = pStackMemory;
        m_RequestedSequenceCount = 0;
    }

    void WorkerThreadPool::Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(m_pStackMemory != nullptr);
        for(int i = 0; i < m_ThreadCount; i++)
        {
            nn::os::DestroyThread(&m_ThreadList[i]);
        }

        for(int i = 0; i < m_ThreadCount; i++)
        {
            m_ContextList[i].m_RequestQueue.Finalize();
            m_ContextList[i].m_ReplyQueue.Finalize();
        }

        free(m_pStackMemory);

        m_ThreadCount = 0;
        m_ActiveThreadCount = 0;
        m_pStackMemory = nullptr;
    }

    void WorkerThreadPool::SetActiveThreadCount(int value) NN_NOEXCEPT
    {
        NN_ASSERT_MINMAX(value, 1, m_ThreadCount);
        m_ActiveThreadCount = value;
    }

    int WorkerThreadPool::GetActiveThreadCount() const NN_NOEXCEPT
    {
        return m_ActiveThreadCount;
    }

    StaticRange<WorkerThreadPool::ThreadCountMax> WorkerThreadPool::GetStaticRangeDivision(int n) const NN_NOEXCEPT
    {
        return StaticRange<ThreadCountMax>(n, m_ActiveThreadCount);
    }

    //-------------------------------------------------------

    void WorkerThreadPool::StartWorker() NN_NOEXCEPT
    {
        for(int i = 0; i < m_ThreadCount; i++)
        {
            nn::os::StartThread(&m_ThreadList[i]);
        }
    }

    void WorkerThreadPool::StopWorker() NN_NOEXCEPT
    {
        detail::WorkerRequest request = {};
        request.m_Tag = detail::WorkerRequestTag_Exit;
        for(int i = 0; i < m_ThreadCount; i++)
        {
            m_ContextList[i].m_RequestQueue.Enqueue(request);
        }

        for(int i = 0; i < m_ThreadCount; i++)
        {
            nn::os::WaitThread(&m_ThreadList[i]);
        }
    }

    WorkerThreadPool::SequenceId WorkerThreadPool::InvokeFunctionAsync(void (*pFunction)(int threadId, void* arg), void* arg) NN_NOEXCEPT
    {
        NN_ASSERT(m_ThreadCount > 0);
        NN_ASSERT(pFunction != nullptr);
        detail::WorkerRequest request = {};
        request.m_Tag = detail::WorkerRequestTag_InvokeFunction;
        request.m_pFunction = pFunction;
        request.m_pArgument = arg;
        for(int i = 0; i < m_ThreadCount; i++)
        {
            m_ContextList[i].m_RequestQueue.Enqueue(request);
        }

        m_RequestedSequenceCount++;
        return m_RequestedSequenceCount;
    }

    void WorkerThreadPool::WaitComplete(SequenceId id) NN_NOEXCEPT
    {
        NN_UNUSED(id);

        for(int i = 0; i < m_ThreadCount; i++)
        {
            detail::WorkerReply reply = {};
            m_ContextList[i].m_ReplyQueue.Dequeue(&reply);
        }
    }

    void WorkerThreadPool::InvokeFunction(void (*pFunction)(int threadId, void* arg), void* arg) NN_NOEXCEPT
    {
        if(m_ThreadCount > 0)
        {
            auto id = InvokeFunctionAsync(pFunction, arg);
            WaitComplete(id);
        }
        else
        {
            pFunction(0, arg);
        }
    }

    //----------------------------------------------------------

    void WorkerThreadPool::WorkerThreadFunction(void* p) NN_NOEXCEPT
    {
        auto pContext = reinterpret_cast<detail::WorkerThreadContext*>(p);

        NN_LOG("[worker] thread %d start\n", pContext->m_ThreadNumber);

        for(;;)
        {
            detail::WorkerRequest request = {};
            pContext->m_RequestQueue.Dequeue(&request);

            if(request.m_Tag == detail::WorkerRequestTag_Exit)
            {
                break;
            }
            else if(request.m_Tag == detail::WorkerRequestTag_InvokeFunction)
            {
                //NN_LOG("[worker][%d] invoke function ... start\n", pContext->m_ThreadNumber);
                request.m_pFunction(pContext->m_ThreadNumber, request.m_pArgument);
                //NN_LOG("[worker][%d] invoke function ... complete\n", pContext->m_ThreadNumber);
                detail::WorkerReply reply = {};
                reply.m_Tag = detail::WorkerReplyTag_Complete;
                pContext->m_ReplyQueue.Enqueue(reply);
            }
            else
            {
                NN_LOG("[worker][%d] unknown tag %d\n", pContext->m_ThreadNumber, request.m_Tag);
            }
        }

        NN_LOG("[worker] thread %d stop\n", pContext->m_ThreadNumber);
    }


}}}
