﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <random>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/image/image_ExifBuilder.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include <nn/capsrv/capsrv_AlbumFileDescription.h>
#include <nn/capsrv/capsrv_ScreenShotOrientation.h>
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_MakerNoteInfo.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_Signature.h"

namespace nnt{ namespace capsrv{

    struct ScreenShotMetaInfo
    {
        uint64_t makerNoteVersion;
        nn::ncm::ApplicationId applicationId;
        nn::capsrv::AlbumFileDateTime time;
        int width;
        int height;
        nn::capsrv::AlbumFileDescriptionType description;
        nn::capsrv::ScreenShotOrientationType orientation;
        int frameCount;
        int frameRateNumerator;
        int frameRateDenominator;
        int dataDurationMilliseconds;
        int keyFrameInterval;
    };

    struct ScreenShotExif
    {
        std::vector<char> buf;
        nn::image::ExifBuilder* pExifBuilder;
    };

    struct ScreenShotData
    {
        std::vector<uint8_t> value;
        uint64_t makerNoteVersion;
        int64_t makerNoteOffset;
        int64_t makerNoteSize;
    };

    struct ScreenShotSignature
    {
        nn::capsrv::server::detail::Signature value;
        int64_t offset;
    };

    class ScreenShotFileCreator
    {
    public:
        static ScreenShotMetaInfo CreateMetaDefault(
            uint64_t makerNoteVersion,
            nn::ncm::ApplicationId applicationId,
            const nn::capsrv::AlbumFileDateTime time
        ) NN_NOEXCEPT;

        static ScreenShotExif CreateExif(
            const ScreenShotMetaInfo& meta
        ) NN_NOEXCEPT;

        static ScreenShotData CreateDataWithRandomImage(
            const ScreenShotMetaInfo& meta,
            const ScreenShotExif& exif,
            std::mt19937& rand
        ) NN_NOEXCEPT;

        static ScreenShotSignature CreateSignature(
            const ScreenShotMetaInfo& meta,
            const ScreenShotData& data
        ) NN_NOEXCEPT;

        static ScreenShotSignature CreateSignature(
            uint64_t makerNoteVersion,
            const ScreenShotData& data
        ) NN_NOEXCEPT;

        static ScreenShotSignature CreateZeroSignature(
            const ScreenShotMetaInfo& meta,
            const ScreenShotData& data
        ) NN_NOEXCEPT;

        // @param nn::Result (*outputFunc)(int64_t offset, const void* buffer, size_t size)
        template<typename OutputFunction>
        static nn::Result WriteFileData(
            OutputFunction outputFunc,
            const ScreenShotData& data,
            const ScreenShotSignature& sign
        ) NN_NOEXCEPT
        {
            std::vector<uint8_t> tmp;
            tmp = data.value;
            std::memcpy(tmp.data() + sign.offset, &sign.value, sizeof(sign.value));

            NN_RESULT_DO(outputFunc(0, tmp.data(), tmp.size()));

            NN_RESULT_SUCCESS;
        }

    };

}}
