﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_ScreenShotFileCreator.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/visrv_ConstructExifBuilder.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptApplicationId.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ExtractJpegMakerNoteRange.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_CalculateJpegMac.h"

namespace nnt{ namespace capsrv{

    ScreenShotMetaInfo ScreenShotFileCreator::CreateMetaDefault(
        uint64_t makerNoteVersion,
        nn::ncm::ApplicationId applicationId,
        const nn::capsrv::AlbumFileDateTime time
    ) NN_NOEXCEPT
    {
        ScreenShotMetaInfo info = {};
        info.makerNoteVersion = makerNoteVersion;
        info.applicationId = applicationId;
        info.time = time;
        info.width = 1280;
        info.height = 720;
        info.description = nn::capsrv::AlbumFileDescription_ScreenShotCaptured;
        info.orientation = nn::capsrv::ScreenShotOrientation_Default;
        info.frameCount = 1;
        info.frameRateNumerator = 0;
        info.frameRateDenominator = 0;
        info.dataDurationMilliseconds = 0;
        info.keyFrameInterval = 0;
        return info;
    }

    //---------------------------------------------

    ScreenShotExif ScreenShotFileCreator::CreateExif(
        const ScreenShotMetaInfo& meta
    ) NN_NOEXCEPT
    {
        nn::capsrv::server::detail::MakerNoteInfo info = {};
        info.signature = {};
        info.version = meta.makerNoteVersion;
        info.encryptedApplicationId = nn::capsrv::server::detail::EncryptApplicationId(meta.applicationId, false);
        info.fileDescription = meta.description;
        info.dateTime = meta.time;
        info.dataWidth = meta.width;
        info.dataHeight = meta.height;
        info.dataOrientation = meta.orientation;
        info.frameCount = meta.frameCount;
        info.frameRateNumerator = meta.frameRateNumerator;
        info.frameRateDenominator = meta.frameRateDenominator;
        info.dataDurationMilliseconds = meta.dataDurationMilliseconds;
        info.keyFrameInterval = meta.keyFrameInterval;

        ScreenShotExif value = {};
        value.buf.resize(6 * 1024);
        value.pExifBuilder = nn::capsrv::server::detail::ConstructExifBuilder(value.buf.data(), value.buf.size(), info, nullptr, 0);
        return value;
    }

    //---------------------------------------------

    ScreenShotData ScreenShotFileCreator::CreateDataWithRandomImage(
        const ScreenShotMetaInfo& meta,
        const ScreenShotExif& exif,
        std::mt19937& rand
    ) NN_NOEXCEPT
    {
        std::vector<uint64_t> raw;
        raw.resize(meta.width * meta.height * 4 / sizeof(uint64_t));
        uint64_t alphaMask = 0x000000FF000000FF;

        std::uniform_int_distribution<uint64_t> dist;
        for(auto& e : raw)
        {
            e = dist(rand) | alphaMask;
        }

        ScreenShotData out = {};
        out.value.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
        std::vector<char> work;
        nn::image::JpegEncoder encoder;
        nn::image::Dimension dim;
        dim.width = meta.width;
        dim.height = meta.height;
        encoder.SetPixelData(raw.data(), dim, 1);
        int quality;
        for(quality = 95; quality > 0; quality -= 5)
        {
            encoder.SetQuality(quality);
            encoder.Analyze();
            work.resize(encoder.GetAnalyzedWorkBufferSize());
            size_t size = 0;
            auto result = encoder.Encode(&size, out.value.data(), out.value.size(), work.data(), work.size(), exif.pExifBuilder);
            if(result == nn::image::JpegStatus_Ok)
            {
                out.value.resize(size);
                break;
            }
            else if(result == nn::image::JpegStatus_ShortOutput)
            {
                continue;
            }
            else
            {
                NN_ABORT("Unexpected result\n");
            }
        }
        NN_ABORT_UNLESS(quality > 0);

        {
            int64_t makerNoteOffset = 0;
            int64_t makerNoteSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::capsrv::server::detail::ExtractJpegMakerNoteRange(&makerNoteOffset, &makerNoteSize, out.value.data(), out.value.size(), work.data(), work.size())
            );
            out.makerNoteVersion = meta.makerNoteVersion;
            out.makerNoteOffset = makerNoteOffset;
            out.makerNoteSize = makerNoteSize;
        }

        return out;
    }

    //---------------------------------------------

    ScreenShotSignature ScreenShotFileCreator::CreateSignature(
        const ScreenShotMetaInfo& meta,
        const ScreenShotData& data
    ) NN_NOEXCEPT
    {
        return CreateSignature(meta.makerNoteVersion, data);
    }

    ScreenShotSignature ScreenShotFileCreator::CreateSignature(
        uint64_t makerNoteVersion,
        const ScreenShotData& data
    ) NN_NOEXCEPT
    {
        ScreenShotSignature out = {};

        switch(makerNoteVersion)
        {
        case nn::capsrv::server::detail::MakerNoteVersion_Version0:
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    nn::capsrv::server::detail::CalculateJpegMac(&out.value, &out.offset, data.value.data(), data.value.size(), makerNoteVersion, data.makerNoteOffset, data.makerNoteSize)
                );
                break;
            }
        case nn::capsrv::server::detail::MakerNoteVersion_Version1:
        case nn::capsrv::server::detail::MakerNoteVersion_Version2:
        case nn::capsrv::server::detail::MakerNoteVersion_Version3:
            {
                char hash[32] = {};
                nn::crypto::GenerateSha256Hash(hash, sizeof(hash), data.value.data(), data.value.size());
                std::memcpy(&out.value, hash, std::min<size_t>(sizeof(out.value), sizeof(hash)));
                out.offset = data.makerNoteOffset + nn::capsrv::server::detail::MakerNoteVersionPolicyV1::SignatureOffset;
                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }

        return out;
    }

    ScreenShotSignature ScreenShotFileCreator::CreateZeroSignature(
        const ScreenShotMetaInfo& meta,
        const ScreenShotData& data
    ) NN_NOEXCEPT
    {
        ScreenShotSignature out = CreateSignature(meta, data);
        out.value = {};
        return out;
    }

}}
