﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/nn_Common.h>

#include "testBpc_ExpectedEntry.h"

namespace nnt { namespace bpc {

const std::vector<ExpectedEntry> Max77620PmicExpectedEntries =
{
    // ===== コンフィグ =====

    /**
     * CNFGGLBL1
     * [7]  LBDAC_EN: 0b1
     * [6]  MBLPD   : 0b0
     * [5:4]LBHYST  : 0b01 (200 mV)
     * [3:1]LBDAC   : 0b001 (2.8 V)
     * [0]  LBRSTEN : 0b0
     *
     * Reset Value と等しい。
     * MBATT を 2.8 V hysteresis 200 mV で見るが Global-Shutdown も Reset もしない。
     */
    { 0x00, 0x92, 0xff },

    /**
     * CNFGGLBL2
     * [7:6]I2CTWD  : 0b00
     * [5]  GLBL_LPM: 0b0 (Global Low-Power Mode 非発動)
     * [3]  WDTSLPC : 0b0
     * [2]  WDTEN   : 0b0
     * [1:0]TWD     : 0b11 (128 s)
     *
     * Reset Value と等しい。
     * ウォッチドッグ無効。 Global Low-Power Mode 非発動。
     */
    { 0x01, 0x03, 0xef },

    /**
     * CNFGGLBL3
     * [1:0]WDTC    : 0b00
     *
     * Reset Value と等しい。
     * ウォッチドッグクリア操作用レジスタ。
     */
    { 0x02, 0x00, 0x03 },

    /**
     * CNFG1_CLK32K
     * [7]  32K_OK     : 0b1
     * [5:4]32K_LOAD   : 0b11 (10 pF)
     * [2]  32K_OUT0_EN: 0b1 (enabled)
     * [1:0]PWR_MD_32K : 0b00 (Low-Power Mode)
     */
    { 0x03, 0xfc, 0xb7 },

    /**
     * CNFGBBC
     * [7:6]BBCRS      : 0b01 (1 kohm)
     * [5]  BBCLOWIEN  : 0b0 (disable)
     * [4:3]BBCVS      : 0b00 (2.5 V)
     * [2:1]BBCCS      : 0b00 (50 uA)
     * [0]  BBCEN      : 0b0 (backup battery charger off)
     *
     * backup battery charger off でボタン電池への逆流防止。
     */
    { 0x04, 0x40, 0xff },

    // ===== 割込み =====
    // Clear when read なので注意。

    { 0x05, 0x00, 0xfe }, // IRQTOP, Reset Value と等しい。
    { 0x06, 0x00, 0x0e }, // INTLBT, Reset Value と等しい。
    { 0x07, 0x00, 0xf0 }, // IRQSD, Reset Value と等しい。
    { 0x08, 0x00, 0x00 /* 0xff */ }, // IRQ_LVL2_L0-7, Reset Value と等しい。SDEV 初期化直後は 0xc2 になる。
    { 0x09, 0x00, 0x00 /* 0x01 */ }, // IRQ_LVL2_L8, Reset Value と等しい。
    { 0x0a, 0x00, 0xff }, // IRQ_LVL2_GPIO, Reset Value と等しい。
    { 0x0b, 0x00, 0xff }, // ONOFFIRQ, Reset Value と等しい。

    // ===== シャットダウン要因 =====

    { 0x0c, 0x00, 0x00 }, // NVERC, ドライバの実装に依存する箇所ではないのでチェック対象外とする。

    // ===== 割込みマスク =====

    { 0x0d, 0x75, 0xfe }, // IRQTOPM, Reset Value と等しい。
    { 0x0e, 0x00, 0x0f }, // INTENLBT, Reset Value と等しい。[0] GLBLM は全ての割り込みを停止する最上位のマスクなので注意して扱うこと。
    { 0x0f, 0xff, 0xf0 }, // IRQMASKSD, Reset Value と等しい。
    { 0x10, 0xff, 0xff }, // IRQ_MSK_L0-7, Reset Value と等しい。
    { 0x11, 0xff, 0x01 }, // IRQ_MSK_L8, Reset Value と等しい。

    /**
     * ONOFFIRQM
     * [7]  ACOK_RM    : 0b0
     * [6]  ACOK_FM    : 0b0
     * [5]  LID_RM     : 0b1
     * [4]  LID_FM     : 0b1
     * [3]  EN0_RM     : 0b0
     * [2]  EN0_FM     : 0b0
     * [1]  EN0_1SECM  : 0b1
     * [0]  MRWRNM     : 0b1
     *
     * LID は入らない。電源ボタン長押し時の 1 秒後の割り込みと 10 秒後(MRT=0b111 前提)の割り込みは入らない。
     */
    { 0x12, 0x33, 0xff },

    // ===== 割込み監視対象の状態 =====

    { 0x13, 0x10, 0x0f }, // STATLBT, Reset Value と等しい。

    /**
     * STATSD
     * [7]  nPOK_SD0   : 0b0
     * [6]  nPOK_SD1   : 0b0
     * [5]  nPOK_SD2   : 0b0
     * [4]  nPOK_SD3   : 0b0
     *
     * 各 SD は POK を超えた電圧で正常に供給されている。
     */
    { 0x14, 0x07, 0xf0 },

    /**
     * ONOFFSTAT
     * [2]  EN0        : 0b0
     * [1]  ACOK       : 0b0 or 0b1
     * [0]  LID        : 0b0
     *
     * ACOK に給電のあるなしが反映される。環境依存なので無視する。
     */
    { 0x15, 0x02, 0x05 },

    // ===== レギュレータ設定値 =====

    // VSDx
    { 0x16, 0x16, 0x1e, 0xff }, // ADSP 576 MHz: 0.8750 V(FF sample) - 0.9250 V - 0.9750V(SS sample)
    { 0x17, 0x2a, 0xff }, // 1.125 V
    { 0x18, 0x3a, 0xff }, // 1.325 V
    { 0x19, 0x60, 0xff }, // Reset Value と等しい。1.800 V
    //{ 0x1a, 0x20, 0xff }, // Reset Value と等しい。未使用の SD。

    { 0x1b, 0x20, 0xff }, // VDVSSD0, Reset Value と等しい。
    { 0x1c, 0x10, 0xff }, // VDVSSD1, Reset Value と等しい。
    //{ 0x5e, 0x00, 0xff }, // VDVSSD4, Reset Value と等しい。未使用の SD。

    /**
     * CNFG1SDx
     * [7:6]SR_SD      : 0b01 (Slew Rate During DVS: 27.50 mV)
     * [5:4]PWR_MD_SDx : 0b11 (Power Mode Configuration: Normal Operation Mode)
     * [3]  nADE_SDx   : 0b0 (Active Discharge: Enabled)
     * [2]  FPWM_SDx   : 0b0 (Forced PWM Mode: Disabled)
     * [0]  nFSRADE_SDx: 0b1 (Falling Slew Rate Active-Discharge: Enabled)
     */
    { 0x1d, 0x71, 0xff },
    { 0x1e, 0x71, 0xff },
    { 0x1f, 0x71, 0xff },
    { 0x20, 0x71, 0xff },
    //{ 0x21, 0x40, 0xff }, // Reset Value と等しい。未使用の SD。

    /**
     * CNFG2SD
     * [3]  ROVS_EN_SD0: 0b1 (Remove Output Voltage Sense: Enabled)
     * [2]  ROVS_EN_SD1: 0b0 (Remove Output Voltage Sense: Disabled)
     */
    { 0x22, 0x04, 0x06 },

    /**
     * CNFG1_Lx
     * [7:6]PWR_MD_Lx
     * [5:0]TV_Lx
     */
    { 0x23, 0xd0, 0x3f }, // ドックアウト時 0xd0(Normal Mode, 1.200 V), ドックイン時 0x10(Output Disabled, 1.200 V)
    { 0x25, 0xca, 0xff }, // Normal Mode, 1.050 V
    { 0x27, 0xd4, 0x00 }, // SDCard の給電状態と給電電圧はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x29, 0x2e, 0x00 }, // GameCard の給電状態と給電電圧はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2b, 0xc4, 0xff }, // Normal Mode, 0.8500 V
    { 0x2d, 0x14, 0x00 }, // GameCard の給電状態と給電電圧はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2f, 0xea, 0xff }, // Normal Mode, 2.90 V
    { 0x31, 0xc5, 0xff }, // Normal Mode, 1.05 V
    { 0x33, 0x05, 0x3f }, // ドックアウト時 0x05(Output Disabled, 1.05 V), ドックイン時 0xc5(NormalMode, 1.05 V)

    /**
     * CNFG2_Lx
     * [7]  OVCLMP_EN_Lx: 0b1 (Overvoltage Clamp: Enabled)
     * [5:4]COMP_Lx
     * [3]  POK_Lx
     * [1]  ADE_Lx      : 0b1 (Active Discharge: Enabled)
     * [0]  SS_Lx       : 0b0 (Soft-Start Slew Rate Configuration: 100 mV/us)
     *
     * COMP_Lx は LDO0, LDO1, LDO7, LDO8 には存在しない。
     */
    { 0x24, 0xca, 0x83 }, // ドックアウト時 0xca, ドックイン時 0xc2
    { 0x26, 0xca, 0x8b },
    { 0x28, 0xda, 0xb3 }, // SDCard の給電状態はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2a, 0xd2, 0xb3 }, // GameCard の給電状態はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2c, 0xda, 0xbb }, // Reset Value と等しい。
    { 0x2e, 0xd2, 0xb3 }, // GameCard の給電状態はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x30, 0xda, 0xbb }, // Reset Value と等しい。
    { 0x32, 0xca, 0x8b },
    { 0x34, 0xc2, 0x83 }, // ドックアウト時 0xc2, ドックイン時 0xca

    /**
     * CNFG3_LDO
     * [0]  L_B_EN      : 0b0 (LDO bias: Disabled)
     *
     * Reset Value と等しい。
     */
    { 0x35, 0x00, 0x01 },

    // ===== GPIO =====

    /**
     * CNFG_GPIOx
     * [7:6]DBNCx       : 0b00 (When set for GPI, Debounce Configuration: no debounce)
     * [5:4]REFE_IRQx   : 0b00 (When set for GPI, Rising edge and falling edge interrupt configuration: mask interrupt)
     * [3]  DOx
     * [2]  DIx
     * [1]  DIRx (When AMEx=0, 0=general purpose output, 1=general purpose input)
     * [0]  PPDRV (When set for GPO, 0=open-drain, 1=Push-Pull)
     */
    { 0x36, 0x06, 0xff }, // VSYS-AP と結線。VSYS-AP の状態を取得する。
    { 0x37, 0x06, 0xff }, // VSYS-AP と結線。VSYS-AP の状態を取得する。
    { 0x38, 0x00, 0xff }, // Reset Value と等しい。結線なし。
    { 0x39, 0x00, 0xff }, // Reset Value と等しい。RP602Z-EN USB Buck-Boost と結線。後述の AME3 により FPS Active-High Output となる。DO は FPS_GPIO3 に依存して変化する。PPDRV3 は参照される。他は無視される。
    { 0x3a, 0x01, 0xff }, // Reset Value と等しい。結線なし。
    { 0x3b, 0x09, 0xff }, // MAX77621Cpu と結線。logic high 出力。
    { 0x3c, 0x09, 0xff }, // MAX77621Gpu と結線。logic high 出力。
    { 0x3d, 0x02, 0xff }, // Reset Value と等しい。GND と結線。

    { 0x3e, 0x00, 0xff }, // PUE_GPIO, Reset Value と等しい。pull-up enable。すべて Disabled になっている。
    { 0x3f, 0x00, 0xff }, // PDE_GPIO, Reset Value と等しい。pull-down enable。すべて Disabled になっている。

    /**
     * AME_GPIO
     * [7]  AME7        : 0b0
     * [6]  AME6        : 0b0
     * [5]  AME5        : 0b0
     * [4]  AME4        : 0b1
     * [3]  AME3        : 0b1
     * [2]  AME2        : 0b1
     * [1]  AME1        : 0b0
     * [0]  AME0        : 0b0
     *
     * Reset Value と等しい。
     */
    { 0x40, 0x1c, 0xff },

    /**
     * ONOFFCNFG1
     * [7]  SFT_RST     : 0b0
     * [5:3]MRT         : 0b111 (12 s)
     * [2]  SLPEN       : 0b0
     * [1]  PWR_OFF     : 0b0
     * [0]  EN0DLY      : 0b0 (The only delay for EN0 is the debounce circit)
     */
    { 0x41, 0x78, 0xbf },

    /**
     * ONOFFCNFG2
     * [7]  SFT_RST_WK  : 0b0
     * [6]  WDT_RST_WK  : 0b0
     * [5]  nSLP_LPM_MSK: 0b0 (masked)
     * [4]  WK_ACOK     : 0b1
     * [3]  WK_LID      : 0b1
     * [2]  WK_ALARM1R  : 0b0
     * [1]  WK_ALARM2R  : 0b0
     * [0]  WK_EN0      : 0b1
     *
     * 通常時は ALARM から wakeup はしない。ALARM 設定時に 0b1 にする。
     * nSLP_LPM_MSK は Sleep 突入時に 0b1 にする。さもなくば Low-Power Mode に入らないので消費電力が高い。
     */
    { 0x42, 0x19, 0xff }, // reset value

    // ===== フレキシブルパワーシーケンサ =====

    /**
     * CNFGFPSx
     * [5:3]TFPSx       : 0b111 (5120 us)
     * [2:1]SRCFPSx
     * [0]  ENFPSx      : 0b0 (Software Disable)
     */
    { 0x43, 0x38, 0x3f }, // EN0
    { 0x44, 0x3a, 0x3f }, // EN1
    { 0x45, 0x38, 0x3f }, // EN0

    /**
     * FPS_Lx
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x46, 0xc7, 0xc0 }, // FPS3
    { 0x47, 0xc7, 0xc0 }, // FPS3
    { 0x48, 0xc7, 0xc0 }, // FPS3
    { 0x49, 0xc7, 0xc0 }, // FPS3
    { 0x4a, 0x0f, 0xff }, // FPS0, LDO4(RTC) は常時有効。
    { 0x4b, 0xc7, 0xc0 }, // FPS3
    { 0x4c, 0xc7, 0xc0 }, // FPS3
    { 0x4d, 0x5c, 0xff }, // FPS1, LDO7(PLL) は通常時有効、スリープ時無効。
    { 0x4e, 0xf1, 0xc0 }, // FPS3

    /**
     * FPS_SDx
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x4f, 0x4f, 0xff }, // FPS1
    { 0x50, 0x29, 0xff }, // FPS0
    { 0x51, 0x55, 0xff }, // FPS1, IAAA-2812 より LDO1 の為に FPS0 にする必要は無いで結論。
    { 0x52, 0x1b, 0xff }, // FPS0
    //{ 0x53, 0xc7, 0xff }, // reserved

    /** FPS_GPIOx
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x54, 0xc7, 0xc0 }, // Reset Value と等しい。AME1=0 の為使用していない。
    { 0x55, 0x07, 0xff }, // Reset Value と等しい。AME1=1, FPS0 だが GPIO2 は結線されていないので効果はない。
    { 0x56, 0x22, 0xff }, // Reset Value と等しい。AME1=1, FPS0。USB Buck-Boost は常時有効。

    /** FPS_RSO
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x57, 0x38, 0xff }, // Reset Value と等しい。AME1=1, FPS0。双方向リセット(nRST_IO)は常時有効。

    // ===== CID =====
    { 0x58, 0x00, 0x00 },
    { 0x59, 0x00, 0x00 },
    { 0x5a, 0x00, 0x00 },
    { 0x5b, 0x00, 0x00 },
    { 0x5c, 0x35, 0xff },
    { 0x5d, 0x80, 0xfe }, // (SIGLO-64156) 0x80, 0x81 が混在します。
};

const std::vector<ExpectedEntry> Max77620PmicExpectedEntriesForMariko =
{
    // ===== コンフィグ =====

    /**
     * CNFGGLBL1
     * [7]  LBDAC_EN: 0b1
     * [6]  MBLPD   : 0b0
     * [5:4]LBHYST  : 0b01 (200 mV)
     * [3:1]LBDAC   : 0b001 (2.8 V)
     * [0]  LBRSTEN : 0b0
     *
     * Reset Value と等しい。
     * MBATT を 2.8 V hysteresis 200 mV で見るが Global-Shutdown も Reset もしない。
     */
    { 0x00, 0x92, 0xff },

    /**
     * CNFGGLBL2
     * [7:6]I2CTWD  : 0b00
     * [5]  GLBL_LPM: 0b0 (Global Low-Power Mode 非発動)
     * [3]  WDTSLPC : 0b0
     * [2]  WDTEN   : 0b0
     * [1:0]TWD     : 0b11 (128 s)
     *
     * Reset Value と等しい。
     * ウォッチドッグ無効。 Global Low-Power Mode 非発動。
     */
    { 0x01, 0x03, 0xef },

    /**
     * CNFGGLBL3
     * [1:0]WDTC    : 0b00
     *
     * Reset Value と等しい。
     * ウォッチドッグクリア操作用レジスタ。
     */
    { 0x02, 0x00, 0x03 },

    /**
     * CNFG1_CLK32K
     * [7]  32K_OK     : 0b1
     * [5:4]32K_LOAD   : 0b11 (10 pF)
     * [2]  32K_OUT0_EN: 0b1 (enabled)
     * [1:0]PWR_MD_32K : 0b00 (Low-Power Mode)
     */
    { 0x03, 0xfc, 0xb7 },

    /**
     * CNFGBBC
     * [7:6]BBCRS      : 0b01 (1 kohm)
     * [5]  BBCLOWIEN  : 0b0 (disable)
     * [4:3]BBCVS      : 0b00 (2.5 V)
     * [2:1]BBCCS      : 0b00 (50 uA)
     * [0]  BBCEN      : 0b0 (backup battery charger off)
     *
     * backup battery charger off でボタン電池への逆流防止。
     */
    { 0x04, 0x40, 0xff },

    // ===== 割込み =====
    // Clear when read なので注意。

    { 0x05, 0x00, 0xfe }, // IRQTOP, Reset Value と等しい。
    { 0x06, 0x00, 0x0e }, // INTLBT, Reset Value と等しい。
    { 0x07, 0x00, 0xf0 }, // IRQSD, Reset Value と等しい。
    { 0x08, 0x00, 0x00 /* 0xff */ }, // IRQ_LVL2_L0-7, Reset Value と等しい。SDEV 初期化直後は 0xc2 になる。
    { 0x09, 0x00, 0x00 /* 0x01 */ }, // IRQ_LVL2_L8, Reset Value と等しい。
    { 0x0a, 0x00, 0xff }, // IRQ_LVL2_GPIO, Reset Value と等しい。
    { 0x0b, 0x00, 0xff }, // ONOFFIRQ, Reset Value と等しい。

    // ===== シャットダウン要因 =====

    { 0x0c, 0x00, 0x00 }, // NVERC, ドライバの実装に依存する箇所ではないのでチェック対象外とする。

    // ===== 割込みマスク =====

    { 0x0d, 0x75, 0xfe }, // IRQTOPM, Reset Value と等しい。
    { 0x0e, 0x00, 0x0f }, // INTENLBT, Reset Value と等しい。[0] GLBLM は全ての割り込みを停止する最上位のマスクなので注意して扱うこと。
    { 0x0f, 0xff, 0xf0 }, // IRQMASKSD, Reset Value と等しい。
    { 0x10, 0xff, 0xff }, // IRQ_MSK_L0-7, Reset Value と等しい。
    { 0x11, 0xff, 0x01 }, // IRQ_MSK_L8, Reset Value と等しい。

    /**
     * ONOFFIRQM
     * [7]  ACOK_RM    : 0b0
     * [6]  ACOK_FM    : 0b0
     * [5]  LID_RM     : 0b1
     * [4]  LID_FM     : 0b1
     * [3]  EN0_RM     : 0b0
     * [2]  EN0_FM     : 0b0
     * [1]  EN0_1SECM  : 0b1
     * [0]  MRWRNM     : 0b1
     *
     * LID は入らない。電源ボタン長押し時の 1 秒後の割り込みと 10 秒後(MRT=0b111 前提)の割り込みは入らない。
     */
    { 0x12, 0x33, 0xff },

    // ===== 割込み監視対象の状態 =====

    { 0x13, 0x10, 0x0f }, // STATLBT, Reset Value と等しい。

    /**
     * STATSD
     * [7]  nPOK_SD0   : 0b0
     * [6]  nPOK_SD1   : 0b0
     * [5]  nPOK_SD2   : 0b0
     * [4]  nPOK_SD3   : 0b0
     *
     * 各 SD は POK を超えた電圧で正常に供給されている。
     */
    { 0x14, 0x07, 0xf0 },

    /**
     * ONOFFSTAT
     * [2]  EN0        : 0b0
     * [1]  ACOK       : 0b0 or 0b1
     * [0]  LID        : 0b0
     *
     * ACOK に給電のあるなしが反映される。環境依存なので無視する。
     */
    { 0x15, 0x02, 0x05 },

    // ===== レギュレータ設定値 =====

    // VSDx
    { 0x16, 0x02, 0x10, 0xff }, // 最低電圧（0.6250 V）以上 0.8000 V 以下。
    { 0x17, 0x28, 0xff }, // Reset Value と等しい。1.1000 V
    { 0x18, 0x3a, 0xff }, // 1.3250 V
    { 0x19, 0x60, 0xff }, // Reset Value と等しい。1.8000 V
    //{ 0x1a, 0x20, 0xff }, // Reset Value と等しい。未使用の SD。

    { 0x1b, 0x20, 0xff }, // VDVSSD0, Reset Value と等しい。
    { 0x1c, 0x10, 0xff }, // VDVSSD1, Reset Value と等しい。
    //{ 0x5e, 0x00, 0xff }, // VDVSSD4, Reset Value と等しい。未使用の SD。

    /**
     * CNFG1SDx
     * [7:6]SR_SD      : 0b01 (Slew Rate During DVS: 27.50 mV)
     * [5:4]PWR_MD_SDx : 0b11 (Power Mode Configuration: Normal Operation Mode)
     * [3]  nADE_SDx   : 0b0 (Active Discharge: Enabled)
     * [2]  FPWM_SDx   : 0b0 (Forced PWM Mode: Disabled)
     * [0]  nFSRADE_SDx: 0b1 (Falling Slew Rate Active-Discharge: Enabled)
     */
    { 0x1d, 0x71, 0xff },
    { 0x1e, 0x71, 0xff },
    { 0x1f, 0x71, 0xff },
    { 0x20, 0x71, 0xff },
    //{ 0x21, 0x40, 0xff }, // Reset Value と等しい。未使用の SD。

    /**
     * CNFG2SD
     * [3]  ROVS_EN_SD0: 0b1 (Remove Output Voltage Sense: Enabled)
     * [2]  ROVS_EN_SD1: 0b0 (Remove Output Voltage Sense: Disabled)
     */
    { 0x22, 0x04, 0x06 },

    /**
     * CNFG1_Lx
     * [7:6]PWR_MD_Lx
     * [5:0]TV_Lx
     */
    { 0x23, 0xd0, 0x3f }, // ドックアウト時 0xd0(Normal Mode, 1.200 V), ドックイン時 0x10(Output Disabled, 1.200 V)
    { 0x25, 0xc8, 0xff }, // Normal Mode, 1.000 V
    { 0x27, 0x14, 0x00 }, // SDCard の給電状態と給電電圧はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x29, 0x2e, 0x00 }, // GameCard の給電状態と給電電圧はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2b, 0xc0, 0xff }, // Normal Mode, 0.8000 V
    { 0x2d, 0x14, 0x00 }, // GameCard の給電状態と給電電圧はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2f, 0xea, 0xff }, // Normal Mode, 2.90 V
    { 0x31, 0x04, 0xff }, // Output Disabled, 1.00 V
    { 0x33, 0x04, 0x3f }, // ドックアウト時 0x04(Output Disabled, 1.00 V), ドックイン時 0xc4(Normal Mode, 1.00 V)

    /**
     * CNFG2_Lx
     * [7]  OVCLMP_EN_Lx: 0b1 (Overvoltage Clamp: Enabled)
     * [5:4]COMP_Lx
     * [3]  POK_Lx
     * [1]  ADE_Lx      : 0b1 (Active Discharge: Enabled)
     * [0]  SS_Lx       : 0b1 (Soft-Start Slew Rate Configuration: 100 mV/us)
     *
     * COMP_Lx は LDO0, LDO1, LDO7, LDO8 には存在しない。
     */
    { 0x24, 0xcb, 0x83 }, // ドックアウト時 0xcb, ドックイン時 0xc3
    { 0x26, 0xcb, 0x8b },
    { 0x28, 0xd3, 0xb3 }, // SDCard の給電状態はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2a, 0xd3, 0xb3 }, // GameCard の給電状態はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x2c, 0xdb, 0xbb }, // Reset Value と等しい。
    { 0x2e, 0xd3, 0xb3 }, // GameCard の給電状態はチェックしない。対向の有無によりドライバが操作する為、期待する初期値が維持されていない。
    { 0x30, 0xdb, 0xbb }, // Reset Value と等しい。
    { 0x32, 0xc3, 0x8b },
    { 0x34, 0xc3, 0x83 }, // ドックアウト時 0xc3, ドックイン時 0xcb

    /**
     * CNFG3_LDO
     * [0]  L_B_EN      : 0b0 (LDO bias: Disabled)
     *
     * Reset Value と等しい。
     */
    { 0x35, 0x00, 0x01 },

    // ===== GPIO =====

    /**
     * CNFG_GPIOx
     * [7:6]DBNCx       : 0b00 (When set for GPI, Debounce Configuration: no debounce)
     * [5:4]REFE_IRQx   : 0b00 (When set for GPI, Rising edge and falling edge interrupt configuration: mask interrupt)
     * [3]  DOx
     * [2]  DIx
     * [1]  DIRx (When AMEx=0, 0=general purpose output, 1=general purpose input)
     * [0]  PPDRV (When set for GPO, 0=open-drain, 1=Push-Pull)
     */
    { 0x36, 0x06, 0xff }, // VSYS-AP と結線。VSYS-AP の状態を取得する。
    { 0x37, 0x00, 0xff }, // Reset Value と等しい。1.8V に pull up。MAX77812 の CE に接続。特に役割は無い。
    { 0x38, 0x00, 0xff }, // Reset Value と等しい。1.8V に pull up。特に役割は無い。
    { 0x39, 0x00, 0xff }, // Reset Value と等しい。RP602Z-EN USB Buck-Boost と結線。後述の AME3 により FPS Active-High Output となる。DO は FPS_GPIO3 に依存して変化する。PPDRV3 は参照される。他は無視される。
    { 0x3a, 0x01, 0xff }, // Reset Value と等しい。結線なし。
    { 0x3b, 0x02, 0xff }, // Reset Value と等しい。GND と結線。
    { 0x3c, 0x02, 0xff }, // Reset Value と等しい。GND と結線。
    { 0x3d, 0x02, 0xff }, // Reset Value と等しい。GND と結線。

    { 0x3e, 0x00, 0xff }, // PUE_GPIO, Reset Value と等しい。pull-up enable。すべて Disabled になっている。
    { 0x3f, 0x00, 0xff }, // PDE_GPIO, Reset Value と等しい。pull-down enable。すべて Disabled になっている。

    /**
     * AME_GPIO
     * [7]  AME7        : 0b0
     * [6]  AME6        : 0b0
     * [5]  AME5        : 0b0
     * [4]  AME4        : 0b1
     * [3]  AME3        : 0b1
     * [2]  AME2        : 0b1
     * [1]  AME1        : 0b1
     * [0]  AME0        : 0b0
     *
     * Reset Value と等しい。
     */
    { 0x40, 0x1e, 0xff },

    /**
     * ONOFFCNFG1
     * [7]  SFT_RST     : 0b0
     * [5:3]MRT         : 0b111 (12 s)
     * [2]  SLPEN       : 0b0
     * [1]  PWR_OFF     : 0b0
     * [0]  EN0DLY      : 0b0 (The only delay for EN0 is the debounce circit)
     */
    { 0x41, 0x78, 0xbf },

    /**
     * ONOFFCNFG2
     * [7]  SFT_RST_WK  : 0b0
     * [6]  WDT_RST_WK  : 0b0
     * [5]  nSLP_LPM_MSK: 0b0 (masked)
     * [4]  WK_ACOK     : 0b1
     * [3]  WK_LID      : 0b1
     * [2]  WK_ALARM1R  : 0b0
     * [1]  WK_ALARM2R  : 0b0
     * [0]  WK_EN0      : 0b1
     *
     * 通常時は ALARM から wakeup はしない。ALARM 設定時に 0b1 にする。
     * nSLP_LPM_MSK は Sleep 突入時に 0b1 にする。さもなくば Low-Power Mode に入らないので消費電力が高い。
     */
    { 0x42, 0x19, 0xff }, // reset value

    // ===== フレキシブルパワーシーケンサ =====

    /**
     * CNFGFPSx
     * [5:3]TFPSx       : 0b111 (5120 us)
     * [2:1]SRCFPSx
     * [0]  ENFPSx      : 0b0 (Software Disable)
     */
    { 0x43, 0x38, 0x3f }, // EN0
    { 0x44, 0x3a, 0x3f }, // EN1
    { 0x45, 0x38, 0x3f }, // EN0

    /**
     * FPS_Lx
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x46, 0xc0, 0xc0 }, // FPS3
    { 0x47, 0xc0, 0xc0 }, // FPS3
    { 0x48, 0xc0, 0xc0 }, // FPS3
    { 0x49, 0xc0, 0xc0 }, // FPS3
    { 0x4a, 0x07, 0xff }, // FPS0, LDO4(RTC) は常時有効。
    { 0x4b, 0xc0, 0xc0 }, // FPS3
    { 0x4c, 0xc0, 0xc0 }, // FPS3
    { 0x4d, 0xc0, 0xff }, // FPS3
    { 0x4e, 0xc0, 0xc0 }, // FPS3

    /**
     * FPS_SDx
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x4f, 0x4f, 0xff }, // FPS1
    { 0x50, 0x2a, 0xff }, // FPS0
    { 0x51, 0xc0, 0xff }, // FPS1
    { 0x52, 0x14, 0xff }, // FPS0
    //{ 0x53, 0xc7, 0xff }, // reserved

    /** FPS_GPIOx
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x54, 0x28, 0xc0 }, // Reset Value と等しい。AME1=0 の為使用していない。
    { 0x55, 0x30, 0xff }, // Reset Value と等しい。AME1=1, FPS0 だが GPIO2 は結線されていないので効果はない。
    { 0x56, 0x18, 0xff }, // Reset Value と等しい。AME1=1, FPS0。USB Buck-Boost は常時有効。

    /** FPS_RSO
     * [7:6] FPSSRC
     * [5:3] FPSPU
     * [2:0] FPSPD
     */
    { 0x57, 0x38, 0xff }, // Reset Value と等しい。AME1=1, FPS0。双方向リセット(nRST_IO)は常時有効。

    // ===== CID =====
    { 0x58, 0x00, 0x00 },
    { 0x59, 0x00, 0x00 },
    { 0x5a, 0x00, 0x00 },
    { 0x5b, 0x00, 0x00 },
    { 0x5c, 0x53, 0xff },
    { 0x5d, 0x80, 0xfe }, // (SIGLO-64156) pre-production:0x80, production:0x81 の両方がパスするようにしておきます
};

}} // namespace nnt::bpc
