﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <vector>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/bpc/bpc_ResultPrivate.h>
#include <nn/i2c/i2c.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nnt/gtest/gtest.h>

#include "testBpc_Max77620PmicExpectationList.h"
#include "testBpc_Max77620RtcExpectationList.h"
#include "testBpc_Max77621CpuExpectationList.h"
#include "testBpc_Max77621GpuExpectationList.h"
#include "testBpc_Max77812_2_1_1ExpectationList.h"
#include "testBpc_Max77812_3_1ExpectationList.h"

namespace nnt { namespace bpc {

class PmicRegisterValueTest : public ::testing::Test
{
    virtual void SetUp() NN_OVERRIDE
    {
        nn::i2c::Initialize();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        nn::i2c::Finalize();
    }
};

namespace {

enum class PlatformType
{
    Unknown,
    Icosa,
    Copper,
    Hoag,
    IcosaMariko,
};

PlatformType GetPlatformType() NN_NOEXCEPT
{
    const char* const name = "platformconfig";
    const char* const key = "platformtype";

    size_t size = ::nn::settings::fwdbg::GetSettingsItemValueSize(name, key);
    EXPECT_NE(0, size);

    char buffer[size];
    EXPECT_EQ(size, ::nn::settings::fwdbg::GetSettingsItemValue(buffer, size, name, key));

    if ( std::strcmp("Icosa", buffer) == 0 )
    {
        return PlatformType::Icosa;
    }
    else if ( std::strcmp("Copper", buffer) == 0 )
    {
        return PlatformType::Copper;
    }
    else if ( std::strcmp("Hoag", buffer) == 0 )
    {
        return PlatformType::Hoag;
    }
    else if ( std::strcmp("IcosaMariko", buffer) == 0 )
    {
        return PlatformType::IcosaMariko;
    }

    return PlatformType::Unknown;
}

nn::Result ReadRtcDoubleBufferredRegister(uint8_t* pOutValue, nn::i2c::I2cSession& session, const uint8_t* pAddress) NN_NOEXCEPT
{
    const uint8_t RtcUpdate0Address = 0x04;
    const uint8_t RbudrMask = 0x10;
    const int MaxRetryCount = 4;

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, &RtcUpdate0Address, &RbudrMask));

    int count = 0;
    uint8_t value = 0x00;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, session, &RtcUpdate0Address));
        count++;
    }
    while ( ((value & RbudrMask) == RbudrMask) && count < MaxRetryCount );

    if ( (value & RbudrMask) == RbudrMask )
    {
        return nn::bpc::ResultRtcAccessFailed();
    }

    NN_RESULT_DO(nn::i2c::ReadSingleRegister(pOutValue, session, pAddress));

    NN_RESULT_SUCCESS;
}

void RtcRegisterValueTestInternal(nn::i2c::I2cSession& session, const ExpectedEntry& expected) NN_NOEXCEPT
{
    uint8_t value = 0x00;
    uint8_t address = expected.GetAddress();
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadRtcDoubleBufferredRegister(&value, session, &address));

    EXPECT_TRUE(expected.CheckValue(value));
}

void PmicRegisterValueTestInternal(nn::i2c::I2cSession& session, const ExpectedEntry& expected) NN_NOEXCEPT
{
    uint8_t value = 0x00;
    uint8_t address = expected.GetAddress();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&value, session, &address));

    EXPECT_TRUE(expected.CheckValue(value));
}

} // namespace

void TestMax77620Pmic(const std::vector<ExpectedEntry>& expectedEntries) NN_NOEXCEPT
{
    NN_LOG("%s\n", __FUNCTION__);

    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77620Pmic);

    for ( std::vector<ExpectedEntry>::const_iterator ite = expectedEntries.cbegin(); ite != expectedEntries.cend(); ite++ )
    {
        PmicRegisterValueTestInternal(session, *ite);
    }

    nn::i2c::CloseSession(session);
}

void TestMax77620Rtc(const std::vector<ExpectedEntry>& expectedEntries) NN_NOEXCEPT
{
    NN_LOG("%s\n", __FUNCTION__);

    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77620Rtc);

    for ( std::vector<ExpectedEntry>::const_iterator ite = expectedEntries.cbegin(); ite != expectedEntries.cend(); ite++ )
    {
        auto address = ite->GetAddress();
        if ( address == 0 || address == 4 || address == 5 || address == 6 )
        {
            // 通常のレジスタのチェック
            PmicRegisterValueTestInternal(session, *ite);
        }
        else
        {
            // ダブルバッファドレジスタのチェック
            RtcRegisterValueTestInternal(session, *ite);
        }
    }

    nn::i2c::CloseSession(session);
}

void TestMax77621Cpu(const std::vector<ExpectedEntry>& expectedEntries) NN_NOEXCEPT
{
    NN_LOG("%s\n", __FUNCTION__);

    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77621Cpu);

    for ( std::vector<ExpectedEntry>::const_iterator ite = expectedEntries.cbegin(); ite != expectedEntries.cend(); ite++ )
    {
        PmicRegisterValueTestInternal(session, *ite);
    }

    nn::i2c::CloseSession(session);
}

void TestMax77621Gpu(const std::vector<ExpectedEntry>& expectedEntries) NN_NOEXCEPT
{
    NN_LOG("%s\n", __FUNCTION__);

    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77621Gpu);

    for ( std::vector<ExpectedEntry>::const_iterator ite = expectedEntries.cbegin(); ite != expectedEntries.cend(); ite++ )
    {
        PmicRegisterValueTestInternal(session, *ite);
    }

    nn::i2c::CloseSession(session);
}

bool IsMax77812_3_1() NN_NOEXCEPT
{
    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77812_3_1);

    const uint8_t IntSrcAddress = 0x01; // Readable register and not read clear register)
    uint8_t value;
    auto result = nn::i2c::ReadSingleRegister(&value, session, &IntSrcAddress);

    nn::i2c::CloseSession(session);

    return result.IsSuccess();
}

void TestMax77812_3_1(const std::vector<ExpectedEntry>& expectedEntries) NN_NOEXCEPT
{
    NN_LOG("%s\n", __FUNCTION__);

    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77812_3_1);

    for ( std::vector<ExpectedEntry>::const_iterator ite = expectedEntries.cbegin(); ite != expectedEntries.cend(); ite++ )
    {
        PmicRegisterValueTestInternal(session, *ite);
    }

    nn::i2c::CloseSession(session);
}

void TestMax77812_2_1_1(const std::vector<ExpectedEntry>& expectedEntries) NN_NOEXCEPT
{
    NN_LOG("%s\n", __FUNCTION__);

    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77812_2_1_1);

    for ( std::vector<ExpectedEntry>::const_iterator ite = expectedEntries.cbegin(); ite != expectedEntries.cend(); ite++ )
    {
        PmicRegisterValueTestInternal(session, *ite);
    }

    nn::i2c::CloseSession(session);
}

TEST_F(PmicRegisterValueTest, TestAll)
{
    switch ( GetPlatformType() )
    {
    case PlatformType::Icosa:
        TestMax77620Pmic(Max77620PmicExpectedEntries);
        TestMax77620Rtc(Max77620RtcExpectedEntries);
        TestMax77621Cpu(Max77621CpuExpectedEntries);
        TestMax77621Gpu(Max77621GpuExpectedEntries);
        break;
    case PlatformType::Copper:
        TestMax77620Pmic(Max77620PmicExpectedEntries);
        TestMax77620Rtc(Max77620RtcExpectedEntries);
        TestMax77621Cpu(Max77621CpuExpectedEntries);
        TestMax77621Gpu(Max77621GpuExpectedEntries);
        break;
    case PlatformType::Hoag:
        TestMax77620Pmic(Max77620PmicExpectedEntries);
        TestMax77620Rtc(Max77620RtcExpectedEntries);
        TestMax77621Cpu(Max77621CpuExpectedEntries);
        TestMax77621Gpu(Max77621GpuExpectedEntries);
        break;
    case PlatformType::IcosaMariko:
        TestMax77620Pmic(Max77620PmicExpectedEntriesForMariko);
        TestMax77620Rtc(Max77620RtcExpectedEntries);
        if ( IsMax77812_3_1() )
        {
            TestMax77812_3_1(Max77812_3_1ExpectedEntries);
        }
        else
        {
            TestMax77812_2_1_1(Max77812_2_1_1ExpectedEntries);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

}} // namespace nnt::bpc
