﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_TypesHal.h>

#include "../Common/testBluetooth_TargetDevices.h"
#include "testBluetooth_Bonding.h"

using namespace nn::bluetooth;

nn::os::SystemEventType g_SystemEvent;
nn::os::SystemEventType g_SystemEventForHid;
nn::os::MultiWaitType g_MultiWait;
nn::os::MultiWaitHolderType g_HolderForBluetoothCore;
nn::os::MultiWaitHolderType g_HolderForHid;

namespace BondingTestSpace
{

char* BdaddrToString(const uint8_t bdaddr[])
{
    static char strBdaddr[18];
    sprintf(strBdaddr, "%02X:%02X:%02X:%02X:%02X:%02X",
        bdaddr[0], bdaddr[1], bdaddr[2], bdaddr[3], bdaddr[4], bdaddr[5]);
    return strBdaddr;
}

void PrintEventInfo(nn::bluetooth::EventType eventType, const uint8_t buffer[])
{
    NN_LOG("<- Event Core: ");
    switch (eventType)
    {
    case nn::bluetooth::EventFromDiscoveryStateChangedCallback:
        {
            NN_LOG("DiscoveryStateChanged\n");
            BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
            if (state == BT_DISCOVERY_STOPPED)
            {
                NN_LOG("STOPPED\n");
            }
            else
            {
                NN_LOG("STARTED\n");
            }
        }
        break;
    case nn::bluetooth::EventFromDeviceFoundCallback:
        {
            NN_LOG("DeviceFound\n");
            const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromDeviceFoundCallback*>(&buffer[0]);
            NN_LOG("%s\n", reinterpret_cast<const char*>(&pInfo->BdName.name[0]));
            NN_LOG("%s\n", BdaddrToString(pInfo->BdAddress.address));
        }
        break;
    case nn::bluetooth::EventFromPinRequestCallback:
        {
            NN_LOG("PinRequest\n");
        }
        break;
    case nn::bluetooth::EventFromSspRequestCallback:
        {
            NN_LOG("SspRequest\n");
            const nn::bluetooth::InfoFromSspRequestCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromSspRequestCallback*>(&buffer[0]);
            NN_LOG("%s [%s]\n", reinterpret_cast<const char*>(&pInfo->bluetoothName.name[0]), BdaddrToString(pInfo->bluetoothAddress.address));
        }
        break;
    case nn::bluetooth::EventFromBondStateChangedCallback:
        {
            NN_LOG("BondStateChanged\n");
            const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromBondStateChangedCallback*>(&buffer[0]);
            NN_LOG("%s\n", BdaddrToString(pInfo->bluetoothAddress.address));
            if (pInfo->status == BT_STATUS_SUCCESS)
            {
                if (pInfo->state == BT_BOND_STATE_NONE)
                {
                    NN_LOG("BT_BOND_STATE_NONE\n");
                }
                else if (pInfo->state == BT_BOND_STATE_BONDING)
                {
                    NN_LOG("BT_BOND_STATE_BONDING\n");
                }
                else if (pInfo->state == BT_BOND_STATE_BONDED)
                {
                    NN_LOG("BT_BOND_STATE_BONDED\n");
                }
            }
        }
        break;
    case nn::bluetooth::EventFromAdapterPropertiesCallback:
        {
            NN_LOG("AdapterProperties\n");
        }
        break;
    default:
        {
            NN_LOG("Other. Event type =%d\n", eventType);
        }
    }
}

void PrintEventInfo(nn::bluetooth::HidEventType eventType, const uint8_t buffer[])
{
    if (eventType != nn::bluetooth::EventFromGetReportCallback)
    {
        NN_LOG("<- Event Hid: ");
    }
    switch (eventType)
    {
    case nn::bluetooth::EventFromConnectionStateCallback:
        {
            NN_LOG("ConnectionState\n");
            const nn::bluetooth::InfoFromConnectionStateCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromConnectionStateCallback*>(&buffer[0]);
            if (pInfo->state == BTHH_CONN_STATE_CONNECTED)
            {
                NN_LOG("Connected %s\n", BdaddrToString(pInfo->bluetoothAddress.address));
            }
            else
            {
                NN_LOG("Disconnected %s\n", BdaddrToString(pInfo->bluetoothAddress.address));
            }
        }
        break;
    case nn::bluetooth::EventFromGetReportCallback:
        {
            const nn::bluetooth::InfoFromGetReportCallback* pInfo = reinterpret_cast<const nn::bluetooth::InfoFromGetReportCallback*>(&buffer[0]);
            const nn::bluetooth::HidReport* pReport = &pInfo->rptData;

            //"Basic Input Report" came.
            if (pReport->reportData.data[0] == 0x30)
            {
                //NN_LOG(".");
            }

            //"Reply for the Get Device Info" came.
            else if (pReport->reportData.data[0] == 0x21)
            {
                NN_LOG("[!!!!]");
                //Time stamp for measuring the latency of data transaction.
                //NN_LOG("[MeasuringLatency]ending time   = %d (msec)\n",nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds());
                //PrintByteDataAsHex(32,&pUipcReport->report.report_data.data[0]);
            }
            else
            {
                NN_LOG("[????]");
            }
        }
        break;
    case nn::bluetooth::EventFromExtensionCallbacks:
        {
            NN_LOG("Extension %d\n", reinterpret_cast<const nn::bluetooth::InfoFromExtensionCallbacks*>(&buffer[0])->eventType);
        }
        break;
    default:
        {
            NN_LOG("\n\n other event. EventType= %d", eventType);
        }
    }
}

void WaitDeviceFoundCallback(nn::bluetooth::BluetoothAddress* pFoundAddr, nn::bluetooth::BluetoothName* pFoundName)
{
    for (;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        if (holderType == EVENTTYPE_FROM_CORE)
        {
            nn::os::TryWaitSystemEvent(&g_SystemEvent);
            nn::bluetooth::EventType type;
            uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
            nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
            BondingTestSpace::PrintEventInfo(type, buffer);
            if (type == nn::bluetooth::EventFromDiscoveryStateChangedCallback)
            {
                BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
                if (state == BT_DISCOVERY_STOPPED)
                {
                    nn::bluetooth::StartDiscovery();
                }
            }
            else if (type == nn::bluetooth::EventFromDeviceFoundCallback)
            {
                nn::bluetooth::InfoFromDeviceFoundCallback* info = reinterpret_cast<nn::bluetooth::InfoFromDeviceFoundCallback*>(buffer);
                std::memcpy(pFoundAddr, &info->BdAddress, sizeof(nn::bluetooth::BluetoothAddress));
                std::memcpy(pFoundName, &info->BdName, sizeof(nn::bluetooth::BluetoothName));
                if (nnt::bluetooth::IsTardetDevice(pFoundName, pFoundAddr))
                {
                    // Cancel discovery to allow for create bond once discovery stops
                    nn::bluetooth::CancelDiscovery();
                    break;
                }
            }
        }
    }
}


}
