﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/oe/oe_HdcpApis.private.h>
#include <nn/fs.h>

#include "../Integration/Common/Common.h"
#include "../Integration/Common/Graphics.h"
#include "../Integration/Common/AudioRenderer.h"
#include "../Integration/Common/ConvNpad.h"

namespace {

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// HDCP 認証状態監視スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_HdcpStateThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// デバッグコントローラスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

// Audio スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

nn::os::ThreadType  g_HdcpStateThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;
nn::os::ThreadType  g_AudioThread;

enum AudioState
{
    AudioState_Disabled,
    AudioState_Enabled,
    AudioState_Stress,
};

AudioState g_AudioState = AudioState_Disabled;

// Audio が有効かどうか
bool g_IsAudioEnabled = false;

// スレッド制御
ThreadControl g_ApplicationThreadControl;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize()
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // OE ライブラリの初期化
    nn::oe::Initialize();

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.3f, 0.1f, 0.1f, 1.0f };
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredSubmission;
    Graphics::InitializeGraphics( clearColor, "testOe_Hdcp", frameworkMode, false );

    // Audio の初期化
    nnt::applet::audiorenderer::InitializeAudio();
    nnt::applet::audiorenderer::LoadBgm( nnt::applet::audiorenderer::Bgm_B );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();
}

void Finalize()
{
    // Audio 関連の終了処理
    nnt::applet::audiorenderer::FinalizeAudio();

    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

} // namespace

//-----------------------------------------------------------------------------
//  Pad 入力ハンドリングスレッド
//
void DebugPadThreadFunction(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState debugPadState;
    nn::hid::DebugPadButtonSet pressed;
    nn::hid::DebugPadButtonSet trigger;

    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::WaitVsync();
        nnt::applet::hid::GetConvNpadState( &debugPadState );

        // ボタン情報の更新
        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // DebugPad の Select が押された
        if ( (trigger & nn::hid::DebugPadButton::Select::Mask).IsAnyOn() )
        {
            if( g_AudioState == AudioState_Stress )
            {
                NN_LOG( "testOe_Hdcp: Disable Audio\n" );
                g_IsAudioEnabled = false;
                g_AudioState = AudioState_Disabled;
                Graphics::SetAudioStatus( g_IsAudioEnabled );
                nnt::applet::audiorenderer::PauseBgm();
                nnt::applet::audiorenderer::PauseSine();
            }
            else if( g_AudioState == AudioState_Disabled )
            {
                NN_LOG( "testOe_Hdcp: Enable Audio\n" );
                g_IsAudioEnabled = true;
                g_AudioState = AudioState_Enabled;
                Graphics::SetAudioStatus( g_IsAudioEnabled );
                nnt::applet::audiorenderer::PlayBgm();
            }
            else
            {
                NN_LOG( "testOe_Hdcp: Add Audio\n" );
                g_IsAudioEnabled = true;
                g_AudioState = AudioState_Stress;
                Graphics::SetAudioStatus( g_IsAudioEnabled );
                nnt::applet::audiorenderer::PlaySine();
            }
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::GraphicsRenderer();
    }
}

//-----------------------------------------------------------------------------
//  オーディオ再生スレッド
//
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        nnt::applet::audiorenderer::WaitAudio();
        nnt::applet::audiorenderer::UpdateAudio();
        Graphics::WaitVsync();
    }
}

//-----------------------------------------------------------------------------
//  HDCP 認証状態監視スレッド
//
void HdcpStateThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // SIGLO-69853: ハンドルリークしないかの確認
    for (int i=0; i<1000; ++i)
    {
        nn::os::SystemEvent event;
        nn::oe::GetHdcpAuthenticationStateChangeEvent(&event);
    }

    nn::os::SystemEvent hdcpAuthenticationStateChangeEvent;
    nn::oe::GetHdcpAuthenticationStateChangeEvent(&hdcpAuthenticationStateChangeEvent);

    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        hdcpAuthenticationStateChangeEvent.Wait();
        NN_LOG("HDCP authentication state changed. HDCP authentication state is %d\n",
            nn::oe::GetHdcpAuthenticationState());
    }
}

extern "C" void nnMain()
{
    NN_LOG( "testOe_Hdcp: start.\n" );

    // すでに InFocus 状態である
    g_ApplicationThreadControl.SetInFocusState(true);

    // 初期化処理
    Initialize();

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, AppletPriorityForAudio ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_HdcpStateThread, HdcpStateThreadFunction, NULL, g_HdcpStateThreadStack, ThreadStackSize, AppletPriorityForLimit10msec) );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );
    nn::os::StartThread( &g_AudioThread );
    nn::os::StartThread( &g_HdcpStateThread );

    // スレッドの終了を待機
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );
    nn::os::WaitThread( &g_AudioThread );
    nn::os::WaitThread( &g_HdcpStateThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );
    nn::os::DestroyThread( &g_AudioThread );
    nn::os::DestroyThread( &g_HdcpStateThread );

    // 終了処理
    Finalize();

    NN_LOG( "testOe_Hdcp: end.\n" );
}
