﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * HW評価アプリ：照度センサーデータロガー
 * 参考：http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=112900334
 */

#include <cstdlib> // new, free
#include <vector>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/font.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h> // SD への FAT アクセス
#include <nn/hid.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>

#include <eval_Screen.h>
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#include <nnd/bh1730fvc/bh1730fvc.h>

#define USE_TIME

namespace {

const int AppVerMajor = 1;
const int AppVerMinor = 1;


struct ConfigVal{
    ::nn::Bit8              id;
    ::nnd::bh1730fvc::Gain  gain;
    int                     cycle;
};
ConfigVal g_ConfigVlaue;

::nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> g_TouchState;

static int g_FileCount = 0;
static bool g_IsLogfileWriting = false;
static bool g_Finalize = false;
static bool g_Push = false;
static bool g_SwitchMode = false;
static bool g_IsFirstTouch = true;
static ::nn::os::Tick g_BeginTouchTick;
static ::nn::os::Tick g_BeginMeasureTick;

static int64_t g_WriteOffset = 0;

// マウントするファイルシステムに対する名前
::nn::fs::FileHandle g_FileHandle;
const ::std::string SdcardMountName = "sd";


struct LogDate {
    ::nnd::bh1730fvc::MeasurementValue  readVal;
    ::std::string                       dateStr;
    ::std::string                       timeStr;
};

// 画面解像度
//const float DisplayWidth = 1280;
const float DisplayHeight = 720;

// Defaultとするフォントサイズ
const float FontSizeDefault = 33.0f;
const float FontSizeLarge = FontSizeDefault * 1.5f;

struct Position {
    float x;
    float y;
};

const float MsPerCycle = 2.7f;  // 1 cycle の ミリ秒
int         WaitTimeMs = 0;     // 計測時間に併せて待つための時間[ミリ秒]

// 枠線 --------------------------------
// 縦線の X 座標
const float     PosFrameVerticalX           =  950.0f;

// タイトルの下枠
const Position  PosFrameHorizontalTitle     = { 0.0f, FontSizeDefault};

// 縦線から右に線を引く時のマージンとX座標
const float     PosFrameVerticalMargine     = FontSizeDefault * 0.36f; // 実測による比率
const float     PosFrameHorizontalRightX    = PosFrameVerticalX + PosFrameVerticalMargine;

// 縦線から右に引く枠線
const Position  PosFrameHorizontalStartStop = { PosFrameHorizontalRightX, 590.0f};
//const Position  PosFrameHorizontalConfig    = { PosFrameHorizontalRightX, 200.0f};

// テキスト ----------------------------
// 外周からの余白
const float MarginTextFromFrame = 5.0f;

const float PosTextHorizontalRightX = PosFrameHorizontalRightX + 10.0f;

// タイトル
const Position PosTextTitle = {
        MarginTextFromFrame,
        MarginTextFromFrame };

// 日時
const Position PosTextDate = {
        PosTextHorizontalRightX,
        MarginTextFromFrame };

// START/STOP
const Position PosTextStartStop = {
        PosTextHorizontalRightX,
        PosFrameHorizontalStartStop.y + (DisplayHeight - PosFrameHorizontalStartStop.y - FontSizeLarge) / 2 };

// コンフィグ値
const Position PosTextConfig = {
        PosTextHorizontalRightX,
        PosFrameHorizontalTitle.y + FontSizeDefault / 2};

// ログデータの項目名
const float PosTextLogTitleY = PosFrameHorizontalTitle.y + FontSizeDefault * 3 / 2;
const float PosTextLogDataY = PosFrameHorizontalTitle.y + FontSizeDefault * 5 / 2;

const float PosTextDateX = MarginTextFromFrame; // 日付
const float PosTextTimeX = PosTextDateX + 180;
const float PosTextLuxX = PosTextTimeX + 150;
const float PosTextData0X = PosTextLuxX + 210;
const float PosTextData1X = PosTextData0X + 200;


// フラグ -----------------------
bool    g_IsSdMounted = false;
}

void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* ptr, size_t)
{
    free(ptr);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

::std::string GetCurrentTimeStrForFileName(::nn::time::PosixTime posixTime)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
//    NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
//        calendarAdditionalInfo.timeZone.standardTimeName,
//        calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    ::std::string out = ::std::to_string(calendarTime.year) + ::std::to_string(calendarTime.month) + ::std::to_string(calendarTime.day)
        + "_" + ::std::to_string(calendarTime.hour) + ::std::to_string(calendarTime.minute) + ::std::to_string(calendarTime.second);

    return out;
}

::std::string GetCurrentDateStr(::nn::time::PosixTime posixTime)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
//    NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
//        calendarAdditionalInfo.timeZone.standardTimeName,
//        calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    ::std::string out = ::std::to_string(calendarTime.year) + "/" + ::std::to_string(calendarTime.month) + "/" + ::std::to_string(calendarTime.day);

    return out;
}

::std::string GetCurrentTimeStr(::nn::time::PosixTime posixTime)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
//    NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
//        calendarAdditionalInfo.timeZone.standardTimeName,
//        calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    ::std::string out = ::std::to_string(calendarTime.hour) + ":" + ::std::to_string(calendarTime.minute) + ":" + ::std::to_string(calendarTime.second);

    return out;
}


void DrawHorizontalLine(nn::gfx::util::DebugFontTextWriter* pTextWriter, Position pos, int size, uint32_t flag)
{
    float width = pTextWriter->CalculateStringWidth("─");

//    pTextWriter->SetDrawFlag(flag);

    pTextWriter->SetCursor(pos.x, pos.y);
    for(auto i = 0; i < size ; i++)
    {
        pTextWriter->Print("─");

        if (flag & ::nn::font::TextWriter::PositionFlag_HorizontalOriginRight)
        {
            pos.x -= width;
            pTextWriter->SetCursor(pos.x, pos.y);
        }
    }
}

void DrawVerticalLine(nn::gfx::util::DebugFontTextWriter* pTextWriter, Position pos, float size)
{
    float height = pTextWriter->GetLineHeight() * 0.76f; // 実測による比率
    for (auto i = 0; i < size; i ++)
    {
        pTextWriter->SetCursor(pos.x, pos.y);
        pTextWriter->Print("│");
        pos.y += height;
    }
}


void DrawOutline(nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    // 縦線 ----------------------
    Position pos;
    uint32_t flag = ::nn::font::TextWriter::DefaultDrawFlag;
    pTextWriter->SetFontSize(FontSizeDefault);
    pTextWriter->SetTextColor(::nnt::eval::White);

    pos = { PosFrameVerticalX, 0.0f};
    DrawVerticalLine(pTextWriter, pos, 30);

    // 横線 ----------------------
    // タイトル&時計の枠
    pos = PosFrameHorizontalTitle;
    DrawHorizontalLine(pTextWriter, pos, 50, flag);


    pTextWriter->SetTextColor(::nnt::eval::Gray);
    // 表の項目名の枠
    pos.x = PosFrameVerticalX + PosFrameVerticalMargine;
    pos.y += pTextWriter->GetFontHeight() * 2;
    flag = ::nn::font::TextWriter::PositionFlag_HorizontalOriginRight;
    DrawHorizontalLine(pTextWriter, pos, 40, flag);
    flag = ::nn::font::TextWriter::DefaultDrawFlag;

//    // デバイス動作設定値の枠
//    pos = PosFrameHorizontalConfig;
//    DrawHorizontalLine(textWriter, pos, 15, flag);

    // 制御ボタンの枠
    pTextWriter->SetTextColor(::nnt::eval::White);
    pos = PosFrameHorizontalStartStop;
    DrawHorizontalLine(pTextWriter, pos, 15, flag);
}


void DisplayTextOnLcd(::nnt::eval::Screen* pScreen, ::std::vector<LogDate>& inData)
{
//    NN_LOG("--- Display text on LCD.\n");

    // LCD へのテキスト表示
    nn::gfx::util::DebugFontTextWriter* pTextWriter = pScreen->GetDebugFontTextWriter();

    // 枠線
    DrawOutline(pTextWriter);

    Position pos;

    //----------- テキスト
    pTextWriter->SetTabWidth(2);
    pTextWriter->SetTextColor(::nnt::eval::White);

    // タイトル
    pos = PosTextTitle;
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->Print("[HW Eval: ALS DataLogger (Ver.%d.%d)] ",AppVerMajor,AppVerMinor);

    // SDのマウント状態
    if(!g_IsSdMounted)
    {
        pTextWriter->SetTextColor(::nnt::eval::Red);
        pTextWriter->Print("SD card is NOT mounted");
        pTextWriter->SetTextColor(::nnt::eval::White);
    }
    else
    {
        pTextWriter->SetTextColor(::nnt::eval::Green);
        pTextWriter->Print("SD card is mounted");
        pTextWriter->SetTextColor(::nnt::eval::White);
    }

    // 時計
    pos = PosTextDate;
    pTextWriter->SetCursor(pos.x, pos.y);
    ::nn::time::PosixTime posixTime;
    ::nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    ::std::string date = GetCurrentDateStr(posixTime);
    ::std::string time = GetCurrentTimeStr(posixTime);
    pTextWriter->Print("%s %s", date.c_str(), time.c_str());

    // デバイス設定値
    pos = PosTextConfig;
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->Print("ID\nCycle\nGain\n");
    pos = PosTextConfig;
    pos.x += pTextWriter->CalculateStringWidth("Cycle ");;
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->Print(": 0x%X\n: %d(%4.1fms)\n: %s\n", g_ConfigVlaue.id, g_ConfigVlaue.cycle, g_ConfigVlaue.cycle * MsPerCycle,
                (g_ConfigVlaue.gain == ::nnd::bh1730fvc::Gain::X1    ? "x1" :
                (g_ConfigVlaue.gain == ::nnd::bh1730fvc::Gain::X2    ? "x2":
                (g_ConfigVlaue.gain == ::nnd::bh1730fvc::Gain::X64   ? "x64":
                (g_ConfigVlaue.gain == ::nnd::bh1730fvc::Gain::X128  ? "x128": "Unknown")))));

    // 列名称
    pTextWriter->SetTextColor(::nnt::eval::Gray);
    pTextWriter->SetCursor(PosTextDateX, PosTextLogTitleY);
    pTextWriter->Print("%s","Date");
    pTextWriter->SetCursorX(PosTextTimeX);
    pTextWriter->Print("%s","Time");
    pTextWriter->SetCursorX(PosTextLuxX);
    pTextWriter->Print("%s","Lux");
    pTextWriter->SetCursorX(PosTextData0X);
    pTextWriter->Print("%s","DATA0");
    pTextWriter->SetCursorX(PosTextData1X);
    pTextWriter->Print("%s","DATA1");
    pTextWriter->SetTextColor(::nnt::eval::White);

    // ログデータ
    // TODO: vector で過去のログも出力したい
    pTextWriter->SetCursor(PosTextDateX, PosTextLogDataY);
    pTextWriter->Print("%s",inData[0].dateStr.c_str());
    pTextWriter->SetCursorX(PosTextTimeX);
    pTextWriter->Print("%s",inData[0].timeStr.c_str());
    pTextWriter->SetCursorX(PosTextLuxX);
    pTextWriter->Print("%-10.4f",inData[0].readVal.lux);
    pTextWriter->SetCursorX(PosTextData0X);
    pTextWriter->Print("0x%04X",inData[0].readVal.visible);
//    pTextWriter->Print("0x%04X(%_5d)",inData[0].readVal.visible, inData[0].readVal.visible);
    pTextWriter->SetCursorX(PosTextData1X);
    pTextWriter->Print("0x%04X",inData[0].readVal.infrared);
//    pTextWriter->Print("0x%04X(%_5d)",inData[0].readVal.infrared, inData[0].readVal.infrared);

    // 操作ボタン
    pos = PosTextStartStop;
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->SetFontSize(FontSizeLarge);
    if (g_IsLogfileWriting)
    {
        pTextWriter->SetTextColor(::nnt::eval::Red);
        pTextWriter->Print("Tap to STOP\n");
//        NN_LOG("Tap to STOP\n");
    }
    else
    {
        pTextWriter->SetTextColor(::nnt::eval::Blue);
        pTextWriter->Print("Tap to START\n");
//        NN_LOG("Tap to START\n");
    }
    pTextWriter->SetFontSize(FontSizeDefault);
    pTextWriter->SetTextColor(::nnt::eval::White);

    // 描画
    pScreen->Draw();
}

void DetectTouchState()
{
    //-----------------------------------
    // タッチ状態の確認
    //-----------------------------------
    // 計測状態の切り替え要求を確認
    nn::hid::GetTouchScreenState(&g_TouchState);
    // 始めてタッチを検出した時に計測開始
    if (g_IsFirstTouch && ((g_TouchState.count == 1) || (g_TouchState.count == 4)))
    {
        g_BeginTouchTick = ::nn::os::GetSystemTick();
        g_IsFirstTouch = false;
    }
    if (g_TouchState.count == 0)
    {
        g_IsFirstTouch = true;
        g_Push = false;
    }
    else
    {
        g_IsFirstTouch = false;
        auto endTick = ::nn::os::GetSystemTick();
        auto totalTick = endTick - g_BeginTouchTick;
        auto durationMsec = ::nn::os::ConvertToTimeSpan(totalTick).GetMilliSeconds();
        if (g_TouchState.count == 1)
        {
            // タッチが行われており「START/STOP」の位置なら
            if (!g_Push && g_TouchState.touches[0].x > PosFrameHorizontalStartStop.x
                    && g_TouchState.touches[0].y > PosFrameHorizontalStartStop.y)
            {
                if (durationMsec > 100)
                {
                    NN_LOG("Touch!! START/STOP button.\n");
                    g_Push = true;
                    g_SwitchMode = true;
                }
            }
        }
        else if (g_TouchState.count == 4 && !g_IsLogfileWriting)
        {
            // 4本の指で5秒タッチされたらアプリ終了
            if (durationMsec > 5000)
            {
                NN_LOG("Touch!! Finalize Application.\n");
                g_Finalize = true;
            }
        }
    }
}

void OpenOutputFile()
{
    // 時間からファイル名を生成        // 時間を取得
    ::nn::time::PosixTime posixTime;
    ::nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    ::std::string timeStr = GetCurrentTimeStrForFileName(posixTime);
    const ::std::string OutFileName = "AlsDataLogger-" + timeStr + "-" + ::std::to_string(g_FileCount) +".txt";

    // 出力するファイルのパス
    const ::std::string filePath = SdcardMountName + ":/" + OutFileName;
    NN_LOG("FilePath: %s\n", filePath.c_str());

    ::nn::fs::DirectoryEntryType type;
    auto result = ::nn::fs::GetEntryType(&type, filePath.c_str());
    if (::nn::fs::ResultPathNotFound().Includes(result))
    {
        result = ::nn::fs::CreateFile(filePath.c_str(), 0);
        NN_ASSERT(result.IsSuccess(), "Failed to create file");
    }
    else
    {
        NN_ASSERT(result.IsSuccess(), "Failed to get entry type");
    }

    // ファイルをオープン
    result = ::nn::fs::OpenFile(&g_FileHandle, filePath.c_str(), ::nn::fs::OpenMode_Write | ::nn::fs::OpenMode_AllowAppend);
    NN_ASSERT(result.IsSuccess(), "Failed to open log");

}

void CloseOutputFile()
{
    // ファイルをクローズ
    ::nn::fs::CloseFile(g_FileHandle);
    g_WriteOffset = 0;
    g_FileCount++;
}


extern "C" void nnMain()
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    NN_LOG("/// START ///\n");

    // フォント表示ライブラリの初期化
    ::nnt::eval::Screen* pScreen = new ::nnt::eval::Screen();
    pScreen->Initialize();

    // ライブラリの初期化
    ::nnd::bh1730fvc::Initialize();

    // 操作制御用 hid の初期化
    ::nn::hid::InitializeTouchScreen();

    ::nn::time::Initialize();

    // SD カードをマウント
    auto result = ::nn::fs::MountSdCardForDebug(SdcardMountName.c_str());
    if (!result.IsSuccess())
    {
        NN_LOG("!!!!! Failed to mount SD card. !!!!!!\n");
        g_IsSdMounted = false;
    }
    else
    {
        g_IsSdMounted = true;
    }

    // Gain をシステムの設定と同じモノにする
//    ::nnd::bh1730fvc::SetGain(::nnd::bh1730fvc::Gain::X1);
//    ::nnd::bh1730fvc::SetGain(::nnd::bh1730fvc::Gain::X2);
    ::nnd::bh1730fvc::SetGain(::nnd::bh1730fvc::Gain::X64);
//    ::nnd::bh1730fvc::SetGain(::nnd::bh1730fvc::Gain::X128);

    // 各種 計測設定値の取得
    ::nnd::bh1730fvc::ReadRevisionId(&g_ConfigVlaue.id);
    ::nnd::bh1730fvc::GetIntegralCycle(&g_ConfigVlaue.cycle);
    ::nnd::bh1730fvc::GetGain(&g_ConfigVlaue.gain);
    WaitTimeMs = ceil(g_ConfigVlaue.cycle * MsPerCycle);

    ::nnd::bh1730fvc::SetMeasurementMode(::nnd::bh1730fvc::MeasurementMode::Continuous);


    // ループ
    ::std::vector<LogDate> dataBuffer;
    dataBuffer.reserve(30);

    // 時計表示部分のバッファの初期化
    dataBuffer[0].dateStr = "0000/00/00";
    dataBuffer[0].timeStr = "00:00:00";

    while (!g_Finalize)
    {
        //-----------------------------------
        // タッチ状態の確認
        //-----------------------------------
        // 計測状態の切り替え要求を確認
        DetectTouchState();

        //-------------------------------------
        // ログファイルの出力状態を切り替え
        //-------------------------------------
        if (g_Push && g_SwitchMode)
        {
            // フラグをクリア
            g_SwitchMode = false;

            // モードを反転
            g_IsLogfileWriting = !g_IsLogfileWriting;
            if (g_IsLogfileWriting)
            {
                g_BeginMeasureTick = ::nn::os::GetSystemTick();
            }

            // SDカードがマウントされている場合のみログファイルをオープン
            if (g_IsLogfileWriting && g_IsSdMounted)
            {
                OpenOutputFile();
            }
            else
            {
                CloseOutputFile();
            }
        }

        // --------------------------------
        // センサー値の取得とログ出力
        // --------------------------------
        if (g_IsLogfileWriting)
        {
            auto endTick = ::nn::os::GetSystemTick();
            auto totalTick = endTick - g_BeginMeasureTick;
            auto durationMsec = ::nn::os::ConvertToTimeSpan(totalTick).GetMilliSeconds();

            // posixTime を WaitTimeMs で割って
            if(durationMsec > WaitTimeMs)
            {
                g_BeginMeasureTick = ::nn::os::GetSystemTick();

                // センサー値の取得
                // TODO: vector なので複数バッファを利用したい。
                ::nnd::bh1730fvc::MeasurementValue readData;
                ::nnd::bh1730fvc::GetMeasurementValue(&readData);
                dataBuffer[0].readVal = readData;

                // 時間取得
                // 協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数で現在時刻を取得します。
                nn::time::PosixTime posixTime;
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
                dataBuffer[0].dateStr = GetCurrentDateStr(posixTime);
                dataBuffer[0].timeStr = GetCurrentTimeStr(posixTime);

                // ログファイルへの書き出し
                ::std::string OutStrData;
                OutStrData = dataBuffer[0].dateStr.c_str();
                OutStrData += "\t";
                OutStrData += dataBuffer[0].timeStr.c_str();
                OutStrData += "\t";
                OutStrData += ::std::to_string(dataBuffer[0].readVal.lux);
                OutStrData += "\t";
                OutStrData += ::std::to_string(dataBuffer[0].readVal.visible);
                OutStrData += "\t";
                OutStrData += ::std::to_string(dataBuffer[0].readVal.infrared);
                OutStrData += "\n";
                result = ::nn::fs::WriteFile(g_FileHandle, g_WriteOffset, OutStrData.c_str(),
                        OutStrData.size(), ::nn::fs::WriteOption::MakeValue(0));
                g_WriteOffset += OutStrData.size();
                NN_ASSERT(result.IsSuccess(), "Failed to write log");

                // コンソールにも出力
                NN_LOG("%s", OutStrData.c_str());

                // ファイルへの書き出し
                result = ::nn::fs::FlushFile(g_FileHandle);
                NN_ASSERT(result.IsSuccess(), "Failed to flush log");
            }
        }
        else
        {
            // 計測も停止しないが、ログファイルへの出力は行わない。LCDの更新も止める。
        }

        DisplayTextOnLcd(pScreen, dataBuffer);

    }

    // 終了処理 -----------------------------------------

    ::nn::time::Finalize();

    // ファイルシステムのマウントを解除
    ::nn::fs::Unmount(SdcardMountName.c_str());

    // FIXME: lbl プロセスで照度値が取得できなくなる
    ::nnd::bh1730fvc::Finalize();

    pScreen->Finalize();
    delete pScreen;

    NN_LOG("/// END ///\n");

#ifndef CI_EXECUTION
    // CI の実行テストで無視したい処理はで通らない
    NN_LOG("waiting for something...\n");
#endif

#ifdef CI_EXECUTION
    // CI の実行テストで通る
    NN_LOG("CI Pass\n");
#endif
} // NOLINT(impl/function_size)

