﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/album.h>
#include <nn/album/album_AlbumFileAccess.private.h>
#include <nn/album/album_AlbumFileAccessForDebug.private.h>

#include <cstring>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_TimeSpan.h>

#include <nnt.h>

namespace {

NN_ALIGNAS(4096) uint8_t g_ScreenShotImage[ nn::album::AlbumScreenShotImageDataSize ];

// 自 ApplicationId のスクリーンショット静止画ファイルを削除する
void DeleteAllScreenShotFiles()
{
    // 現在のファイルリストを取得して全削除
    int fileCount1 = 0;
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(20000);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetAlbumScreenshotFileList(&fileCount1, entryList.data(), entryList.size()));
    NN_LOG("DeleteAllScreenShotFiles(): Begin: file count = %d\n", fileCount1);

    auto startTick = nn::os::GetSystemTick();

    for (int i=0; i<fileCount1; ++i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteAlbumFileForDebug(entryList[i]));
    }

    auto endTick = nn::os::GetSystemTick();

    // 現在のファイルリストを再取得して残ファイルがないことを確認
    int fileCount2 = -1;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetAlbumScreenshotFileList(&fileCount2, entryList.data(), entryList.size()));
    ASSERT_EQ(0, fileCount2);
    entryList.clear();
    NN_LOG("DeleteAllScreenShotFiles(): End:   file count = %d\n", fileCount2);

    auto timeSpan = (endTick - startTick).ToTimeSpan();
    NN_LOG("Elapsed time to delete all files = %lld (usec)\n", timeSpan.GetMicroSeconds());
}

void PrepareScreenShotImage(void *pBuffer, int width, int height)
{
    auto p = reinterpret_cast<uint32_t*>(pBuffer);
    for (int y=0; y<height; ++y)
    {
        for (int x=0; x<width; ++x)
        {
            uint32_t b = 255 * y / height;
            uint32_t g = 255 * x / width;
            p[y * width + x] = 0xff000000 | (b << 16) | (g << 8);
        }
    }
}

}   // namespace

//-----------------------------------------------------------------------------
// スクリーンショットファイルの削除時間の計測
//
TEST(DeleteAlbumFile, MeasureElapsedTimeOfDeleteAlbumFile)
{
    // album の初期化
    nn::album::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::album::Finalize(); };

    PrepareScreenShotImage(g_ScreenShotImage, 1280, 720);

    // 最初にゴミファイルを削除
    DeleteAllScreenShotFiles();

    // ファイル数を指定
    const int fileNum = 1000;

    // 静止画ファイルを作成する
    NN_LOG("Now creating screenshot files.\n");
    for (int i=0; i<fileNum; ++i)
    {
        NN_LOG(".");
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );
    }
    NN_LOG("\n");

    // ファイルの削除時間を計測
    NN_LOG("Now deleting all screenshot files.\n");
    DeleteAllScreenShotFiles();
}

