﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

#include <nn/ahid/hdr/hdr.h>
#include <nn/ahid/ahid.h>
#include "../../../../../Programs/Eris/Sources/Libraries/ahid/hdr/libnn_hdrServer/ahid_Hdr.h"
#include "../../../../../Programs/Eris/Sources/Libraries/cdhid/libnn_cdhidWin32Hid/cdhidWin32Hid_Main.h"

// Set TEST_VENDOR_ID and TEST_PRODUCT_ID to IDs from your contoller
enum
{
    TEST_VENDOR_ID  = 0x000012bd,
    TEST_PRODUCT_ID = 0x0000d015,
};


//////////////////////////////////////////////////////////////////////////////
static void printString(uint8_t *p)
{
    int len = *p++;

    if (*p++ != nn::ahid::hdr::AhidDescriptorTypeString)
    {
        NN_SDK_LOG("invalid string\n");
        return;
    }

    len -= 2;

    while (len)
    {
        NN_SDK_LOG("%c", *p++);
        p++;
        len -= 2;
    }

    NN_SDK_LOG("\n");
}



//////////////////////////////////////////////////////////////////////////////
static void testInitialize(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters* pDeviceParameters)
{
    nn::Result result;

    NN_SDK_LOG("calling Initialize\n");
    result = pAhid->Initialize(pDeviceParameters);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetString(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling GetString\n");
    result = pAhid->GetString(data, 4096, 2);
    NN_SDK_LOG("result %d\n", result);

    if (result.IsSuccess())
    {
        printString(data);
    }
}


//////////////////////////////////////////////////////////////////////////////
static void testSetIdle(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling SetIdle\n");
    result = pAhid->SetIdle(0, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetIdle(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t idle;

    NN_SDK_LOG("calling GetIdle\n");
    result = pAhid->GetIdle(&idle, 0);
    NN_SDK_LOG("result %d idle %d\n", result, idle);
}


//////////////////////////////////////////////////////////////////////////////
static void testSetProtocol(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling SetProtocol\n");
    result = pAhid->SetProtocol(0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetProtocol(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t protocol;

    NN_SDK_LOG("calling GetProtocol\n");
    result = pAhid->GetProtocol(&protocol);
    NN_SDK_LOG("result %d protocol %d\n", result, protocol);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetReport(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[9];

    NN_SDK_LOG("calling GetReport\n");
    result = pAhid->GetReport(data, 9, 0x01, 0);
    NN_SDK_LOG("result %d\n", result);

    if (result.IsSuccess())
    {
        NN_SDK_LOG("Report:\n");

        for (int i = 0; i < 9; i++)
        {
            NN_SDK_LOG("%02x ", data[i]);
        }

        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void testSetReport(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling SetReport\n");
    result = pAhid->SetReport(data, 64, 0x02, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testRead(nn::ahid::Ahid *pAhid)
{
    bool        run = true;
    uint8_t     data[4096];
    uint32_t    bytesRead;
    nn::Result  result;

    NN_SDK_LOG("press some buttons on the game controller\n");

    while (run)
    {
        result = pAhid->Read(&bytesRead, data, 4096);

        if (result.IsSuccess())
        {
            if (bytesRead)
            {
                NN_SDK_LOG("read data: ");

                for (uint32_t i = 0; i < bytesRead; i++)
                {
                    NN_SDK_LOG("%02x ", data[i]);
                }

                NN_SDK_LOG("\n");
            }
        }
        else
        {
            NN_SDK_LOG("result %d\n", result);
            run = false;
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
static void testWrite(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[4];
    uint32_t bytesWritten;

    NN_SDK_LOG("calling Write\n");
    result = pAhid->Write(&bytesWritten, data, 4);
    NN_SDK_LOG("result %d bytesWritten %d\n", result, bytesWritten);
}


//////////////////////////////////////////////////////////////////////////////
static void testFinalize(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling Finalize\n");
    result = pAhid->Finalize();
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void clientFunction()
{
    bool run = true;

    nn::ahid::hdr::Hdr hdr;
    nn::ahid::Ahid     ahid; // use one client per device

    NN_SDK_LOG("->%s\n", __FUNCTION__);

    hdr.Initialize();

    nn::ahid::hdr::AttachFilter attachFilter;

    memset(&attachFilter, 0, sizeof(nn::ahid::hdr::AttachFilter));

    NN_SDK_LOG("- please attach device with vendorID %08x and productID %08x\n", TEST_VENDOR_ID, TEST_PRODUCT_ID);
    NN_SDK_LOG("- disconnect device to end program\n");

    // Client queries HDR for attached devices
    while (run)
    {
        size_t outEntries, inEntries;

        nn::Result result = hdr.GetDeviceEntries(&outEntries);

        if (result.IsSuccess() && outEntries)
        {
            nn::ahid::hdr::DeviceHandle deviceHandle[nn::ahid::hdr::AhidDevicesCountMax];

            inEntries = nn::ahid::hdr::AhidDevicesCountMax;

            result = hdr.GetDeviceList(&outEntries, inEntries, deviceHandle, &attachFilter);

            attachFilter.attachedAfterTicks = nn::os::GetSystemTick().GetInt64Value();

            if (result.IsSuccess() && outEntries)
            {
                for (size_t j = 0; j < outEntries; j++)
                {
                    nn::ahid::hdr::DeviceParameters deviceParameters;

                    result = hdr.GetDeviceParameters(&deviceParameters, deviceHandle[j]);

                    if (result.IsSuccess())
                    {
                        if ((deviceParameters.vendorId == TEST_VENDOR_ID) && (deviceParameters.productId == TEST_PRODUCT_ID))
                        {
                            NN_SDK_LOG("HDR DeviceHandle:   %08x\n",    deviceHandle[j]);
                            NN_SDK_LOG("    deviceHandle:   %08x\n",    deviceParameters.deviceHandle);
                            NN_SDK_LOG("    servicePath:    %s\n",      deviceParameters.servicePath);
                            NN_SDK_LOG("    usagePage:      %04x\n",    deviceParameters.usagePage);
                            NN_SDK_LOG("    usageId:        %04x\n",    deviceParameters.usageId);
                            NN_SDK_LOG("    busId:          %d\n",      deviceParameters.busId);
                            NN_SDK_LOG("    vendorId:       %08x\n",    deviceParameters.vendorId);
                            NN_SDK_LOG("    productId:      %08x\n",    deviceParameters.productId);
                            NN_SDK_LOG("    versionNumber:  %08x\n",    deviceParameters.versionNumber);
                            NN_SDK_LOG("    manufacturer:   ");
                            printString(deviceParameters.manufacturer);
                            NN_SDK_LOG("    product:        ");
                            printString(deviceParameters.product);
                            NN_SDK_LOG("    serialNumber:   ");
                            printString(deviceParameters.serialNumber);

                            testInitialize(&ahid, &deviceParameters);
                            testGetString(&ahid);
                            testSetProtocol(&ahid);
                            testGetProtocol(&ahid);
                            testSetIdle(&ahid);
                            testGetIdle(&ahid);
                            testSetReport(&ahid);
                            testGetReport(&ahid);
                            testRead(&ahid);
                            testWrite(&ahid);
                            testFinalize(&ahid);

                            run = false;
                        }
                    }
                }
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    hdr.Finalize();

    NN_SDK_LOG("<-%s\n", __FUNCTION__);
}


//////////////////////////////////////////////////////////////////////////////
extern "C" void nnMain()
{
    NN_SDK_LOG("HdrTest built %s %s\n", __DATE__, __TIME__);

    nn::ahid::hdr::HdrInitialize();         // HDR server
    nn::cdhid::win32::Win32HidInitialize(); // Win32 AHID HID driver

    clientFunction();

    nn::cdhid::win32::Win32HidFinalize();
    nn::ahid::hdr::HdrFinalize();

    NN_SDK_LOG("End of HdrTest\n");
}
