﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>

#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include <memory>

#include <nn/nn_ApplicationId.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#define NNT_ACCOUNT_ENABLE_SAVEDATA_TEST
#endif

#if defined(NNT_ACCOUNT_ENABLE_SAVEDATA_TEST)

namespace
{

NN_ALIGNAS(4096) uint8_t g_InBuffer[256 * 1024];
NN_ALIGNAS(4096) uint8_t g_OutBuffer[256 * 1024];

bool TestTargetNotFound(const nn::account::Uid& uid, const nn::ApplicationId& appId, const size_t BufferSize) NN_NOEXCEPT
{
    for (auto i = 0; i < 2; ++i)
    {
        bool existence;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetSaveDataThumbnailImageExistence(&existence, uid, appId));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(!existence);

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::fs::ResultTargetNotFound,
            nn::account::StoreSaveDataThumbnailImage(uid, appId, g_InBuffer, BufferSize));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::fs::ResultTargetNotFound,
            nn::account::LoadSaveDataThumbnailImage(g_OutBuffer, BufferSize, uid, appId));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::fs::ResultTargetNotFound,
            nn::account::DeleteSaveDataThumbnailImage(uid, appId));
    }

    return true;
}

bool TestBasic(const nn::account::Uid& uid, const nn::ApplicationId& appId, const size_t BufferSize) NN_NOEXCEPT
{
    for (auto i = 0; i < 2; ++i)
    {
        bool existence;

        // -----------------------------------------------------------------------
        // 書き込み 1

        // バッファ書き換え
        std::memset(g_InBuffer, 0x1D, BufferSize / 2);
        std::memset(g_InBuffer + (BufferSize / 2), 0xC8, BufferSize - BufferSize / 2);

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::StoreSaveDataThumbnailImage(uid, appId, g_InBuffer, BufferSize));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetSaveDataThumbnailImageExistence(&existence, uid, appId));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(existence);

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::LoadSaveDataThumbnailImage(g_OutBuffer, BufferSize, uid, appId));

        EXPECT_EQ(0, std::memcmp(g_OutBuffer, g_InBuffer, BufferSize));

        // -----------------------------------------------------------------------
        // 書き込み 2

        // バッファ書き換え
        std::memset(g_InBuffer, 0xE4, BufferSize / 2);
        std::memset(g_InBuffer + (BufferSize / 2), 0x79, BufferSize - BufferSize / 2);

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::StoreSaveDataThumbnailImage(uid, appId, g_InBuffer, BufferSize));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetSaveDataThumbnailImageExistence(&existence, uid, appId));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(existence);

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::LoadSaveDataThumbnailImage(g_OutBuffer, BufferSize, uid, appId));

        EXPECT_EQ(0, std::memcmp(g_OutBuffer, g_InBuffer, BufferSize));

        // -----------------------------------------------------------------------
        // 削除 1

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::DeleteSaveDataThumbnailImage(uid, appId));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetSaveDataThumbnailImageExistence(&existence, uid, appId));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(!existence);

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultSaveDataThumbnailEmpty,
            nn::account::LoadSaveDataThumbnailImage(g_OutBuffer, BufferSize, uid, appId));

        // -----------------------------------------------------------------------
        // 削除 2

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::DeleteSaveDataThumbnailImage(uid, appId));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetSaveDataThumbnailImageExistence(&existence, uid, appId));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(!existence);

        // -----------------------------------------------------------------------
        // 破損 1

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CorruptSaveDataThumbnailFileForDebug(appId.value, nn::fs::ConvertAccountUidToFsUserId(uid)));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::fs::ResultDataCorrupted,
            nn::account::GetSaveDataThumbnailImageExistence(&existence, uid, appId));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::fs::ResultDataCorrupted,
            nn::account::LoadSaveDataThumbnailImage(g_OutBuffer, BufferSize, uid, appId));
    }

    return true;
}

} // ~namespace <anonymous>

TEST(AccountService, SaveDataThumbnail)
{
    nn::account::Uid users[1];

    // UA 作成
    nnt::account::CreateUsers(&users);
    NN_UTIL_SCOPE_EXIT
    {
        nnt::account::Cleanup();
    };

    // システムプログラム権限でテスト
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    struct TestCase
    {
        nn::ApplicationId appId;
        size_t bufferSize;
    } tests[] = {
        {{0x010000000000B123ull}, nn::account::detail::SaveDataThumbnailSize},
    };

    for (const auto& test : tests)
    {
        for (const auto& uid : users)
        {
            // セーブデータ作成前にサムネイルを作れない, 読み取れないことの検査
            EXPECT_TRUE(TestTargetNotFound(users[0], test.appId, test.bufferSize));

            // セーブデータ作成後にサムネイルを作れる, 読み取れることの検査
            {
                nn::ncm::ApplicationId appIdNcm = {test.appId.value};
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateSaveData(
                    appIdNcm, nn::fs::ConvertAccountUidToFsUserId(uid), test.appId.value, 0x10000, 0x10000, 0));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::ns::ProgressMonitorForDeleteUserSaveDataAll prog;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&prog, uid));
                    nn::os::SystemEvent e;
                    prog.GetSystemEvent(&e);
                    e.Wait();
                    NN_ABORT_UNLESS_RESULT_SUCCESS(prog.GetResult());
                };

                EXPECT_TRUE(TestBasic(uid, test.appId, test.bufferSize));
            }

            // セーブデータ削除後にサムネイルを作れない, 読み取れないことの検査
            EXPECT_TRUE(TestTargetNotFound(uid, test.appId, test.bufferSize));
        }
    }
}

#endif
