﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/ndas/account_ResultForNdas.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>

#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include <functional>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

extern void CleanupAccounts() NN_NOEXCEPT;

#define NNT_ACCOUNT_ENABLE_BAAS_TEST

#if defined(NNT_ACCOUNT_ENABLE_BAAS_TEST) && defined(NNT_ACCOUNT_NETWORK_TEST_ENABLE)

namespace
{

/* --------------------------------------------------------------------------------------------
    アサーション
*/
// Uid 版
bool AssertNetworkServiceAccountIdTokenExists(nn::account::NetworkServiceAccountManager& manager) NN_NOEXCEPT
{
    size_t sizeActual;
    nnt::account::Buffer b(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.LoadNetworkServiceAccountIdTokenCache(&sizeActual, b.Get<char>(), b.GetSize()));
    return true;
}
bool AssertNetworkServiceAccountIdTokenNotExist(nn::account::NetworkServiceAccountManager& manager) NN_NOEXCEPT
{
    size_t sizeActual;
    nnt::account::Buffer b(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultBaasTokenCacheNotExist, manager.LoadNetworkServiceAccountIdTokenCache(&sizeActual, b.Get<char>(), b.GetSize()));
    return true;
}
bool AssertNetworkServiceAccountAccessTokenExists(const nn::account::Uid& user) NN_NOEXCEPT
{
    size_t sizeActual;
    nnt::account::Buffer b(nn::account::BaasAccessTokenSizeMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::LoadBaasAccessTokenCache(&sizeActual, b.Get<char>(), b.GetSize(), user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(sizeActual <= nn::account::BaasAccessTokenSizeMax);
    return true;
}
bool AssertNetworkServiceAccountAccessTokenNotExist(const nn::account::Uid& user) NN_NOEXCEPT
{
    size_t sizeActual;
    nnt::account::Buffer b(nn::account::BaasAccessTokenSizeMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultBaasTokenCacheNotExist, nn::account::LoadBaasAccessTokenCache(&sizeActual, b.Get<char>(), b.GetSize(), user));
    return true;
}
// ハンドル版
#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_TEST)
bool AssertNetworkServiceAccountIdTokenExists(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    size_t sizeActual;
    nnt::account::Buffer b(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::LoadNetworkServiceAccountIdTokenCache(&sizeActual, b.Get<char>(), b.GetSize(), handle));
    return true;
}
bool AssertNetworkServiceAccountIdTokenNotExist(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    size_t sizeActual;
    nnt::account::Buffer b(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultBaasTokenCacheNotExist, nn::account::LoadNetworkServiceAccountIdTokenCache(&sizeActual, b.Get<char>(), b.GetSize(), handle));
    return true;
}
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_TEST

/* --------------------------------------------------------------------------------------------
    テストコード
*/
nn::Result DoAsync(std::function<nn::Result(nn::account::AsyncContext*)> f) NN_NOEXCEPT
{
    // 処理
    nn::account::AsyncContext ctx;
    NN_RESULT_DO(f(&ctx));

    nn::os::SystemEvent e;
    NN_RESULT_DO(ctx.GetSystemEvent(&e));
    e.Wait();
    bool done;
    NN_RESULT_DO(ctx.HasDone(&done));
    NN_ABORT_UNLESS(done);
    NN_RESULT_DO(ctx.GetResult());

    NN_RESULT_SUCCESS;
}
template <typename T>
nn::Result DoAsync(std::function<nn::Result(nn::account::AsyncContext*, const T&)> f, const T& t) NN_NOEXCEPT
{
    // 処理
    nn::account::AsyncContext ctx;
    NN_RESULT_DO(f(&ctx, t));

    nn::os::SystemEvent e;
    NN_RESULT_DO(ctx.GetSystemEvent(&e));
    e.Wait();
    bool done;
    NN_RESULT_DO(ctx.HasDone(&done));
    NN_ABORT_UNLESS(done);
    NN_RESULT_DO(ctx.GetResult());

    NN_RESULT_SUCCESS;
}
template <typename T1, typename T2>
nn::Result DoAsync(std::function<nn::Result(nn::account::AsyncContext*, const T1&, const T2&)> f, const T1& t1, const T2& t2) NN_NOEXCEPT
{
    // 処理
    nn::account::AsyncContext ctx;
    NN_RESULT_DO(f(&ctx, t1, t2));

    nn::os::SystemEvent e;
    NN_RESULT_DO(ctx.GetSystemEvent(&e));
    e.Wait();
    bool done;
    NN_RESULT_DO(ctx.HasDone(&done));
    NN_ABORT_UNLESS(done);
    NN_RESULT_DO(ctx.GetResult());

    NN_RESULT_SUCCESS;
}
/* ------------------------------------------------------------
    BaaS Device Account
*/
bool TestBaasUserDeleteForcibly(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        DoAsync([&](nn::account::AsyncContext* p) -> nn::Result {return admin.RegisterAsync(p);}));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.DeleteRegistrationInfoLocally());

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
    return true;
}
bool TestBaasUserRegistration(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        DoAsync([&](nn::account::AsyncContext* p) -> nn::Result {return admin.RegisterAsync(p);}));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));
    return true;
}
bool TestBaasUserUnregistration(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        DoAsync([&](nn::account::AsyncContext* p) -> nn::Result {return admin.UnregisterAsync(p);}));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
    return true;
}
/* ------------------------------------------------------------
    BaaS ID Token
*/
bool TestBaasUserIdTokenCache(const nn::account::Uid& user) NN_NOEXCEPT
{
    NN_LOG("TestBaasUserIdTokenCache: Ensure\n");
    {
        nn::account::InitializeForSystemService();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        nn::account::NetworkServiceAccountManager manager;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
        nn::account::SystemProgramIdentification identification = {{0x010000000000B121ull}, 0x00, };
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetSystemProgramIdentification(identification));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(manager));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenNotExist(manager));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            DoAsync([&](nn::account::AsyncContext* p) -> nn::Result {return manager.EnsureNetworkServiceAccountIdTokenCacheAsync(p); }));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenExists(manager));
    }

    NN_LOG("TestBaasUserIdTokenCache: Invalidate & Ensure\n");
    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        nn::account::NetworkServiceAccountAdministrator admin;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
        nn::account::SystemProgramIdentification identification = {{0x010000000000B121ull}, 0x00, };
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.SetSystemProgramIdentification(identification));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenExists(admin));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.InvalidateNetworkServiceAccountIdTokenCache(identification.id));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenNotExist(admin));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            DoAsync([&](nn::account::AsyncContext* p) -> nn::Result {return admin.EnsureNetworkServiceAccountIdTokenCacheAsync(p); }));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenExists(admin));
    }

    return true;
}
#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_TEST)
bool TestBaasUserIdTokenCacheForApplication(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::UserHandle handle;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, user));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handle);
    };

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(handle));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenNotExist(handle));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(DoAsync<nn::account::UserHandle>(nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync, handle));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenExists(handle));
    return true;
}
#endif
/* ------------------------------------------------------------
    BaaS Access Token
*/
bool TestBaasUserAccessTokenCache(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };
    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
    nn::account::SystemProgramIdentification identification = { {0x010000000000B121ull}, 0x00 };
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        manager.SetSystemProgramIdentification(identification));

    nn::account::InitializeBaasAccessTokenAccessor();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::FinalizeBaasAccessTokenAccessor();
    };
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(manager));
    uint64_t daId;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetBaasDeviceAccountId(&daId, user));
    EXPECT_NE(0ull, daId);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountAccessTokenNotExist(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenNotExist(manager));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(DoAsync<nn::account::Uid>(nn::account::EnsureBaasAccessTokenCacheAsync, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountAccessTokenExists(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenNotExist(manager)); // ここは NOT
    return true;
}
bool TestBaasChannelResource(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeBaasAccessTokenAccessor();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::FinalizeBaasAccessTokenAccessor();
    };

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(DoAsync<nn::account::Uid>(
        nn::account::DeleteBaasChannelResourceAsync, user));

    nn::npns::NotificationToken nt = {"0123456789abcdef0123456789ABCDEF0123456"};
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS((DoAsync<nn::account::Uid, nn::npns::NotificationToken>(
        nn::account::PutBaasChannelResourceAsync, user, nt)));

    nn::npns::NotificationToken nt2 = {"0123456789ABCDEF01234560123456789abcdef"};
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS((DoAsync<nn::account::Uid, nn::npns::NotificationToken>(
        nn::account::PutBaasChannelResourceAsync, user, nt2)));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(DoAsync<nn::account::Uid>(
        nn::account::DeleteBaasChannelResourceAsync, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(DoAsync<nn::account::Uid>(
        nn::account::DeleteBaasChannelResourceAsync, user));
    return true;
}
bool TestBaasUserAccessTokenCacheByIdTokenCache(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
    nn::account::SystemProgramIdentification identification = { {0x010000000000B121ull}, 0x00 };
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        manager.SetSystemProgramIdentification(identification));

    nn::account::InitializeBaasAccessTokenAccessor();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::FinalizeBaasAccessTokenAccessor();
    };
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(manager));
    uint64_t daId;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetBaasDeviceAccountId(&daId, user));
    EXPECT_NE(0ull, daId);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountAccessTokenNotExist(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenNotExist(manager));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        DoAsync([&](nn::account::AsyncContext* p) -> nn::Result {return manager.EnsureNetworkServiceAccountIdTokenCacheAsync(p);}));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountAccessTokenExists(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(AssertNetworkServiceAccountIdTokenExists(manager));
    return true;
}

bool TestNetworkServiceAccount(const nn::account::Uid& user) NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserRegistration(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserIdTokenCache(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserUnregistration(user));

#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_TEST)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserRegistration(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserIdTokenCacheForApplication(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserUnregistration(user));
#endif

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserRegistration(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserAccessTokenCache(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasChannelResource(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserUnregistration(user));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserRegistration(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserAccessTokenCacheByIdTokenCache(user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserUnregistration(user));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestBaasUserDeleteForcibly(user));
    return true;
}
} // ~namespace <anonymous>

TEST(AccountService, NetworkServiceAccount)
{
    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    const int NumUsers = 2;
    nn::account::Uid users[NumUsers];
    nnt::account::CreateUsers(&users);

    for (int i = 0; i < NumUsers; ++ i)
    {
        ASSERT_TRUE(TestNetworkServiceAccount(users[i]));
    }
}

TEST(AccountService, NetworkServiceAvailability)
{
    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    nn::account::SystemProgramIdentification identification = { {0x010000000000B121ull}, 0x00 };

    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    // test Error
    {
        nn::account::AsyncContext async;
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CheckNetworkServiceAvailabilityAsync(&async, identification));
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::account::ResultNoAvailableNetworkServiceAccounts, async.GetResult());
    }

    // 作成
    const int NumUsers = 2;
    nn::account::Uid users[NumUsers];
    nnt::account::CreateUsers(&users);

    // test Error
    {
        nn::account::AsyncContext async;
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CheckNetworkServiceAvailabilityAsync(&async, identification));
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::account::ResultNoAvailableNetworkServiceAccounts, async.GetResult());
    }

    // NSA 作成
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[1]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    }

    // test Success
    {
        nn::account::AsyncContext async;
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CheckNetworkServiceAvailabilityAsync(&async, identification));
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(async.GetResult());
    }

    // NSA 作成
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[0]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    }

    // test Success
    {
        nn::account::AsyncContext async;
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CheckNetworkServiceAvailabilityAsync(&async, identification));
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(async.GetResult());
    }

    // NSA 削除
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[1]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
    }

    // test Success
    {
        nn::account::AsyncContext async;
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CheckNetworkServiceAvailabilityAsync(&async, identification));
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(async.GetResult());
    }

    // NSA 削除
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[0]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
    }

    // test Error
    {
        nn::account::AsyncContext async;
        nn::os::SystemEvent e;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::CheckNetworkServiceAvailabilityAsync(&async, identification));
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::account::ResultNoAvailableNetworkServiceAccounts, async.GetResult());
    }
}

#endif // NNT_ACCOUNT_ENABLE_BAAS_TEST && NNT_ACCOUNT_ENABLE_NETWORK_TEST
