﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_CachedNintendoAccountInfo.h>

#include <nn/account/nas/account_NasTypes.h>

#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include "testAccount_LinkageApi.h"

#include <type_traits>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Country.h>
#include <nn/util/util_ScopeExit.h>

namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_NAS_USERRESOURCE

#if defined(NNT_ACCOUNT_ENABLE_NAS_USERRESOURCE)

namespace
{
bool CompareLoginInfo(
    const nnt::account::NasLoginInfo& loginInfo,
    const nn::account::CachedNintendoAccountInfoForSystemService& cachedInfo) NN_NOEXCEPT
{
    NN_LOG("[account] User resource comparison\n");
    const char* value;
    size_t bytes;
    if (std::strchr(loginInfo.email, '@') != nullptr)
    {
        value = cachedInfo.GetEmailAddress(&bytes);
    }
    else
    {
        value = cachedInfo.GetLoginId(&bytes);
    }
    EXPECT_EQ(strlen(loginInfo.email) + 1, bytes);
    return std::strncmp(loginInfo.email, value, sizeof(loginInfo.email)) == 0;
}

#define NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(o0, o1, method) \
    do \
    { \
        size_t bytes0, bytes; \
        auto value0 = o0.method(&bytes0); \
        auto value = o1.method(&bytes); \
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(bytes0, strlen(value0) + 1); \
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(bytes0, bytes); \
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, std::strncmp(value0, value, bytes0)); \
    } while (NN_STATIC_CONDITION(false))

bool CompareCachedInfoForSystemService(
    const nn::account::CachedNintendoAccountInfoForSystemService& cachedInfo0,
    const nn::account::CachedNintendoAccountInfoForSystemService& cachedInfo) NN_NOEXCEPT
{
    nn::account::NintendoAccountId naId = cachedInfo.GetId();
    EXPECT_NE(0x00ull, naId.id);

    EXPECT_EQ(cachedInfo0.IsChild(), cachedInfo.IsChild());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetBirthday);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetCountry);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetEmailAddress);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetGender);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetLanguage);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetLoginId);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetRegion);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetScreenName);
    EXPECT_EQ(cachedInfo0.GetAnalyticsOptedInFlag(), cachedInfo.GetAnalyticsOptedInFlag());

    return true;
}
bool CompareCachedInfoForApplication(
    const nn::account::CachedNintendoAccountInfoForSystemService& cachedInfo0,
    const nn::account::CachedNintendoAccountInfo& cachedInfo) NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetBirthday);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetCountry);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_MATCH_STRING_ELEMENT(cachedInfo0, cachedInfo, GetGender);
    return true;
}

bool TestBasic(const nn::account::Uid& user, const nnt::account::NasLoginInfo& loginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::CachedNintendoAccountInfoForSystemService cachedInfo0;
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
            &admin, user));
        t::Buffer buffer(nn::account::RequiredBufferSizeForCachedNintendoAccountInfo);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.LoadCachedNintendoAccountInfo(
            &cachedInfo0, buffer.GetAddress(), buffer.GetSize()));

        nn::account::AsyncContext async;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.RefreshCachedNintendoAccountInfoAsync(
            &async));
        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetResult());

        EXPECT_TRUE(CompareLoginInfo(loginInfo, cachedInfo0));
    }

    nn::account::Finalize();
    nn::account::InitializeForSystemService();

    {
        nn::account::CachedNintendoAccountInfoForSystemService cachedInfo;

        nn::account::NetworkServiceAccountManager manager;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(
            &manager, user));
        t::Buffer buffer(nn::account::RequiredBufferSizeForCachedNintendoAccountInfo);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.LoadCachedNintendoAccountInfo(
            &cachedInfo, buffer.GetAddress(), buffer.GetSize()));

        bool matched;
        nn::account::AsyncContext async;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(
            &matched,
            &async, nn::TimeSpan::FromSeconds(0xFFFF)));
        EXPECT_FALSE(matched);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(
            &matched,
            &async, nn::TimeSpan::FromSeconds(3)));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(matched);
        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetResult());

        EXPECT_TRUE(CompareCachedInfoForSystemService(cachedInfo0, cachedInfo));
    }

    nn::account::Finalize();
    nn::account::Initialize();

    {
        nn::account::CachedNintendoAccountInfo cachedInfo;

        nn::account::UserHandle handle;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, user));
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::CloseUser(handle);
        };

        t::Buffer buffer(nn::account::RequiredBufferSizeForCachedNintendoAccountInfo);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::LoadCachedNintendoAccountInfo(
            &cachedInfo, handle, buffer.GetAddress(), buffer.GetSize()));

        EXPECT_TRUE(CompareCachedInfoForApplication(cachedInfo0, cachedInfo));
    }

    return true;
}
} // ~namespace <anonymous>

TEST(AccountNintendoAccount, UserResource)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        for (auto i = 0; i < naList.Count(); ++ i)
        {
            nn::account::NetworkServiceAccountAdministrator admin;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
                &admin, users[i]));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(LinkNintendoAccount(naList[i], admin));
        }
    }

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(TestBasic(users[i], naList[i]));
    }
}

TEST(AccountNintendoAccount, UserCountry)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto na = nnt::account::LoadNaList()[0];
    nn::account::Uid user;
    nnt::account::CreateUsers(&user, 1);

    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        nn::account::NetworkServiceAccountAdministrator admin;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(LinkNintendoAccount(na, admin));
    }

    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
    t::Buffer buffer(nn::account::RequiredBufferSizeForCachedNintendoAccountInfo);

    nn::account::CachedNintendoAccountInfoForSystemService cachedInfo;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(manager.LoadCachedNintendoAccountInfo(&cachedInfo, buffer.GetAddress(), buffer.GetSize()));

    size_t countryCodeSize;
    auto* countryCode = cachedInfo.GetCountry(&countryCodeSize);
    nn::util::Country country;
    ASSERT_TRUE(nn::util::GetCountryFromIso31661Alpha2(&country, countryCode, countryCodeSize));
    ASSERT_TRUE(false
        || country == nn::util::Country_Jp
        || country == nn::util::Country_Us
        || country == nn::util::Country_Gb);
}

#endif // NNT_ACCOUNT_ENABLE_NAS_USERRESOURCE
