﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>

#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/nas/account_ResultForNas.h>

#include "testAccount_NasProxy.h"
#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include "testAccount_LinkageApi.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#define NNT_ACCOUNT_ENABLE_NAS_LINKAGE
#define NNT_ACCOUNT_ENABLE_NAS_LINKAGE_NNID
#define NNT_ACCOUNT_ENABLE_NAS_LINKAGE_URL
#define NNT_ACCOUNT_ENABLE_NAS_DUPLICATIVE

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#define NNT_ACCOUNT_ENABLE_NAS_LINKAGE_ERROR
#endif

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#define NNT_ACCOUNT_ENABLE_NAS_LINKAGE_STATE
#endif

#if defined(NNT_ACCOUNT_ENABLE_NAS_LINKAGE)

namespace linkage
{
bool IsNintendoAccountLinked(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    auto registered = nnt::account::IsNetworkServiceAccountRegistered(admin);
    if (!registered)
    {
        return false;
    }
    bool linked;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&linked));
    return linked;
}
bool IsNintendoAccountNotLinked(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool registered;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&registered));
    if (!registered)
    {
        bool linked;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultNetworkServiceAccountRegistrationRequired, admin.IsLinkedWithNintendoAccount(&linked));
        return true;
    }
    bool linked;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&linked));
    return !linked;
}
bool IsNintendoAccountAvailable(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    auto r = admin.CheckNetworkServiceAccountAvailability();
    if (!(false
        || r.IsSuccess()
        || nn::account::ResultNintendoAccountLinkageRequired::Includes(r)))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    return r.IsSuccess();
}
bool IsNintendoAccountNotAvailable(nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    auto r = admin.CheckNetworkServiceAccountAvailability();
    if (!(r.IsSuccess() || nn::account::ResultNintendoAccountLinkageRequired::Includes(r)))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    return r.IsFailure();
}

nn::Result UpdateLinkageStateOfNintendoAccount(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool isLinked;
    NN_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);

    nn::account::NintendoAccountLinkageStateUpdateProcedure fixProc;
    NN_RESULT_DO(admin.CreateProcedureToUpdateLinkageStateOfNintendoAccount(&fixProc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NN_RESULT_DO(fixProc.GetRequest(&requestUrl, &callbackUri));

    auto response = nnt::account::GetAuthorizationViaNasProxy(loginInfo, requestUrl.url);
    nn::account::AsyncContext linkTask;
    NN_RESULT_DO(fixProc.ApplyResponseAsync(&linkTask, response.Get<char>(), response.GetSize()));

    nn::os::SystemEvent linkEvent;
    NN_RESULT_DO(linkTask.GetSystemEvent(&linkEvent));
    linkEvent.Wait();

    NN_RESULT_DO(linkTask.GetResult());

    NN_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);
    NN_RESULT_SUCCESS;
}

nn::Result UpdateLinkageStateOfNintendoAccountWithSuspend(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    bool isLinked;
    NN_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);

    nn::account::SessionId sessionId;
    nnt::account::Buffer response;

    // 中断
    {
        nn::account::NintendoAccountLinkageStateUpdateProcedure fixProc;
        NN_RESULT_DO(admin.CreateProcedureToUpdateLinkageStateOfNintendoAccount(&fixProc));

        nn::account::RequestUrl requestUrl;
        nn::account::CallbackUri callbackUri;
        NN_RESULT_DO(fixProc.GetRequest(&requestUrl, &callbackUri));

        NN_RESULT_DO(fixProc.Suspend(&sessionId));
        fixProc = nn::account::NintendoAccountLinkageStateUpdateProcedure();

        // ブラウザ呼び出し相当の処理
        response = nnt::account::GetAuthorizationViaNasProxy(loginInfo, requestUrl.url);
    }

    // 再開
    {
        nn::account::NintendoAccountLinkageStateUpdateProcedure fixProc;
        NN_RESULT_DO(admin.ResumeProcedureToUpdateLinkageStateOfNintendoAccount(&fixProc, sessionId));

        nn::account::AsyncContext linkTask;
        NN_RESULT_DO(fixProc.ApplyResponseAsync(&linkTask, response.Get<char>(), response.GetSize()));

        nn::os::SystemEvent linkEvent;
        NN_RESULT_DO(linkTask.GetSystemEvent(&linkEvent));
        linkEvent.Wait();

        NN_RESULT_DO(linkTask.GetResult());

        NN_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
        EXPECT_TRUE(isLinked);
        NN_RESULT_SUCCESS;
    }
}

nn::Result RecoveryNintendoAccountUserStateImplicitly(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    auto r = admin.CheckNetworkServiceAccountAvailability();
    EXPECT_TRUE(r.IsSuccess() || nn::account::ResultNintendoAccountStateInteractionRequired::Includes(r));

    nn::account::AsyncContext recoveryTask;
    NN_RESULT_DO(admin.TryRecoverNintendoAccountUserStateAsync(&recoveryTask));

    nn::os::SystemEvent recoveryEvent;
    NN_RESULT_DO(recoveryTask.GetSystemEvent(&recoveryEvent));
    recoveryEvent.Wait();

    NN_RESULT_DO(recoveryTask.GetResult());

    NN_RESULT_DO(admin.CheckNetworkServiceAccountAvailability());
    NN_RESULT_SUCCESS;
}

nn::Result LinkNnidWithNintendoAccount(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    NN_LOG("LinkNnidWithNintendoAccount\n");
    NN_RESULT_DO(admin.CheckNetworkServiceAccountAvailability());

    nn::account::NintendoAccountNnidLinkageProcedure nnidProc;
    NN_RESULT_DO(admin.CreateProcedureToLinkNnidWithNintendoAccount(&nnidProc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NN_RESULT_DO(nnidProc.GetRequest(&requestUrl, &callbackUri));

    NN_LOG("- Request URL: %s\n", requestUrl.url);
    NN_LOG("- Callback URI: %s\n", callbackUri.uri);

    auto response = nnt::account::GetAuthorizationViaNasProxyForGuest(loginInfo, requestUrl.url);
    nn::account::AsyncContext nnidTask;
    NN_RESULT_DO(nnidProc.ApplyResponseAsync(&nnidTask, response.Get<char>(), response.GetSize()));

    nn::os::SystemEvent e;
    NN_RESULT_DO(nnidTask.GetSystemEvent(&e));
    e.Wait();

    NN_RESULT_DO(nnidTask.GetResult());

    NN_RESULT_SUCCESS;
}

nn::Result LinkNnidWithNintendoAccountWithSuspend(
    const nnt::account::NasLoginInfo& loginInfo,
    nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
{
    NN_LOG("LinkNnidWithNintendoAccountWithSuspend\n");
    NN_RESULT_DO(admin.CheckNetworkServiceAccountAvailability());

    nn::account::SessionId sessionId;
    nnt::account::Buffer response;

    // 中断
    {
        nn::account::NintendoAccountNnidLinkageProcedure nnidProc;
        NN_RESULT_DO(admin.CreateProcedureToLinkNnidWithNintendoAccount(&nnidProc));

        nn::account::RequestUrl requestUrl;
        nn::account::CallbackUri callbackUri;
        NN_RESULT_DO(nnidProc.GetRequest(&requestUrl, &callbackUri));

        NN_LOG("- Request URL: %s\n", requestUrl.url);
        NN_LOG("- Callback URI: %s\n", callbackUri.uri);

        NN_RESULT_DO(nnidProc.Suspend(&sessionId));
        nnidProc = nn::account::NintendoAccountNnidLinkageProcedure();

        // ブラウザ呼び出し相当の処理
        response = nnt::account::GetAuthorizationViaNasProxyForGuest(loginInfo, requestUrl.url);
    }

    // 再開
    {
        nn::account::NintendoAccountNnidLinkageProcedure nnidProc;
        NN_RESULT_DO(admin.ResumeProcedureToLinkNnidWithNintendoAccount(&nnidProc, sessionId));

        nn::account::AsyncContext nnidTask;
        NN_RESULT_DO(nnidProc.ApplyResponseAsync(&nnidTask, response.Get<char>(), response.GetSize()));

        nn::os::SystemEvent e;
        NN_RESULT_DO(nnidTask.GetSystemEvent(&e));
        e.Wait();

        NN_RESULT_DO(nnidTask.GetResult());
    }
    NN_RESULT_SUCCESS;
}

bool TestBasic(const nn::account::Uid& user, const nnt::account::NasLoginInfo& LoginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
#endif

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
#endif

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UnlinkNintendoAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
#endif

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccountWithSuspend(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UpdateLinkageStateOfNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UpdateLinkageStateOfNintendoAccountWithSuspend(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
#endif

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UnlinkNintendoAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotAvailable(admin));
#endif

    return true;
}

bool TestNnid(const nn::account::Uid& user, const nnt::account::NasLoginInfo& LoginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNnidWithNintendoAccount(LoginInfo, admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNnidWithNintendoAccountWithSuspend(LoginInfo, admin));

    return true;
}

bool TestRequestUrl(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));

    nn::account::NintendoAccountLinkageProcedure linkProc;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.CreateProcedureToLinkWithNintendoAccount(&linkProc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkProc.GetRequest(&requestUrl, &callbackUri));
    NN_LOG("URL: %s\n", requestUrl.url);
    EXPECT_TRUE(std::strstr(requestUrl.url, "&theme=intro&") != nullptr);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkProc.GetRequest(
        &requestUrl, &callbackUri, nn::account::NintendoAccountAuthorizationPageTheme_Intro));
    NN_LOG("URL(Intro): %s\n", requestUrl.url);
    EXPECT_TRUE(std::strstr(requestUrl.url, "&theme=intro&") != nullptr);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkProc.GetRequest(
        &requestUrl, &callbackUri, nn::account::NintendoAccountAuthorizationPageTheme_Register));
    NN_LOG("URL(Register): %s\n", requestUrl.url);
    EXPECT_TRUE(std::strstr(requestUrl.url, "&theme=register&") != nullptr);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkProc.GetRequest(
        &requestUrl, &callbackUri, nn::account::NintendoAccountAuthorizationPageTheme_EmailAuthentication));
    NN_LOG("URL(Register): %s\n", requestUrl.url);
    EXPECT_TRUE(std::strstr(requestUrl.url, "&theme=email_authentication&") != nullptr);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkProc.GetRequest(
        &requestUrl, &callbackUri, nn::account::NintendoAccountAuthorizationPageTheme_SimpleAuthentication));
    NN_LOG("URL(Register): %s\n", requestUrl.url);
    EXPECT_TRUE(std::strstr(requestUrl.url, "&theme=simple_authenticate&") != nullptr);

    return true;
}

template <typename ResultType>
bool TestErrorCaseForLinkage(
    nn::account::NetworkServiceAccountAdministrator& admin,
    const char* error, const char* detail = nullptr) NN_NOEXCEPT
{
    bool isLinked;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_FALSE(isLinked);

    nn::account::NintendoAccountLinkageProcedure linkProc;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.CreateProcedureToLinkWithNintendoAccount(&linkProc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkProc.GetRequest(&requestUrl, &callbackUri));

    auto response = nnt::account::GetAuthorizationForError(requestUrl.url, error, detail);

    nn::account::AsyncContext linkTask;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        ResultType,
        linkProc.ApplyResponseAsync(&linkTask, response.Get<char>(), response.GetSize()));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_FALSE(isLinked);

    return true;
}

template <typename ResultType>
bool TestErrorCaseForUpdate(
    nn::account::NetworkServiceAccountAdministrator& admin,
    const char* error, const char* detail = nullptr) NN_NOEXCEPT
{
    bool isLinked;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);

    nn::account::NintendoAccountLinkageStateUpdateProcedure updateProc;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.CreateProcedureToUpdateLinkageStateOfNintendoAccount(&updateProc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(updateProc.GetRequest(&requestUrl, &callbackUri));

    auto response = nnt::account::GetAuthorizationForError(requestUrl.url, error, detail);

    nn::account::AsyncContext updateTask;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        ResultType,
        updateProc.ApplyResponseAsync(&updateTask, response.Get<char>(), response.GetSize()));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&isLinked));
    EXPECT_TRUE(isLinked);

    return true;
}

bool TestError(const nn::account::Uid& user, const nnt::account::NasLoginInfo& LoginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusUnauthorizedClient>(
        admin, "unauthorized_client"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::ResultCancelledByUser>(
        admin, "access_denied"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::ResultCancelledByUser>(
        admin, "access_denied", "user_terms_agreement_required "));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::ResultCancelledByUser>(
        admin, "access_denied", "user_insufficient_age"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusAccessDeniedIdTokenHintInvalid>(
        admin, "access_denied", "id_token_hint_invalid"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::ResultCancelledByUser>(
        admin, "access_denied", "user_terms_agreement_required"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusAccessDeniedUserDeleted>(
        admin, "access_denied", "user_deleted"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInvalidScope>(
        admin, "invalid_scope"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInvalidScopeScopeTokenUnknown>(
        admin, "invalid_scope", "scope_token_unknown"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInvalidScopeScopeTokenProhibited>(
        admin, "invalid_scope", "scope_token_prohibited"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusServerError>(
        admin, "server_error"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusLoginRequired>(
        admin, "login_required"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusLoginRequiredUserNotLoggedIn>(
        admin, "login_required", "user_not_logged_in"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::ResultUserInputDenied>(
        admin, "login_required", "user_different_from_id_token_hint"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::ResultCancelledByUser>(
        admin, "consent_required"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInteractionRequired>(
        admin, "interaction_required"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInteractionRequired>(
        admin, "interaction_required", "user_banned "));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInteractionRequiredUserBanned>(
        admin, "interaction_required", "user_banned"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInteractionRequiredUserSuspended>(
        admin, "interaction_required", "user_suspended"));

    EXPECT_TRUE(TestErrorCaseForLinkage<nn::account::nas::ResultNasAuthorizationStatusInteractionRequiredUserTermsAgreementRequired>(
        admin, "interaction_required", "user_terms_agreement_required"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    // Update linkage
    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusUnauthorizedClient>(
        admin, "unauthorized_client"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultCancelledByUser>(
        admin, "access_denied"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultCancelledByUser>(
        admin, "access_denied", "user_terms_agreement_required_"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultCancelledByUser>(
        admin, "access_denied", "user_insufficient_age"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultCancelledByUser>(
        admin, "access_denied", "user_terms_agreement_required"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInvalidScope>(
        admin, "invalid_scope"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInvalidScopeScopeTokenUnknown>(
        admin, "invalid_scope", "scope_token_unknown"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInvalidScopeScopeTokenProhibited>(
        admin, "invalid_scope", "scope_token_prohibited"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusServerError>(
        admin, "server_error"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusLoginRequired>(
        admin, "login_required"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusLoginRequiredUserNotLoggedIn>(
        admin, "login_required", "user_not_logged_in"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultUserInputDenied>(
        admin, "login_required", "user_different_from_id_token_hint"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultCancelledByUser>(
        admin, "consent_required"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInteractionRequired>(
        admin, "interaction_required"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInteractionRequired>(
        admin, "interaction_required", "user_banned "));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInteractionRequiredUserBanned>(
        admin, "interaction_required", "user_banned"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInteractionRequiredUserSuspended>(
        admin, "interaction_required", "user_suspended"));

    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusInteractionRequiredUserTermsAgreementRequired>(
        admin, "interaction_required", "user_terms_agreement_required"));

    // ID トークン破損による連携状態の破壊
    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::nas::ResultNasAuthorizationStatusAccessDeniedIdTokenHintInvalid>(
        admin, "access_denied", "id_token_hint_invalid"));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.CheckNetworkServiceAccountAvailability());

    // EULA 再同意必須状態からの削除
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.DebugSetNetworkServiceAccountAvailabilityError(
        nn::account::ResultNintendoAccountStateTermsAgreementRequired()));
    EXPECT_TRUE(TestErrorCaseForUpdate<nn::account::ResultNintendoAccountStateDeleted>(
        admin, "access_denied", "user_deleted"));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultNintendoAccountStateDeleted, admin.CheckNetworkServiceAccountAvailability());

    return true;
} // NOLINT(readability/fn_size)

bool TestDuplicative(const nn::account::Uid& user, const nn::account::Uid& user1, const nnt::account::NasLoginInfo& LoginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountLinked(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountAvailable(admin));

    {
        nn::account::NetworkServiceAccountAdministrator admin1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin1, user1));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountNotRegistered(admin1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin1));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin1));

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultDuplicativeNintendoAccount,
            LinkNintendoAccount(LoginInfo, admin1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin1));
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultNintendoAccountLinkageRequired,
            admin1.CheckNetworkServiceAccountAvailability());
#endif
    }

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UnlinkNintendoAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(nnt::account::IsNetworkServiceAccountRegistered(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotLinked(admin));

    return true;
}

} // ~namespace linkage

#if defined(NNT_ACCOUNT_ENABLE_NAS_LINKAGE_BASIC)
TEST(AccountNintendoAccount, Linkage)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(linkage::TestBasic(users[i], naList[i]));
    }
}
#endif // NNT_ACCOUNT_ENABLE_NAS_LINKAGE_BASIC


#if defined(NNT_ACCOUNT_ENABLE_NAS_LINKAGE_NNID)
TEST(AccountNintendoAccount, LinkageNnid)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(linkage::TestNnid(users[i], naList[i]));
    }
}
#endif // NNT_ACCOUNT_ENABLE_NAS_LINKAGE_NNID

#if defined(NNT_ACCOUNT_ENABLE_NAS_LINKAGE_URL)
TEST(AccountNintendoAccount, LinkageUrl)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    const int NumUsers = 1;
    nn::account::Uid users[NumUsers];
    nnt::account::CreateUsers(&users);

    for (auto& u : users)
    {
        ASSERT_TRUE(linkage::TestRequestUrl(u));
    }
}
#endif // NNT_ACCOUNT_ENABLE_NAS_LINKAGE_URL

#if defined(NNT_ACCOUNT_ENABLE_NAS_LINKAGE_ERROR)
TEST(AccountNintendoAccount, LinkageError)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(linkage::TestError(users[i], naList[i]));
    }
}
#endif // NNT_ACCOUNT_ENABLE_NAS_LINKAGE_ERROR

#if defined(NNT_ACCOUNT_ENABLE_NAS_DUPLICATIVE)
TEST(AccountNintendoAccount, LinkageDuplicative)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    nn::account::Uid users[2];
    nnt::account::CreateUsers(&users);

    ASSERT_TRUE(linkage::TestDuplicative(users[0], users[1], naList[0]));
}
#endif // NNT_ACCOUNT_ENABLE_NAS_DUPLICATIVE

#if defined(NNT_ACCOUNT_ENABLE_NAS_LINKAGE_STATE)

namespace linkage
{
bool TestLinkageState(const nn::account::Uid& user, const nnt::account::NasLoginInfo& LoginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    nn::account::NintendoAccountId naId;
    nn::account::NintendoAccountId testNaId;

    // 連携
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountRegistrationRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&naId));

    // 解除
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UnlinkNintendoAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotAvailable(admin));

    // 更新
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UpdateLinkageStateOfNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);

    // User state 破壊
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        admin.DebugSetNetworkServiceAccountAvailabilityError(nn::account::ResultNintendoAccountStateTermsAgreementRequired()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountStateTermsAgreementRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountStateInteractionRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountInvalidState, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);

    // 暗黙的回復
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(RecoveryNintendoAccountUserStateImplicitly(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);

    // User state 破壊
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        admin.DebugSetNetworkServiceAccountAvailabilityError(nn::account::ResultNintendoAccountStateOtherButInteractionRequired()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountStateOtherButInteractionRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountStateInteractionRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountInvalidState, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);

    // 回復
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(UpdateLinkageStateOfNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);

    // 連携 破壊, 回復不可
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        admin.DebugSetNetworkServiceAccountAvailabilityError(nn::account::ResultNintendoAccountLinkageBroken()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountLinkageBroken, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountInvalidState, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(
        !nn::account::ResultNintendoAccountStateInteractionRequired::Includes(admin.CheckNetworkServiceAccountAvailability()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNintendoAccountLinkageBroken, UpdateLinkageStateOfNintendoAccount(LoginInfo, admin));

    // NSA 破棄, 再取得, NA 再連携
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountRegistrationRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountRegistrationRequired, admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));

    // NSA 破壊, 回復不可
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        admin.DebugSetNetworkServiceAccountAvailabilityError(nn::account::ResultNetworkServiceAccountCredentialBroken()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountCredentialBroken, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountInvalidState, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(
        !nn::account::ResultNintendoAccountInvalidState::Includes(admin.CheckNetworkServiceAccountAvailability()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountCredentialBroken, admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountCredentialBroken, UpdateLinkageStateOfNintendoAccount(LoginInfo, admin));

    // NSA 破棄, 再取得, NA 再連携
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountRegistrationRequired, admin.CheckNetworkServiceAccountAvailability());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultNetworkServiceAccountRegistrationRequired, admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountNotAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(LoginInfo, admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(IsNintendoAccountAvailable(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.GetNintendoAccountId(&testNaId));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == testNaId);

    return true;
}
} // namespace linkage

TEST(AccountNintendoAccount, LinkageState)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(linkage::TestLinkageState(users[i], naList[i]));
    }
}

#endif // NNT_ACCOUNT_ENABLE_NAS_LINKAGE_STATE

#endif // NNT_ACCOUNT_ENABLE_NAS_LINKAGE
