﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForMigrationService.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>

#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/nas/account_NasSessionPool.h>

#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include "testAccount_LinkageApi.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ
#define NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ_FOR_SHOP

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#define NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZERROR
#endif

#define NNT_ACCOUNT_ENABLE_USER_IMPORT
#define NNT_ACCOUNT_ENABLE_NSA_ID_TOKEN

namespace {
NN_ALIGNAS(4096) char AuthorizationContextBuffer[nn::os::MemoryPageSize];
} // ~namespace <anonymous>

#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ)
namespace
{
bool TestBasic(const nn::account::Uid& user, const nnt::account::NasLoginInfo& loginInfo) NN_NOEXCEPT
{
    uint64_t clientId = 0x57d3dbaa12cb06a9ull; // td1
    // uint64_t clientId = 0x2c44652b8960c92eull; // dd1
    const char redirectUri[] = "nintendo://0100000000002803.app"; // td1,td1

    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    {
        nn::account::NintendoAccountAuthorizationRequestParameters param = {
            "user openid", // scope
            "ZZZ", // state
            "YYY", // nonce
        };

        nn::account::NetworkServiceAccountAdministrator admin;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
            &admin, user));

        nn::account::NintendoAccountAuthorizationRequestContext ctx;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.CreateNintendoAccountAuthorizationRequest(
            &ctx, clientId, redirectUri, sizeof(redirectUri), param, AuthorizationContextBuffer, sizeof(AuthorizationContextBuffer)));

        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultNintendoAccountAuthorizationInteractionRequired,
            ctx.GetResult());
        bool done;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.HasDone(&done));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(done);

        // 認可取得
        {
            nn::account::SessionId sessionId;
            ctx.DebugGetInfo(reinterpret_cast<uint64_t*>(&sessionId), sizeof(sessionId) / sizeof(uint64_t));
            nn::account::NintendoAccountApplicationAuthorizationProcedure proc;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(&proc, sessionId));

            nn::account::AsyncContext async;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.PrepareAsync(&async));
            NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
            e.Wait();
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetResult());

            nn::account::RequestUrl request;
            nn::account::CallbackUri callback;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.GetRequest(&request, &callback));
            NN_LOG("Request URL: %s\n", request.url);

            auto response = nnt::account::GetApplicationAuthorizationViaNasProxy(loginInfo, request.url);
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.ApplyResponse(response.Get<char>(), response.GetSize()));
        }

        size_t codeSize;
        t::Buffer code(nn::account::NintendoAccountAuthorizationCodeLengthMax + 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&codeSize, code.Get<char>(), code.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(codeSize <= nn::account::NintendoAccountAuthorizationCodeLengthMax);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(code.Get<char>(), code.GetSize()) == codeSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&codeSize, code.Get<char>(), code.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(codeSize <= nn::account::NintendoAccountAuthorizationCodeLengthMax);

        size_t idTokenSize;
        t::Buffer idToken(nn::account::NintendoAccountIdTokenLengthMax + 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&idTokenSize, idToken.Get<char>(), idToken.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(idTokenSize <= nn::account::NintendoAccountIdTokenLengthMax);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(idToken.Get<char>(), idToken.GetSize()) == idTokenSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&idTokenSize, idToken.Get<char>(), idToken.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(idTokenSize <= nn::account::NintendoAccountIdTokenLengthMax);

        size_t stateSize;
        t::Buffer state(256);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetState(&stateSize, state.Get<char>(), state.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(stateSize <= 255);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(state.Get<char>(), state.GetSize()) == stateSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetState(&stateSize, state.Get<char>(), state.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(stateSize <= 255);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::strncmp(param.state, state.Get<char>(), state.GetSize()) == 0);
    }

    nn::account::Finalize();
    nn::account::InitializeForSystemService();

    {
        nn::account::NintendoAccountAuthorizationRequestParameters param = {
            "user openid", // scope
            "longstatelongstatelongstatelongstatelongstatelongstatelongstatelongstatelongstatelongstatelongstatelongstatelongstatelongstate", // state
            "longnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnoncelongnonce", // nonce
        };

        nn::account::NetworkServiceAccountManager manager;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(
            &manager, user));

        nn::account::NintendoAccountAuthorizationRequestContext ctx;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.CreateNintendoAccountAuthorizationRequest(
            &ctx, clientId, redirectUri, sizeof(redirectUri), param, AuthorizationContextBuffer, sizeof(AuthorizationContextBuffer)));

        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
        bool done;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.HasDone(&done));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(done);

        size_t codeSize;
        t::Buffer code(nn::account::NintendoAccountAuthorizationCodeLengthMax + 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&codeSize, code.Get<char>(), code.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(codeSize <= nn::account::NintendoAccountAuthorizationCodeLengthMax);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(code.Get<char>(), code.GetSize()) == codeSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&codeSize, code.Get<char>(), code.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(codeSize <= nn::account::NintendoAccountAuthorizationCodeLengthMax);

        size_t idTokenSize;
        t::Buffer idToken(nn::account::NintendoAccountIdTokenLengthMax + 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&idTokenSize, idToken.Get<char>(), idToken.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(idTokenSize <= nn::account::NintendoAccountIdTokenLengthMax);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(idToken.Get<char>(), idToken.GetSize()) == idTokenSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetAuthorizationCode(&idTokenSize, idToken.Get<char>(), idToken.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(idTokenSize <= nn::account::NintendoAccountIdTokenLengthMax);

        size_t stateSize;
        t::Buffer state(128);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetState(&stateSize, state.Get<char>(), state.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(stateSize == strlen(param.state));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(state.Get<char>(), state.GetSize()) == stateSize);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetState(&stateSize, state.Get<char>(), state.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(stateSize == strlen(param.state));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(std::strncmp(param.state, state.Get<char>(), state.GetSize()) == 0);
    }
    return true;
} // NOLINT(readability/fn_size)

#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ_FOR_SHOP)
NN_ALIGNAS(4096) static char g_AuthRequestWork[nn::account::nas::NasSessionPoolBase::SessionCountMax][4096];
bool TestShop(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));

    nn::nsd::NasServiceSetting shopSetting;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::nsd::GetNasServiceSetting(&shopSetting, nn::nsd::NasServiceNameOfNxShop));

    {
        nn::account::NintendoAccountAuthorizationRequestContext ctx;

        nn::account::NintendoAccountAuthorizationRequestParameters params = { "openid", "a", "b" };
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.CreateNintendoAccountAuthorizationRequest(
            &ctx,
            shopSetting.clientId,
            shopSetting.redirectUri.value,
            nn::nsd::NasServiceSetting::RedirectUri::Size,
            params,
            g_AuthRequestWork[0],
            sizeof(g_AuthRequestWork[0])));

        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
        bool done;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.HasDone(&done));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(done);
    }
    return true;
}
bool TestShopMultiple(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));

    nn::nsd::NasServiceSetting shopSetting;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::nsd::GetNasServiceSetting(&shopSetting, nn::nsd::NasServiceNameOfNxShop));

    {
        nn::account::NintendoAccountAuthorizationRequestContext ctxs[nn::account::nas::NasSessionPoolBase::SessionCountMax];

        for (int i = 0; i < std::extent<decltype(ctxs)>::value; ++i)
        {
            nn::account::NintendoAccountAuthorizationRequestParameters params = {"openid", "a", "b"};
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.CreateNintendoAccountAuthorizationRequest(
                &(ctxs[i]),
                shopSetting.clientId,
                shopSetting.redirectUri.value,
                nn::nsd::NasServiceSetting::RedirectUri::Size,
                params,
                g_AuthRequestWork[i],
                sizeof(g_AuthRequestWork[i])));
        }

        for (int i = 0; i < std::extent<decltype(ctxs)>::value; ++i)
        {
            nn::os::SystemEvent e;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctxs[i].GetSystemEvent(&e));
            e.Wait();
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctxs[i].GetResult());
            bool done;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctxs[i].HasDone(&done));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(done);
        }
    }
    return true;
}
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ_FOR_SHOP

} // ~namespace <anonymous>

TEST(AccountNintendoAccount, ApplicationAuthorization)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        for (auto i = 0; i < naList.Count(); ++ i)
        {
            nnt::account::RevokeAuthorization(naList[i]);

            nn::account::NetworkServiceAccountAdministrator admin;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
                &admin, users[i]));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(LinkNintendoAccount(naList[i], admin));
        }
    }

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(TestBasic(users[i], naList[i]));
    }
#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ_FOR_SHOP)
    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(TestShop(users[i]));
        ASSERT_TRUE(TestShopMultiple(users[i]));
    }
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ_FOR_SHOP
}
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZ

#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZERROR)
namespace {
template <typename ExpectResultType>
bool TestError(const nn::account::Uid& user, const char* error, const char* detail) NN_NOEXCEPT
{
    uint64_t clientId = 0xDEADBEEF;
    const char redirectUri[] = "nintendo://deadbeef.app";

    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NintendoAccountAuthorizationRequestParameters param = {
        "example_scope", // scope
        "example_state", // state
        "example_nonce", // nonce
    };

    nn::account::NetworkServiceAccountAdministrator admin;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
        &admin, user));

    nn::account::NintendoAccountAuthorizationRequestContext ctx;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.DebugCreateNintendoAccountAuthorizationRequest(
        &ctx, clientId, redirectUri, sizeof(redirectUri), param, AuthorizationContextBuffer, sizeof(AuthorizationContextBuffer)));

    // 認可取得
    nn::account::SessionId sessionId;
    ctx.DebugGetInfo(reinterpret_cast<uint64_t*>(&sessionId), sizeof(sessionId) / sizeof(uint64_t));
    nn::account::NintendoAccountApplicationAuthorizationProcedure proc;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(admin.ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(&proc, sessionId));

    nn::account::AsyncContext async;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.PrepareAsync(&async));
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
    e.Wait();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetResult());

    nn::account::RequestUrl request;
    nn::account::CallbackUri callback;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.GetRequest(&request, &callback));
    NN_LOG("Request URL: %s\n", request.url);

    // エラー応答
    auto response = nnt::account::GetAuthorizationForError(request.url, error, detail);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        ExpectResultType,
        proc.ApplyResponse(response.Get<char>(), response.GetSize()));

    return true;
}

template <typename ExpectResultType,  typename ExpectStatusType>
bool TestApplicationAuthorizationError(const nn::account::Uid& user, const nnt::account::NasLoginInfo& loginInfo, const char* error, const char* detail) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(LinkNintendoAccount(loginInfo, admin));
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
    };

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestError<ExpectResultType>(user, error, detail));

    // Availability チェック
    auto r = admin.CheckNetworkServiceAccountAvailability();
    return (r.GetModule() == ExpectStatusType().GetModule()) && (r.GetDescription() == ExpectStatusType().GetDescription());
}
} // ~namespace

TEST(AccountNintendoAccount, ApplicationAuthorizationError)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto na = nnt::account::LoadNaList()[0];
    nn::account::Uid user;
    nnt::account::CreateUsers(&user, 1);

    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    EXPECT_TRUE((TestApplicationAuthorizationError<nn::account::nas::ResultNasAuthorizationStatusAccessDeniedIdTokenHintInvalid, nn::ResultSuccess>(
        user, na, "access_denied", "id_token_hint_invalid")));

    EXPECT_TRUE((TestApplicationAuthorizationError<nn::account::ResultCancelledByUser, nn::ResultSuccess>(
        user, na, "access_denied", "user_terms_agreement_required")));

    EXPECT_TRUE((TestApplicationAuthorizationError<nn::account::ResultNintendoAccountStateDeleted, nn::account::ResultNintendoAccountStateDeleted>(
        user, na, "access_denied", "user_deleted")));

    EXPECT_TRUE((TestApplicationAuthorizationError<nn::account::ResultUserInputDenied, nn::ResultSuccess>(
        user, na, "login_required", "user_different_from_id_token_hint")));

    EXPECT_TRUE((TestApplicationAuthorizationError<nn::account::ResultCancelledByUser, nn::ResultSuccess>(
        user, na, "consent_required", nullptr)));
}
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_AUTHZERROR

#if defined(NNT_ACCOUNT_ENABLE_USER_IMPORT)
namespace {
NN_ALIGNAS(4096) static char g_UserImportWork[nn::account::RequiredBufferSizeForExternalNetworkServiceAccountRegistrar];
bool TestUserImport(
    const nn::account::Uid& user, const nn::account::NetworkServiceAccountId& nsaId, const t::NasLoginInfo& LoginInfo,
    const nn::account::SystemProgramIdentification& programId) NN_NOEXCEPT
{
    nn::account::Uid imported;
    nn::account::NintendoAccountId naId;

    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        nn::account::SessionId sessionId;
        nn::account::ExternalNetworkServiceAccountRegistrar registrar;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CreateExternalNetworkServiceAccountRegistrar(
            &registrar, g_UserImportWork, sizeof(g_UserImportWork)));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.GetOAuthProcedureSessionId(&sessionId));

        {
            nn::account::ExternalNetworkServiceAccountIntroducingProcedure proc;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::ProxyProcedureToIntroduceExternalNetworkServiceAccountForRegistration(&proc, sessionId));

            NN_SDK_LOG("PrepareAsync: begin\n");
            auto begin = nn::os::GetSystemTick();
            nn::account::AsyncContext async;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.PrepareAsync(&async));
            nn::os::SystemEvent e;
            NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
            e.Wait();
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetResult());
            NN_SDK_LOG("PrepareAsync: end: %lld [msec]\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());

            nn::account::RequestUrl requestUrl;
            nn::account::CallbackUri callbackUri;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.GetRequest(&requestUrl, &callbackUri));

            auto response = t::GetAuthorizationViaNasProxy(LoginInfo, requestUrl.url);

            NN_SDK_LOG("ApplyResponseAsync: begin\n");
            begin = nn::os::GetSystemTick();
            nn::account::AsyncContext linkTask;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.ApplyResponseAsync(&linkTask, response.Get<char>(), response.GetSize()));
            nn::os::SystemEvent linkEvent;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkTask.GetSystemEvent(&linkEvent));
            linkEvent.Wait();
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(linkTask.GetResult());
            NN_SDK_LOG("ApplyResponseAsync: end: %lld [msec]\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());

            nn::account::NetworkServiceAccountId loggedIn;
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.GetNetworkServiceAccountId(&loggedIn));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(nsaId, loggedIn);

            size_t imageSize;
            t::Buffer image(1024);
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.GetProfileImage(&imageSize, image.GetAddress(), image.GetSize()));
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, imageSize);

            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(proc.GetLinkedNintendoAccountId(&naId));
        }

        nn::account::NetworkServiceAccountId loggedIn;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.GetNetworkServiceAccountId(&loggedIn));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(nsaId, loggedIn);

        size_t imageSize;
        t::Buffer image(1024);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.GetProfileImage(&imageSize, image.GetAddress(), image.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, imageSize);

        // NSA ID トークン
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.SetSystemProgramIdentification(programId));
        nn::account::AsyncContext ctx;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx));
        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetResult());

        size_t tokenSize;
        t::Buffer token(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.LoadNetworkServiceAccountIdTokenCache(&tokenSize, token.Get<char>(), token.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(tokenSize <= nn::account::NetworkServiceAccountIdTokenLengthMax);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(token.Get<char>(), token.GetSize()) == tokenSize);

        // 永続化
        if (user)
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.RegisterUserAs(user));
            imported = user;
        }
        else
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.RegisterUser(&imported));
        }
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(registrar.RegisterNetworkServiceAccountAsync(&ctx));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
    }

    {
        nn::account::InitializeForSystemService();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        size_t imageSize;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
            nn::account::ResultUserNotExist,
            nn::account::LoadProfileImage(&imageSize, nullptr, 0, imported));

        nn::account::NetworkServiceAccountManager manager;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, imported));

        nn::account::NetworkServiceAccountId registered;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetNetworkServiceAccountId(&registered));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(nsaId, registered);

        // NSA ID トークン
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetSystemProgramIdentification(programId));
        nn::account::AsyncContext ctx;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx));
        nn::os::SystemEvent e;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetResult());

        size_t tokenSize;
        t::Buffer token(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.LoadNetworkServiceAccountIdTokenCache(&tokenSize, token.Get<char>(), token.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(tokenSize <= nn::account::NetworkServiceAccountIdTokenLengthMax);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(token.Get<char>(), token.GetSize()) == tokenSize);

        // NA
        nn::account::NintendoAccountId naId1;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetNintendoAccountId(&naId1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(naId, naId1);
    }
    return true;
} // NOLINT(readability/fn_size)
} // ~namespace <anonymous>
TEST(AccountNintendoAccount, UserImport)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    // NSA ID
    std::unique_ptr<nn::account::NetworkServiceAccountId[]> nsaIds(new nn::account::NetworkServiceAccountId[naList.Count()]);
    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        for (auto i = 0; i < naList.Count(); ++ i)
        {
            nnt::account::RevokeAuthorization(naList[i]);
            nn::account::NetworkServiceAccountAdministrator admin;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[i]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(LinkNintendoAccount(naList[i], admin));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(admin.GetNetworkServiceAccountId(&nsaIds[i]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
        }

        for (auto i = 0; i < naList.Count(); ++i)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::DeleteUser(users[i]));
        }
    }

    nn::account::SystemProgramIdentification programIds[] = {
        {{0x010000000000B121ull}, 0x00},
    };
    // Uid を指定して登録
    for (auto i = 0; i < naList.Count(); ++ i)
    {
        for (const auto& pid : programIds)
        {
            ASSERT_TRUE(TestUserImport(users[i], nsaIds[i], naList[i], pid));
        }
    }
    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        // ユーザーアカウントを削除
        for (auto i = 0; i < naList.Count(); ++i)
        {
            {
                nn::account::NetworkServiceAccountAdministrator admin;
                NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, users[i]));
                NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::UnregisterNetworkServiceAccount(admin));
            }
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::DeleteUser(users[i]));
        }
    }
    // Uid を指定せず登録
    for (auto i = 0; i < naList.Count(); ++ i)
    {
        for (const auto& pid : programIds)
        {
            ASSERT_TRUE(TestUserImport(nn::account::InvalidUid, nsaIds[i], naList[i], pid));
        }
    }
}
#endif // NNT_ACCOUNT_ENABLE_USER_IMPORT


#if defined(NNT_ACCOUNT_ENABLE_NSA_ID_TOKEN)
namespace {
bool TestNetworkServiceAccountIdToken(const nn::account::Uid& user, const nn::account::SystemProgramIdentification& programId) NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.SetSystemProgramIdentification(programId));

    nn::account::AsyncContext ctx;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx));
    nn::os::SystemEvent e;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
    bool done;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(ctx.HasDone(&done));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(done);

    size_t tokenSize;
    t::Buffer token(nn::account::NetworkServiceAccountIdTokenLengthMax + 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.LoadNetworkServiceAccountIdTokenCache(&tokenSize, token.Get<char>(), token.GetSize()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(tokenSize <= nn::account::NetworkServiceAccountIdTokenLengthMax);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(strnlen(token.Get<char>(), token.GetSize()) == tokenSize);
    return true;
}
} // ~namespace <anonymous>
TEST(AccountNintendoAccount, NsaAuthentication)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        for (auto i = 0; i < naList.Count(); ++ i)
        {
            nnt::account::RevokeAuthorization(naList[i]);

            nn::account::NetworkServiceAccountAdministrator admin;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
                &admin, users[i]));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(LinkNintendoAccount(naList[i], admin));
        }
    }
    nn::account::SystemProgramIdentification programIds[] = {
        {{0x010000000000B121ull}, 0x00},
    };

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        for (const auto& pid : programIds)
        {
            ASSERT_TRUE(TestNetworkServiceAccountIdToken(users[i], pid));
        }
    }
}
#endif // NNT_ACCOUNT_ENABLE_NSA_ID_TOKEN
