﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>

#include <cctype>
#include <cstring>

#include <nn/nn_ApplicationId.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

namespace {
// 16進数文字の整数変換
inline uint8_t ConvertHexToInteger(char c) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(std::isxdigit(c));
    return std::isdigit(c)
        ? static_cast<uint8_t>(c - '0')
        : static_cast<uint8_t>(std::tolower(c) - 'a' + 10);
}

// 16進数文字列の整数変換
template <typename T>
inline T ExtractHexadecimal(const char* pStr, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT(length == sizeof(T) * 2);
    NN_UNUSED(length);
    T out = 0;
    for (auto i = 0u; i < sizeof(T) * 2; ++ i)
    {
        out = (out << 4 | ConvertHexToInteger(*(pStr ++)));
    }
    return out;
}

// nn::ae のメッセージ処理
bool ExpectMessage(nn::ae::Message expectMessage, nn::os::SystemEventType& e) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto message = nn::ae::WaitForNotificationMessage(&e);
        if (message == expectMessage)
        {
            return true;
        }

        NN_LOG("[account:Launcher] Unexpected message 0x%08x (expect=0x%08x)\n", message, expectMessage);
        if (message <= nn::ae::Message_ApplicationExited)
        {
            // アプリケーション起動と関係する予期しないメッセージが来たら失敗
            break;
        }
    }
    return false;
}

void WaitApplicationExit(nn::os::SystemEventType& e) NN_NOEXCEPT
{
    nn::Bit32 messageFlags = 0;
    while (messageFlags != 0x3)
    {
        auto message = nn::ae::WaitForNotificationMessage(&e);
        if (message == nn::ae::Message_ChangeIntoForeground)
        {
            nn::ae::AcquireForegroundRights();
            messageFlags |= 0x1;
        }
        else if (message == nn::ae::Message_ApplicationExited)
        {
            messageFlags |= 0x2;
        }
    }
}

char g_ApplicationControlBuffer[1024 * 1024];
const size_t StorageSizeForAccount = 1024;
nn::Result SetupApplicationHandle(nn::account::Uid* pOutUid, const nn::ae::ApplicationHandle& handle, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    nn::ns::Initialize();

    // アプリケーション管理属性の取得
    size_t outSize;
    NN_RESULT_DO(nn::ns::GetApplicationControlData(
        &outSize, g_ApplicationControlBuffer, sizeof(g_ApplicationControlBuffer),
        nn::ns::ApplicationControlSource::Storage, applicationId));
    auto& property = nn::ns::ApplicationControlDataAccessor(g_ApplicationControlBuffer, outSize).GetProperty();

    // 起動時のユーザーアカウント選択フラグ
    switch (property.startupUserAccount)
    {
    case nn::ns::StartupUserAccount::None:
        *pOutUid = nn::account::InvalidUid;
        break;
    case nn::ns::StartupUserAccount::Required:
    case nn::ns::StartupUserAccount::RequiredWithNetworkServiceAccountAvailable:
        {
            // 使用するユーザーアカウントの決定
            int count;
            nn::account::Uid users[nn::account::UserCountMax];
            NN_RESULT_DO(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
            NN_ABORT_UNLESS(count >= 1, "[account:Launcher] requires 1 or more user accounts\n");
            auto user = users[count >= 2? 1: 0];
            auto info = nn::account::MakePreselectionInfo(user);

            // LaunchParameter の設定
            nn::applet::StorageHandle storageHandle;
            NN_RESULT_DO(CreateStorage(&storageHandle, StorageSizeForAccount));
            bool releaseRequired = true;
            NN_UTIL_SCOPE_EXIT
            {
                if (releaseRequired)
                {
                    nn::applet::ReleaseStorage(storageHandle);
                }
            };
            NN_RESULT_DO(WriteToStorage(storageHandle, 0, &info, sizeof(info)));
            releaseRequired = false;
            nn::ae::PushApplicationLaunchParameter(handle, nn::applet::LaunchParameterKind_Account, storageHandle);

            *pOutUid = user;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

nn::ncm::ApplicationId g_ApplicationId;
void RunApplication(nn::ae::SystemAppletParameters* param) NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_LOG("[account:Launcher] Launching application: 0x%016llx\n", g_ApplicationId.value);

    // SA の起動
    nn::os::SystemEventType e;
    nn::ae::InitializeNotificationMessageEvent(&e);
    NN_ABORT_UNLESS(ExpectMessage(nn::ae::Message_ChangeIntoForeground, e));

    // アプリの起動準備
    nn::ae::ApplicationHandle handle = nn::ae::CreateApplication(g_ApplicationId);
    NN_UTIL_SCOPE_EXIT
    {
        nn::ae::CloseApplicationHandle(handle);
    };

    // 起動時引数の作成
    nn::account::Uid preselected;
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetupApplicationHandle(&preselected, handle, g_ApplicationId));
    if (preselected)
    {
        NN_LOG(
            "[account:Launcher] User %08x_%08x_%08x_%08x is preselected for the application\n",
            static_cast<uint32_t>(preselected._data[0] >> 32),
            static_cast<uint32_t>(preselected._data[0] & 0xFFFFFFFFull),
            static_cast<uint32_t>(preselected._data[1] >> 32),
            static_cast<uint32_t>(preselected._data[1] & 0xFFFFFFFFull));
    }

    // アプリの開始
    nn::ae::StartApplication(handle);
    nn::ae::RequestApplicationGetForeground(handle);
    NN_ABORT_UNLESS(ExpectMessage(nn::ae::Message_ChangeIntoBackground, e));
    nn::ae::ReleaseForegroundRights();

    // アプリの終了待ち
    WaitApplicationExit(e);

    if (preselected)
    {
        nn::ApplicationId id;
        uint32_t version;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserLastOpenedApplication(&id, &version, preselected));
        NN_LOG("[account:Launcher] nn::account::GetUserLastOpenedApplication() returned 0x%016llx\n", id.value);
        NN_ABORT_UNLESS(id.value == g_ApplicationId.value);
    }
    NN_LOG("[account:Launcher] Application 0x%016llx exited\n", g_ApplicationId.value);
}
} // namespace

extern "C" void nninitStartup()
{
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    // 引数の解析
    NN_ABORT_UNLESS(argc > 1);
    auto arg = argv[1];
    auto vstring = std::strncmp(arg, "0x", 2) == 0? arg + 2: arg;
    g_ApplicationId.value = ExtractHexadecimal<uint64_t>(vstring, strlen(vstring));

    // SystemApplet としてアプリを起動する
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, RunApplication);
}
