﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nas/account_NasUserAdaptor.h"

#include "detail/account_CacheUtil.h"
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/account_ResultPrivate.h>

#include "testAccount_RapidJsonInputStream.h"
#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

namespace a = nn::account;
namespace t = nnt::account;

namespace nasUserResource {
const char* Inputs[] = {
    // 0
    R"({"analyticsPermissions":{
    "targetMarketing":{"updatedAt":1521677487,"permitted":false},
    "internalAnalysis":{"permitted":false,"updatedAt":1521677487}},
    "nickname":"naus1",
    "isChild":false,
    "analyticsOptedIn":false,
    "country":"US",
    "links":{},
    "emailOptedInUpdatedAt":1464154161,
    "createdAt":1464154161,
    "gender":"female",
    "language":"en-US",
    "loginId":null,
    "emailVerified":false,
    "email":null,
    "id":"a90947558e5ec9b9",
    "emailOptedIn":false,
    "updatedAt":1464154161,
    "region":null,
    "birthday":"1998-04-05",
    "timezone":{"id":"America/New_York"},
    "analyticsOptedInUpdatedAt":1464154161,
    "screenName":"ue***@e****"})",

    // 1
    R"({"isChild":true,
    "language":"ja-JP",
    "loginId":"ueyama_yoshitonaus1",
    "emailOptedInUpdatedAt":1464154161,
    "createdAt":1464154161,
    "gender":"male",
    "analyticsOptedIn":true,
    "emailVerified":true,
    "screenName":"ue***@e****",
    "nickname":"naus1",
    "email":"ueyama_yoshito\u002bnaus1@exmx.nintendo.co.jp",
    "updatedAt":1464154161,
    "region":"ZZ",
    "timezone":{"id":"America/New_York"},
    "analyticsOptedInUpdatedAt":1464154161,
    "country":"US",
    "links":{
        "facebook":{},
        "twitter":{},
        "nintendoNetwork":{"createdAt":1487290181,"id":"1799588070"},
        "google":{}
    },
    "birthday":"1998-04-05",
    "id":"a90947558e5ec9b9",
    "emailOptedIn":true,
    "analyticsPermissions":{
    "internalAnalysis":{"updatedAt":1521677487,"permitted":false},
    "targetMarketing":{"permitted":true,"updatedAt":1521677487}}})",

    // 2
    R"({"nickname":"naus1",
    "links":{
        "facebook":{"id":"110700126051296"},
        "nintendoNetwork":{"createdAt":1487290181},
        "google":{"createdAt":1487290181},
        "twitter":{"id":"767516344409464832","createdAt":1487293522}
    },
    "isChild":false,
    "analyticsOptedIn":true,
    "country":"US",
    "emailOptedInUpdatedAt":1464154161,
    "createdAt":1464154161,
    "gender":"female",
    "analyticsPermissions":{
    "internalAnalysis":{"updatedAt":1521677487,"permitted":true},
    "targetMarketing":{"permitted":false,"updatedAt":1521677487}},
    "language":"en-US",
    "loginId":null,
    "emailVerified":false,
    "email":null,
    "id":"a90947558e5ec9b9",
    "emailOptedIn":false,
    "updatedAt":1464154161,
    "region":null,
    "birthday":"1998-04-05",
    "timezone":{"id":"America/New_York"},
    "analyticsOptedInUpdatedAt":1464154161,
    "screenName":"ue***@e****"})",

    // 3
    R"({"isChild":true,
    "language":"ja-JP",
    "loginId":"ueyama_yoshitonaus1",
    "emailOptedInUpdatedAt":1464154161,
    "createdAt":1464154161,
    "gender":"male",
    "analyticsOptedIn":true,
    "emailVerified":true,
    "screenName":"ue***@e****",
    "nickname":"naus1",
    "email":"ueyama_yoshito\u002bnaus1@exmx.nintendo.co.jp",
    "updatedAt":1464154161,
    "analyticsPermissions":{
    "internalAnalysis":{"updatedAt":1521677487,"permitted":true},
    "targetMarketing":{"permitted":true,"updatedAt":1521677487}},
    "region":"ZZ",
    "timezone":{"id":"America/New_York"},
    "analyticsOptedInUpdatedAt":1464154161,
    "country":"US",
    "birthday":"1998-04-05",
    "id":"a90947558e5ec9b9",
    "links":{
        "twitter":{"id":"767516344409464832"},
        "facebook":{"createdAt":1487293509,"id":"110700126051296"},
        "nintendoNetwork":{},
        "google":{"createdAt":1487290181}
    },
    "emailOptedIn":true})",

    // 4
    R"({"nickname":"naus1",
    "isChild":false,
    "analyticsOptedIn":true,
    "country":"US",
    "emailOptedInUpdatedAt":1464154161,
    "createdAt":1464154161,
    "gender":"female",
    "links":{
        "twitter":{"id":"767516344409464832","createdAt":1487293522},
        "facebook":{"createdAt":1487293509,"id":"110700126051296"},
        "google":{"createdAt":1487290181,"id":"1799588070"},
        "nintendoNetwork":{"createdAt":1487293513,"id":"104634805069424243428"}
    },
    "language":"en-US",
    "loginId":null,
    "emailVerified":false,
    "email":null,
    "analyticsPermissions":{
    "internalAnalysis":{"permitted":false,"updatedAt":1521677487},
    "targetMarketing":{"updatedAt":1521677487,"permitted":false}},
    "id":"a90947558e5ec9b9",
    "emailOptedIn":false,
    "updatedAt":1464154161,
    "region":null,
    "birthday":"1998-04-05",
    "timezone":{"id":"America/New_York"},
    "analyticsOptedInUpdatedAt":1464154161,
    "screenName":"ue***@e****"})",
};

const struct Expect
{
    a::NintendoAccountId id;
    a::nas::NasUserBase base;
} Expects[] = {
    {
        {0xa90947558e5ec9b9ull},
        {
            false, // isChild
            "ue***@e****", // screenName
            "",
            "", // loginId
            "naus1", // nickname
            "1998-04-05",
            "female",
            false, // analyticsOptedIn
            false, // analyticsForTargetMarketingPermitted
            false, // analyticsForInternalAnalysisPermitted
            "en-US",
            "US",
            "",
            "America/New_York",
            false, // isNnidLinked
            false, // isTwitterLinked
            false, // isFacebookLinked
            false, // isGoogleLinked
        },
    },
    {
        {0xa90947558e5ec9b9ull},
        {
            true, // isChild
            "ue***@e****", // screenName
            "ueyama_yoshito+naus1@exmx.nintendo.co.jp",
            "ueyama_yoshitonaus1", // loginId
            "naus1", // nickname
            "1998-04-05",
            "male",
            true, // analyticsOptedIn
            true, // analyticsForTargetMarketingPermitted
            false, // analyticsForInternalAnalysisPermitted
            "ja-JP",
            "US",
            "ZZ",
            "America/New_York",
            true, // isNnidLinked
            false, // isTwitterLinked
            false, // isFacebookLinked
            false, // isGoogleLinked
        },
    },
    {
        {0xa90947558e5ec9b9ull},
        {
            false, // isChild
            "ue***@e****", // screenName
            "",
            "", // loginId
            "naus1", // nickname
            "1998-04-05",
            "female",
            true, // analyticsOptedIn
            false, // analyticsForTargetMarketingPermitted
            true, // analyticsForInternalAnalysisPermitted
            "en-US",
            "US",
            "",
            "America/New_York",
            false, // isNnidLinked
            true, // isTwitterLinked
            false, // isFacebookLinked
            false, // isGoogleLinked
        },
    },
    {
        {0xa90947558e5ec9b9ull},
        {
            true, // isChild
            "ue***@e****", // screenName
            "ueyama_yoshito+naus1@exmx.nintendo.co.jp",
            "ueyama_yoshitonaus1", // loginId
            "naus1", // nickname
            "1998-04-05",
            "male",
            true, // analyticsOptedIn
            true, // analyticsForTargetMarketingPermitted
            true, // analyticsForInternalAnalysisPermitted
            "ja-JP",
            "US",
            "ZZ",
            "America/New_York",
            false, // isNnidLinked
            false, // isTwitterLinked
            true, // isFacebookLinked
            false, // isGoogleLinked
        },
    },
    {
        {0xa90947558e5ec9b9ull},
        {
            false, // isChild
            "ue***@e****", // screenName
            "",
            "", // loginId
            "naus1", // nickname
            "1998-04-05",
            "female",
            false, // analyticsOptedIn
            false, // analyticsForTargetMarketingPermitted
            false, // analyticsForInternalAnalysisPermitted
            "en-US",
            "US",
            "",
            "America/New_York",
            true, // isNnidLinked
            true, // isTwitterLinked
            true, // isFacebookLinked
            true, // isGoogleLinked
        },
    },
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    // バッファ類
    t::Buffer inputBuffer(1024);
    t::Buffer outputBuffer(1024);

    //
    a::nas::NasUserResourceCache cache;
    cache.Initialize(s);

    // json -> user resource cache (json)
    a::NintendoAccountId naId;
    {
        t::MemoryInputStreamForRapidJson stream;
        stream.Set(json, strlen(json));
        a::nas::NasUserAdaptor adaptor(s, outputBuffer.Get<char>(), outputBuffer.GetSize());
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::json::ImportJsonByRapidJson(adaptor, stream, nullptr));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.PullUserResourceCache(&naId, cache));
    }

    // user resource cache (json) -> user base
    a::nas::NasUserBase base;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Load(&base, naId, inputBuffer.GetAddress(), inputBuffer.GetSize()));

    EXPECT_EQ(expect.id, naId);
    EXPECT_EQ(expect.base.isChild, base.isChild);
    EXPECT_EQ(0, std::strncmp(expect.base.screenName, base.screenName, sizeof(expect.base.email)));
    EXPECT_EQ(0, std::strncmp(expect.base.email, base.email, sizeof(expect.base.email)));
    EXPECT_EQ(0, std::strncmp(expect.base.loginId, base.loginId, sizeof(expect.base.loginId)));
    EXPECT_EQ(0, std::strncmp(expect.base.nickname, base.nickname, sizeof(expect.base.nickname)));
    EXPECT_EQ(0, std::strncmp(expect.base.birthday, base.birthday, sizeof(expect.base.birthday)));
    EXPECT_EQ(0, std::strncmp(expect.base.gender, base.gender, sizeof(expect.base.gender)));
    EXPECT_EQ(expect.base.analyticsOptedIn, base.analyticsOptedIn);
    EXPECT_EQ(expect.base.analyticsForTargetMarketingPermitted, base.analyticsForTargetMarketingPermitted);
    EXPECT_EQ(expect.base.analyticsForInternalAnalysisPermitted, base.analyticsForInternalAnalysisPermitted);
    EXPECT_EQ(0, std::strncmp(expect.base.language, base.language, sizeof(expect.base.language)));
    EXPECT_EQ(0, std::strncmp(expect.base.country, base.country, sizeof(expect.base.country)));
    EXPECT_EQ(0, std::strncmp(expect.base.region, base.region, sizeof(expect.base.region)));
    EXPECT_EQ(0, std::strncmp(expect.base.timezone, base.timezone, sizeof(expect.base.timezone)));
}
} // ~namespace userResource

TEST(AccountAdaptor, NasUserResource)
{
    for (int i = 0; i < sizeof(nasUserResource::Expects) / sizeof(nasUserResource::Expects[0]); ++ i)
    {
        nasUserResource::RunTest(
            nasUserResource::Expects[i],
            nasUserResource::Inputs[i]);
    }
}

namespace compatibility {

const char* Inputs[] = {
    // SDK 6.0.0 以前のバージョン
    R"({
        "isChild":false,
        "screenName":"foo",
        "email":"foo@example.com",
        "loginId":"bar",
        "nickname":"fizz",
        "birthday":"1980-01-01",
        "gender":"male",
        "emailOptedIn":false,
        "analyticsOptedIn":true,
        "language":"ja-JP",
        "country":"JP",
        "timezone":"America/New_York",
        "isNnLinked":false,
        "isTwitterLinked":false,
        "isFacebookLinked":false,
        "isGoogleLinked":false
    })",
};

const struct Expect
{
    a::NintendoAccountId id;
    a::nas::NasUserBase base;
} Expects[] = {
    {
        {0xa90947558e5ec9b9ull},
        {
            false, // isChild
            "foo", // screenName
            "foo@example.com",
            "bar", // loginId
            "fizz", // nickname
            "1980-01-01",
            "male",
            true, // analyticsOptedIn
            false, // analyticsForTargetMarketingPermitted
            false, // analyticsForInternalAnalysisPermitted
            "ja-JP",
            "JP",
            "",
            "America/New_York",
            false, // isNnidLinked
            false, // isTwitterLinked
            false, // isFacebookLinked
            false, // isGoogleLinked
        },
    },
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    // バッファ類
    t::Buffer inputBuffer(1024);

    //
    auto naId = expect.id;
    a::nas::NasUserResourceCache cache;
    cache.Initialize(s);

    a::detail::Uuid cacheId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, json, strlen(json), s));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Store(naId, std::move(cacheId)));

    // user resource cache (json) -> user base
    a::nas::NasUserBase base;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Load(&base, naId, inputBuffer.GetAddress(), inputBuffer.GetSize()));

    EXPECT_EQ(expect.base.isChild, base.isChild);
    EXPECT_EQ(0, std::strncmp(expect.base.screenName, base.screenName, sizeof(expect.base.email)));
    EXPECT_EQ(0, std::strncmp(expect.base.email, base.email, sizeof(expect.base.email)));
    EXPECT_EQ(0, std::strncmp(expect.base.loginId, base.loginId, sizeof(expect.base.loginId)));
    EXPECT_EQ(0, std::strncmp(expect.base.nickname, base.nickname, sizeof(expect.base.nickname)));
    EXPECT_EQ(0, std::strncmp(expect.base.birthday, base.birthday, sizeof(expect.base.birthday)));
    EXPECT_EQ(0, std::strncmp(expect.base.gender, base.gender, sizeof(expect.base.gender)));
    EXPECT_EQ(expect.base.analyticsOptedIn, base.analyticsOptedIn);
    EXPECT_EQ(expect.base.analyticsForTargetMarketingPermitted, base.analyticsForTargetMarketingPermitted);
    EXPECT_EQ(expect.base.analyticsForInternalAnalysisPermitted, base.analyticsForInternalAnalysisPermitted);
    EXPECT_EQ(0, std::strncmp(expect.base.language, base.language, sizeof(expect.base.language)));
    EXPECT_EQ(0, std::strncmp(expect.base.country, base.country, sizeof(expect.base.country)));
    EXPECT_EQ(0, std::strncmp(expect.base.region, base.region, sizeof(expect.base.region)));
    EXPECT_EQ(0, std::strncmp(expect.base.timezone, base.timezone, sizeof(expect.base.timezone)));
}
} // ~namespace compatibility

TEST(AccountAdaptor, NasUserResourceCompatibility)
{
    for (int i = 0; i < sizeof(compatibility::Expects) / sizeof(compatibility::Expects[0]); ++ i)
    {
        compatibility::RunTest(
            compatibility::Expects[i],
            compatibility::Inputs[i]);
    }
}
