﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nas/account_NasOp2Adaptor.h"

#include "detail/account_CacheUtil.h"
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/nas/account_ResultForNas.h>
#include <nn/account/http/account_ResultForHttp.h>

#include "testAccount_RapidJsonInputStream.h"
#include "testAccount_Util.h"

namespace a = nn::account;
namespace t = nnt::account;


namespace membership {

const char* Inputs[] = {
    "{\"user_id\":\"1234567890abcdef\",\"membership\":{\"active\":true,\"expires_at\":1234567890}}",
    "{\"membership\":{\"active\":false},\"user_id\":\"1234567890abcdef\"}",
};

struct Expect
{
    a::NetworkServiceLicense license;
    nn::time::PosixTime expiration;
} Expects[] = {
    {a::NetworkServiceLicense_Common, {1234567}}, // time::PosixTime は sec 単位, サーバーの応答は msec 単位
    {a::NetworkServiceLicense_None, {}},
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    a::nas::NasOp2MembershipAdaptor adaptor;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    a::NetworkServiceLicense license;
    nn::time::PosixTime expiration;
    adaptor.GetMembership(&license, &expiration);
    EXPECT_EQ(expect.license, license);
    if (expect.license != a::NetworkServiceLicense_None)
    {
        EXPECT_EQ(expect.expiration.value, expiration.value);
    }
}

void RunTests() NN_NOEXCEPT
{
    for (int i = 0; i < std::extent<decltype(Inputs)>::value; ++ i)
    {
        RunTest(Expects[i], Inputs[i]);
    }
}

template <typename ResultType>
void TestErrorCase(const char* json, int httpStatus) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    a::nas::NasOp2MembershipAdaptor adaptor;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    EXPECT_TRUE(ResultType::Includes(adaptor.Adapt(httpStatus)));
}

void RunErrorTests() NN_NOEXCEPT
{
    TestErrorCase<a::nas::ResultNasOp2Status400>(
        "{\"user_id\":\"1234567890abcdef\",\"membership\":{\"active\":true,\"expires_at\":1234567890}}", 400);
    TestErrorCase<a::nas::ResultNasOp2Status403>(
        "{\"blah\":\"1234567890abcdef\"}", 403);
    TestErrorCase<a::nas::ResultNasOp2Status503>(
        "{\"blah\":\"1234567890abcdef\"}", 503);
    TestErrorCase<a::http::ResultHttpStatus504GatewayTimeout>(
        "{\"membership\":{\"active\":false},\"user_id\":\"1234567890abcdef\"}", 504);
}

} // ~namespace membership

TEST(AccountAdaptor, NasOp2Membership)
{
    membership::RunTests();
    membership::RunErrorTests();
}
