﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "baas/account_BaasLoginAdaptor.h"

#include "detail/account_UuidUtil.h"
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/json/account_RapidJsonStream.h>
#include <nn/account/account_NintendoAccountTypes.h>

#include "testAccount_Mounter.h"
#include "testAccount_Util.h"
#include "testAccount_JsonPrintAdaptor.h"
#include "testAccount_RapidJsonInputStream.h"

#include <nnt/nntest.h>

#include <nn/nn_Log.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_BAAS_CLIENT_LOGIN
#define NNT_ACCOUNT_ENABLE_BAAS_USER_LOGIN
#define NNT_ACCOUNT_ENABLE_BAAS_CLIENT_LOGIN_EXPIRATION
#define NNT_ACCOUNT_ENABLE_BAAS_USER_LOGIN_EXPIRATION

#if defined(NNT_ACCOUNT_ENABLE_BAAS_CLIENT_LOGIN)
namespace baasClientLogin
{

const char Input[] =
    "{\"accessToken\":\"eyJraWQiOiJlYmFmZTAyYy0yMDlkLTRkYjQtODI0My1iMWEwZjA4ZTFhZ"
    "jEiLCJhbGciOiJSUzI1NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwY"
    "TUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20vMS4wLjAvaW50ZXJuYWxfY2VydGlmaWNhdGVzI"
    "n0.eyJ0eXAiOiJ0b2tlbiIsImJzOnN0cyI6WzM4NSwwLDAsMCwwLDAsMCwwLDAsMCwwLDAsMCwwL"
    "DAsMF0sImlhdCI6MTQ1ODI2ODkwMCwiZXhwIjoxNDU4Mjc5NzAwLCJiczpncnQiOjEsImp0aSI6I"
    "jA4ODg5MDViLWQ0MmYtNDViZi1iMzM5LTgyOTUxMDBlZDUxOSIsInN1YiI6ImI0MzI3MWU0MmFmM"
    "zFlM2YiLCJpc3MiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiL"
    "mJhYXMubmludGVuZG8uY29tIiwiYXVkIjoiYjQzMjcxZTQyYWYzMWUzZiJ9.h0qyffHG7aD8_2Fq"
    "OoemS9cgpMeO3D2N37aYFhn08JKfFycFtXN9m-jpAlfhLuj9eSjcrhbhbci9m9rS_dIHSkGNg6eF"
    "siugwNw5WtlZ_VCkPTiImwRgbal2C0XF1x6_QFTz9Pui25acNn1PeiL2gfKxEG78Fxev3eHD_tjU"
    "7TtHeVn0kMCMoJBLEueNoUb2gsJYHkf3haO8soiq48y264eKyxvEFo_SLGNPVDfIZVoA3Y5cMXLK"
    "vDvB26X5P6bauOTeK48epbvZXdLIBBAEQz4e8J9K99VSNd0ZLhNNWxtl40-nfFZ21JZPO77hJ3lp"
    "AJUV9zu2kipBaBgRmk620Q\",\"tokenType\":\"Bearer\",\"expiresIn\":10800}";
const char Expected[] =
    "eyJraWQiOiJlYmFmZTAyYy0yMDlkLTRkYjQtODI0My1iMWEwZjA4ZTFhZjEiLCJhbGciOiJSUzI1"
    "NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFh"
    "cy5uaW50ZW5kby5jb20vMS4wLjAvaW50ZXJuYWxfY2VydGlmaWNhdGVzIn0.eyJ0eXAiOiJ0b2tl"
    "biIsImJzOnN0cyI6WzM4NSwwLDAsMCwwLDAsMCwwLDAsMCwwLDAsMCwwLDAsMF0sImlhdCI6MTQ1"
    "ODI2ODkwMCwiZXhwIjoxNDU4Mjc5NzAwLCJiczpncnQiOjEsImp0aSI6IjA4ODg5MDViLWQ0MmYt"
    "NDViZi1iMzM5LTgyOTUxMDBlZDUxOSIsInN1YiI6ImI0MzI3MWU0MmFmMzFlM2YiLCJpc3MiOiJo"
    "dHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMubmludGVuZG8u"
    "Y29tIiwiYXVkIjoiYjQzMjcxZTQyYWYzMWUzZiJ9.h0qyffHG7aD8_2FqOoemS9cgpMeO3D2N37a"
    "YFhn08JKfFycFtXN9m-jpAlfhLuj9eSjcrhbhbci9m9rS_dIHSkGNg6eFsiugwNw5WtlZ_VCkPTi"
    "ImwRgbal2C0XF1x6_QFTz9Pui25acNn1PeiL2gfKxEG78Fxev3eHD_tjU7TtHeVn0kMCMoJBLEue"
    "NoUb2gsJYHkf3haO8soiq48y264eKyxvEFo_SLGNPVDfIZVoA3Y5cMXLKvDvB26X5P6bauOTeK48"
    "epbvZXdLIBBAEQz4e8J9K99VSNd0ZLhNNWxtl40-nfFZ21JZPO77hJ3lpAJUV9zu2kipBaBgRmk6"
    "20Q";

void RunTest(const char* token, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::baas::ClientAccessTokenCache cache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(cache.Initialize(s));

    ASSERT_FALSE(cache.IsAvailable());

    a::baas::ClientLoginAdaptor adaptor(cache);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    ASSERT_TRUE(cache.IsAvailable());

    t::Buffer buffer(1024 * 8);
    size_t sizeActual;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(cache.Load(&sizeActual, buffer.Get<char>(), buffer.GetSize()));
    ASSERT_EQ(strlen(token), sizeActual);
    ASSERT_EQ(0, std::memcmp(token, buffer.GetAddress(), sizeActual));
}

} // ~namespace baasClientLogin

TEST(AccountAdaptor, BaasClientLogin)
{
    baasClientLogin::RunTest(
        baasClientLogin::Expected,
        baasClientLogin::Input);
}
#endif // NNT_ACCOUNT_ENABLE_BAAS_CLIENT_LOGIN

#if defined(NNT_ACCOUNT_ENABLE_BAAS_USER_LOGIN)
namespace baasUserLogin
{
// Application Authentication Token なし版
const char Input[] =
    "{\"user\":{\"country\":\"\",\"etag\":\"\\\"e6c98dcb6cc931e6-698679167c0089e7"
    "\\\"\",\"thumbnailUrl\":\"\",\"friendsUpdatedAt\":1458268901,\"blocksUpdated"
    "At\":1458268901,\"createdAt\":1458268901,\"links\":{},\"extras\":{\"self\":{"
    "},\"favoriteFriends\":{},\"foaf\":{},\"friends\":{},\"everyone\":{}},\"updat"
    "edAt\":1458268901,\"presence\":{\"extras\":{\"everyone\":{},\"self\":{},\"fa"
    "voriteFriends\":{},\"friends\":{},\"foaf\":{}},\"updatedAt\":1458268901,\"st"
    "ate\":\"OFFLINE\"},\"id\":\"e6c98dcb6cc931e6\",\"birthday\":\"0000-00-00\",\""
    "permissions\":{\"friendRequestReception\":true,\"personalAnalytics\":true,\""
    "personalNotificationUpdatedAt\":1458268901,\"personalNotification\":true,\"p"
    "ersonalAnalyticsUpdatedAt\":1458268901,\"friends\":\"EVERYONE\",\"presence\""
    ":\"SELF\"},\"nickname\":\"\",\"deviceAccounts\":[{\"id\":\"02441c80ed579bf1\""
    "}]},\"accessToken\":\"eyJhbGciOiJSUzI1NiIsImtpZCI6ImViYWZlMDJjLTIwOWQtNGRiNC"
    "04MjQzLWIxYTBmMDhlMWFmMSIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkOD"
    "IwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20vMS4wLjAvaW50ZXJuYWxfY2VydGlmaWNhdG"
    "VzIn0.eyJqdGkiOiJhNTg5NTdkMC0yODBjLTQ2YjQtODlkMS0zZTFlNTRmYzA5YTYiLCJleHAiOj"
    "E0NTgyNzk3MDIsImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OG"
    "Etc2IuYmFhcy5uaW50ZW5kby5jb20iLCJiczpncnQiOjIsInN1YiI6ImU2Yzk4ZGNiNmNjOTMxZT"
    "YiLCJiczpzdHMiOlsxMDQxNTY4NjUzNDg0NTQ4Niw0Mjk4MzUwMDAzMiwxMDk5NTYxOTY3NjE3LD"
    "Q2NDg2NjUwODgsODU5MDAwMDEyOSwyMzgxOTk1NzM0MzA5MzQ5MSwwLDAsMCwwLDAsMCwwLDAsMC"
    "wwXSwiaWF0IjoxNDU4MjY4OTAyLCJhdWQiOiJiNDMyNzFlNDJhZjMxZTNmIiwidHlwIjoidG9rZW"
    "4ifQ.WKOkPaFjO35dvB1RXdmgDG2EqD9L5yZWmCf7-W-fvN5nGasHLMEnPx6Us0TzkJcQs_hC9ZM"
    "e9hiX17w5I1kwb49ZKGcPb4Sc-gs81BPu00zKm58NFPUUwvuOFbwEIZiltV88Q__57OhNTnsnfEC"
    "XwKSA1gKnOUsfwydUbn9zFPHXUbpxMBIAOl0L1ISwA25x1lIGQNV6vo5XZiP5MQgNZR-bVoXC0N-"
    "Xrpsdv-tSDYWbegdgXAaPLure0m089WHnN1bJybWoHUqd5jHl8KfZY5UNH0iOVz6XN1gasQvJ9CP"
    "lZxFRajcSHX_8vi8JE_IHKB_BXJQlw5X3-8ISpTDtOw\",\"expiresIn\":10800,\"idToken\""
    ":\"eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhY"
    "XMubmludGVuZG8uY29tLzEuMC4wL2NlcnRpZmljYXRlcyIsImtpZCI6IjgwYzRjZGQxLTJlMzMtN"
    "DUzNi1iZTFlLWUwODdjYmU1NTc4ZCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJiNDMyNzFlNDJhZ"
    "jMxZTNmIiwiaWF0IjoxNDU4MjY4OTAyLCJzdWIiOiJlNmM5OGRjYjZjYzkzMWU2IiwiaXNzIjoia"
    "HR0cHM6Ly9lMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvL"
    "mNvbSIsInR5cCI6ImlkX3Rva2VuIiwiYnM6ZGlkIjoiMDI0NDFjODBlZDU3OWJmMSIsImp0aSI6I"
    "jIxYjQyNDdmLTA3OWQtNDZlMC04ZjZiLWRhZWFhMzRmNDI5OCIsImV4cCI6MTQ1ODI3OTcwMn0.U"
    "s-sVwXV9AtKIfrufjV4XP4d_MLmEbv8ryEFt6WniRoPKapiZSQLBBmwflG75wVWp8PD70oE2AvWg"
    "DmNePHc6emz2xGdvmydVPy3lP6yzbf5YxuAUQNAQaUkvZb8bsDU0t7Ah3M_zQCHnWG3YgQdkh0a9"
    "juri9Ybsdrmnizgpq_uJqcEebTtShdkHzvgO9ECXH5OSY_emR8dquFHEtKmWFuK0GxWLM6e_cZAr"
    "G3V5oIZzb9qh0d1hbe9y4p6BZI6nYApV0LDcb-guOUAHVrMR49ifa_zpc-HcTi8Aw-7S8tSTKOP5"
    "vqqBH9rtJUIMkHEgG8vI2oz6sISKYlWhSSbzg\",\"tokenType\":\"Bearer\"}";
const char ExpectedAccessToken[] =
    "eyJhbGciOiJSUzI1NiIsImtpZCI6ImViYWZlMDJjLTIwOWQtNGRiNC04MjQzLWIxYTBmMDhlMWFm"
    "MSIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFh"
    "cy5uaW50ZW5kby5jb20vMS4wLjAvaW50ZXJuYWxfY2VydGlmaWNhdGVzIn0.eyJqdGkiOiJhNTg5"
    "NTdkMC0yODBjLTQ2YjQtODlkMS0zZTFlNTRmYzA5YTYiLCJleHAiOjE0NTgyNzk3MDIsImlzcyI6"
    "Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5k"
    "by5jb20iLCJiczpncnQiOjIsInN1YiI6ImU2Yzk4ZGNiNmNjOTMxZTYiLCJiczpzdHMiOlsxMDQx"
    "NTY4NjUzNDg0NTQ4Niw0Mjk4MzUwMDAzMiwxMDk5NTYxOTY3NjE3LDQ2NDg2NjUwODgsODU5MDAw"
    "MDEyOSwyMzgxOTk1NzM0MzA5MzQ5MSwwLDAsMCwwLDAsMCwwLDAsMCwwXSwiaWF0IjoxNDU4MjY4"
    "OTAyLCJhdWQiOiJiNDMyNzFlNDJhZjMxZTNmIiwidHlwIjoidG9rZW4ifQ.WKOkPaFjO35dvB1RX"
    "dmgDG2EqD9L5yZWmCf7-W-fvN5nGasHLMEnPx6Us0TzkJcQs_hC9ZMe9hiX17w5I1kwb49ZKGcPb"
    "4Sc-gs81BPu00zKm58NFPUUwvuOFbwEIZiltV88Q__57OhNTnsnfECXwKSA1gKnOUsfwydUbn9zF"
    "PHXUbpxMBIAOl0L1ISwA25x1lIGQNV6vo5XZiP5MQgNZR-bVoXC0N-Xrpsdv-tSDYWbegdgXAaPL"
    "ure0m089WHnN1bJybWoHUqd5jHl8KfZY5UNH0iOVz6XN1gasQvJ9CPlZxFRajcSHX_8vi8JE_IHK"
    "B_BXJQlw5X3-8ISpTDtOw";
const char ExpectedIdToken[] =
    "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMu"
    "bmludGVuZG8uY29tLzEuMC4wL2NlcnRpZmljYXRlcyIsImtpZCI6IjgwYzRjZGQxLTJlMzMtNDUz"
    "Ni1iZTFlLWUwODdjYmU1NTc4ZCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJiNDMyNzFlNDJhZjMx"
    "ZTNmIiwiaWF0IjoxNDU4MjY4OTAyLCJzdWIiOiJlNmM5OGRjYjZjYzkzMWU2IiwiaXNzIjoiaHR0"
    "cHM6Ly9lMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvLmNv"
    "bSIsInR5cCI6ImlkX3Rva2VuIiwiYnM6ZGlkIjoiMDI0NDFjODBlZDU3OWJmMSIsImp0aSI6IjIx"
    "YjQyNDdmLTA3OWQtNDZlMC04ZjZiLWRhZWFhMzRmNDI5OCIsImV4cCI6MTQ1ODI3OTcwMn0.Us-s"
    "VwXV9AtKIfrufjV4XP4d_MLmEbv8ryEFt6WniRoPKapiZSQLBBmwflG75wVWp8PD70oE2AvWgDmN"
    "ePHc6emz2xGdvmydVPy3lP6yzbf5YxuAUQNAQaUkvZb8bsDU0t7Ah3M_zQCHnWG3YgQdkh0a9jur"
    "i9Ybsdrmnizgpq_uJqcEebTtShdkHzvgO9ECXH5OSY_emR8dquFHEtKmWFuK0GxWLM6e_cZArG3V"
    "5oIZzb9qh0d1hbe9y4p6BZI6nYApV0LDcb-guOUAHVrMR49ifa_zpc-HcTi8Aw-7S8tSTKOP5vqq"
    "BH9rtJUIMkHEgG8vI2oz6sISKYlWhSSbzg";

// Application Authentication Token あり版
const char InputForApp[] =
    "{\"idToken\":\"eyJraWQiOiI4MGM0Y2RkMS0yZTMzLTQ1MzYtYmUxZS1lMDg3Y2JlNTU3OGQiL"
    "CJhbGciOiJSUzI1NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZ"
    "jI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20vMS4wLjAvY2VydGlmaWNhdGVzIn0.eyJqdGkiOiI1Z"
    "TlkM2RmMi0wOGNmLTRjYjYtYTQ5My00NmVhOTE0YWJiZTciLCJleHAiOjE0NTgyNzk3MDQsImlzc"
    "yI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50Z"
    "W5kby5jb20iLCJuaW50ZW5kbyI6eyJkaSI6IjYxMDAwMDAwMDAwMDAwMjIiLCJzbiI6IlhBVjAwM"
    "DAwMDAwMzQ1IiwicGMiOiJIQUMtUy1aWlpaVihVU1opIiwidGkiOiIwMTIzNDU2Nzg5MDEyMzQ1I"
    "iwiZHQiOiJOWCBWRGV2IDEifSwic3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsImF1ZCI6ImI0MzI3M"
    "WU0MmFmMzFlM2YiLCJpYXQiOjE0NTgyNjg5MDQsImJzOmRpZCI6IjAyNDQxYzgwZWQ1NzliZjEiL"
    "CJ0eXAiOiJpZF90b2tlbiJ9.PU2WJBqNrAhBtmO-jz6_p-Pv7pqc5bfg2X0y2kpShQbRV8OOBfHu"
    "LahIjtLgZcP29z6oZKauHVVS2WaLMPtPDDzMtyYOSl-44bPlsCLRV9bDQS6FTcKltvHKA6KtoDj1"
    "uuPbWdAGPxCMgcYP5Fevt_-TL4LBClsspX43Yg7hKoX79OHLhFhv7gdbtjGl0B43mCph7q5DQirA"
    "rql9lqOOVcfZ6fv2t2qUd0b25HXDjw6R2Ym1RQXZZa1jkBWmRZZqh3M9NxNsALd-zv8W5sqxuPp7"
    "QZ7UpBBw0hsaWsX_h3oAylqX_Ea-yePufC7dG2OXVfS4up7-7zWW8kQbixze1w\",\"tokenType"
    "\":\"Bearer\",\"accessToken\":\"eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5"
    "NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMubmludGVuZG8uY29tLzEuMC4wL2ludGVybmFsX2NlcnRp"
    "ZmljYXRlcyIsImtpZCI6ImViYWZlMDJjLTIwOWQtNGRiNC04MjQzLWIxYTBmMDhlMWFmMSIsImFs"
    "ZyI6IlJTMjU2In0.eyJiczpncnQiOjIsImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1"
    "MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20iLCJhdWQiOiJiNDMyNzFlNDJhZjMx"
    "ZTNmIiwiaWF0IjoxNDU4MjY4OTA0LCJiczpzdHMiOlsxMDQxNTY4NjUzNDg0NTQ4Niw0Mjk4MzUw"
    "MDAzMiwxMDk5NTYxOTY3NjE3LDQ2NDg2NjUwODgsODU5MDAwMDEyOSwyMzgxOTk1NzM0MzA5MzQ5"
    "MSwwLDAsMCwwLDAsMCwwLDAsMCwwXSwic3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsInR5cCI6InRv"
    "a2VuIiwianRpIjoiZjUzNWUyOTctZTMwNi00MGUxLTg2OGUtZTFlZDMyMjE3MzA4IiwiZXhwIjox"
    "NDU4Mjc5NzA0fQ.EWqK1CImKBjnslr7toovMEscPwpKdRRYIRdN3WULaJMYJh5MmPHub92SP5Pbi"
    "Hgi0E8f8WrEIXyKjrY9DTg9vZahW2nXL5aOeiQ66RYVTOR_ZTbeLY9UgSSAdMTZHo-zJcEVDWpFX"
    "7I2YmnIPDXtqj9yePtFyrrNHniVAiXPKt2mywI2PkWtoasxjoAc1c2JHNV6jFjvGkL6zfIWopben"
    "sGobJutTZUaS4lUOToY022ubD0FA9h05x0nY497e_dcA9sbSnYUiU2xy81PZKG6GLW755VsOrpbf"
    "Exc9Xic5FudpI5z7C17924u9pvVzKK-nryQ6blt8WaxMYZNX5UQFA\",\"expiresIn\":10800,"
    "\"user\":{\"country\":\"\",\"etag\":\"\\\"e6c98dcb6cc931e6-698679167c0089e7\\"
    "\"\",\"friendsUpdatedAt\":1458268901,\"thumbnailUrl\":\"\",\"createdAt\":145"
    "8268901,\"links\":{},\"blocksUpdatedAt\":1458268901,\"presence\":{\"state\":"
    "\"OFFLINE\",\"extras\":{\"everyone\":{},\"friends\":{},\"foaf\":{},\"self\":"
    "{},\"favoriteFriends\":{}},\"updatedAt\":1458268901},\"updatedAt\":145826890"
    "1,\"extras\":{\"everyone\":{},\"favoriteFriends\":{},\"self\":{},\"foaf\":{}"
    ",\"friends\":{}},\"birthday\":\"0000-00-00\",\"id\":\"e6c98dcb6cc931e6\",\"n"
    "ickname\":\"\",\"permissions\":{\"personalNotificationUpdatedAt\":1458268901"
    ",\"personalAnalytics\":true,\"friendRequestReception\":true,\"friends\":\"EV"
    "ERYONE\",\"presence\":\"SELF\",\"personalNotification\":true,\"personalAnaly"
    "ticsUpdatedAt\":1458268901},\"deviceAccounts\":[{\"id\":\"02441c80ed579bf1\""
    "}]}}";
const char ExpectedAccessTokenForApp[] =
    "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMu"
    "bmludGVuZG8uY29tLzEuMC4wL2ludGVybmFsX2NlcnRpZmljYXRlcyIsImtpZCI6ImViYWZlMDJj"
    "LTIwOWQtNGRiNC04MjQzLWIxYTBmMDhlMWFmMSIsImFsZyI6IlJTMjU2In0.eyJiczpncnQiOjIs"
    "ImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5u"
    "aW50ZW5kby5jb20iLCJhdWQiOiJiNDMyNzFlNDJhZjMxZTNmIiwiaWF0IjoxNDU4MjY4OTA0LCJi"
    "czpzdHMiOlsxMDQxNTY4NjUzNDg0NTQ4Niw0Mjk4MzUwMDAzMiwxMDk5NTYxOTY3NjE3LDQ2NDg2"
    "NjUwODgsODU5MDAwMDEyOSwyMzgxOTk1NzM0MzA5MzQ5MSwwLDAsMCwwLDAsMCwwLDAsMCwwXSwi"
    "c3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsInR5cCI6InRva2VuIiwianRpIjoiZjUzNWUyOTctZTMw"
    "Ni00MGUxLTg2OGUtZTFlZDMyMjE3MzA4IiwiZXhwIjoxNDU4Mjc5NzA0fQ.EWqK1CImKBjnslr7t"
    "oovMEscPwpKdRRYIRdN3WULaJMYJh5MmPHub92SP5PbiHgi0E8f8WrEIXyKjrY9DTg9vZahW2nXL"
    "5aOeiQ66RYVTOR_ZTbeLY9UgSSAdMTZHo-zJcEVDWpFX7I2YmnIPDXtqj9yePtFyrrNHniVAiXPK"
    "t2mywI2PkWtoasxjoAc1c2JHNV6jFjvGkL6zfIWopbensGobJutTZUaS4lUOToY022ubD0FA9h05"
    "x0nY497e_dcA9sbSnYUiU2xy81PZKG6GLW755VsOrpbfExc9Xic5FudpI5z7C17924u9pvVzKK-n"
    "ryQ6blt8WaxMYZNX5UQFA";
const char ExpectedIdTokenForApp[] =
    "eyJraWQiOiI4MGM0Y2RkMS0yZTMzLTQ1MzYtYmUxZS1lMDg3Y2JlNTU3OGQiLCJhbGciOiJSUzI1"
    "NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFh"
    "cy5uaW50ZW5kby5jb20vMS4wLjAvY2VydGlmaWNhdGVzIn0.eyJqdGkiOiI1ZTlkM2RmMi0wOGNm"
    "LTRjYjYtYTQ5My00NmVhOTE0YWJiZTciLCJleHAiOjE0NTgyNzk3MDQsImlzcyI6Imh0dHBzOi8v"
    "ZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20iLCJu"
    "aW50ZW5kbyI6eyJkaSI6IjYxMDAwMDAwMDAwMDAwMjIiLCJzbiI6IlhBVjAwMDAwMDAwMzQ1Iiwi"
    "cGMiOiJIQUMtUy1aWlpaVihVU1opIiwidGkiOiIwMTIzNDU2Nzg5MDEyMzQ1IiwiZHQiOiJOWCBW"
    "RGV2IDEifSwic3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsImF1ZCI6ImI0MzI3MWU0MmFmMzFlM2Yi"
    "LCJpYXQiOjE0NTgyNjg5MDQsImJzOmRpZCI6IjAyNDQxYzgwZWQ1NzliZjEiLCJ0eXAiOiJpZF90"
    "b2tlbiJ9.PU2WJBqNrAhBtmO-jz6_p-Pv7pqc5bfg2X0y2kpShQbRV8OOBfHuLahIjtLgZcP29z6"
    "oZKauHVVS2WaLMPtPDDzMtyYOSl-44bPlsCLRV9bDQS6FTcKltvHKA6KtoDj1uuPbWdAGPxCMgcY"
    "P5Fevt_-TL4LBClsspX43Yg7hKoX79OHLhFhv7gdbtjGl0B43mCph7q5DQirArql9lqOOVcfZ6fv"
    "2t2qUd0b25HXDjw6R2Ym1RQXZZa1jkBWmRZZqh3M9NxNsALd-zv8W5sqxuPp7QZ7UpBBw0hsaWsX"
    "_h3oAylqX_Ea-yePufC7dG2OXVfS4up7-7zWW8kQbixze1w";

void RunTest(const char* accessToken, const char* idToken, const char* json, const a::detail::ApplicationInfo appInfo) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::baas::UserAccessTokenCache accessTokenCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(accessTokenCache.Initialize(s));
    a::baas::UserIdTokenCache idTokenCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(idTokenCache.Initialize(s));

    a::NetworkServiceAccountId current = {0xFEDCBA9876543210ull};
    a::NetworkServiceAccountId expected = {0xe6c98dcb6cc931e6ull};
    ASSERT_FALSE(accessTokenCache.IsAvailable(current));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(current, appInfo)));
    ASSERT_FALSE(accessTokenCache.IsAvailable(expected));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(expected, appInfo)));

    a::baas::UserLoginAdaptor adaptor(accessTokenCache, idTokenCache, appInfo);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));
    ASSERT_NE(current, adaptor.GetUserId());
    ASSERT_EQ(expected, adaptor.GetUserId());

    ASSERT_FALSE(accessTokenCache.IsAvailable(current));
    ASSERT_TRUE(accessTokenCache.IsAvailable(expected));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(current, appInfo)));
    if (appInfo)
    {
        ASSERT_TRUE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(expected, appInfo)));
    }
    else
    {
        ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(expected, appInfo)));
    }

    t::Buffer buffer(1024 * 32);
    size_t sizeActual;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(accessTokenCache.Load(&sizeActual, buffer.Get<char>(), buffer.GetSize(), expected));
    ASSERT_EQ(strlen(accessToken), sizeActual);
    ASSERT_EQ(0, std::memcmp(accessToken, buffer.GetAddress(), sizeActual));
    if (appInfo)
    {
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(idTokenCache.Load(&sizeActual, buffer.Get<char>(), buffer.GetSize(), expected, appInfo));
        ASSERT_EQ(strlen(idToken), sizeActual);
        ASSERT_EQ(0, std::memcmp(idToken, buffer.GetAddress(), sizeActual));
    }
    a::NintendoAccountId naId;
    ASSERT_FALSE(adaptor.TryGetNintendoAccountId(&naId));
    NN_UNUSED(naId);
}

} // ~namespace baasUserLogin

TEST(AccountAdaptor, BaasUserLogin)
{
    baasUserLogin::RunTest(
        baasUserLogin::ExpectedAccessToken,
        baasUserLogin::ExpectedIdToken,
        baasUserLogin::Input,
        a::detail::InvalidApplicationInfo);
}

TEST(AccountAdaptor, BaasUserLoginForApp)
{
    const auto appInfo = a::detail::ApplicationInfo::Get(0x012345678ABCDEFull, 0, a::detail::ApplicationMediaType::GameCard);
    NN_ABORT_UNLESS(appInfo);

    baasUserLogin::RunTest(
        baasUserLogin::ExpectedAccessTokenForApp,
        baasUserLogin::ExpectedIdTokenForApp,
        baasUserLogin::InputForApp,
        appInfo);
}

namespace federation
{
// Application Authentication Token なし版
const char Input[] =
    "{\"user\":{\"country\":\"\",\"etag\":\"\\\"e6c98dcb6cc931e6-698679167c0089e7"
    "\\\"\",\"thumbnailUrl\":\"\",\"friendsUpdatedAt\":1458268901,\"blocksUpdated"
    "At\":1458268901,\"createdAt\":1458268901,\"links\":{\"nintendoAccount\":{\"i"
    "d\":\"2f71d35f47ea8b01\",\"createdAt\":1453780220,\"updatedAt\":1453780220}}"
    ",\"extras\":{\"self\":{"
    "},\"favoriteFriends\":{},\"foaf\":{},\"friends\":{},\"everyone\":{}},\"updat"
    "edAt\":1458268901,\"presence\":{\"extras\":{\"everyone\":{},\"self\":{},\"fa"
    "voriteFriends\":{},\"friends\":{},\"foaf\":{}},\"updatedAt\":1458268901,\"st"
    "ate\":\"OFFLINE\"},\"id\":\"e6c98dcb6cc931e6\",\"birthday\":\"0000-00-00\",\""
    "permissions\":{\"friendRequestReception\":true,\"personalAnalytics\":true,\""
    "personalNotificationUpdatedAt\":1458268901,\"personalNotification\":true,\"p"
    "ersonalAnalyticsUpdatedAt\":1458268901,\"friends\":\"EVERYONE\",\"presence\""
    ":\"SELF\"},\"nickname\":\"\",\"deviceAccounts\":[{\"id\":\"02441c80ed579bf1\""
    "}]},\"accessToken\":\"eyJhbGciOiJSUzI1NiIsImtpZCI6ImViYWZlMDJjLTIwOWQtNGRiNC"
    "04MjQzLWIxYTBmMDhlMWFmMSIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkOD"
    "IwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20vMS4wLjAvaW50ZXJuYWxfY2VydGlmaWNhdG"
    "VzIn0.eyJqdGkiOiJhNTg5NTdkMC0yODBjLTQ2YjQtODlkMS0zZTFlNTRmYzA5YTYiLCJleHAiOj"
    "E0NTgyNzk3MDIsImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OG"
    "Etc2IuYmFhcy5uaW50ZW5kby5jb20iLCJiczpncnQiOjIsInN1YiI6ImU2Yzk4ZGNiNmNjOTMxZT"
    "YiLCJiczpzdHMiOlsxMDQxNTY4NjUzNDg0NTQ4Niw0Mjk4MzUwMDAzMiwxMDk5NTYxOTY3NjE3LD"
    "Q2NDg2NjUwODgsODU5MDAwMDEyOSwyMzgxOTk1NzM0MzA5MzQ5MSwwLDAsMCwwLDAsMCwwLDAsMC"
    "wwXSwiaWF0IjoxNDU4MjY4OTAyLCJhdWQiOiJiNDMyNzFlNDJhZjMxZTNmIiwidHlwIjoidG9rZW"
    "4ifQ.WKOkPaFjO35dvB1RXdmgDG2EqD9L5yZWmCf7-W-fvN5nGasHLMEnPx6Us0TzkJcQs_hC9ZM"
    "e9hiX17w5I1kwb49ZKGcPb4Sc-gs81BPu00zKm58NFPUUwvuOFbwEIZiltV88Q__57OhNTnsnfEC"
    "XwKSA1gKnOUsfwydUbn9zFPHXUbpxMBIAOl0L1ISwA25x1lIGQNV6vo5XZiP5MQgNZR-bVoXC0N-"
    "Xrpsdv-tSDYWbegdgXAaPLure0m089WHnN1bJybWoHUqd5jHl8KfZY5UNH0iOVz6XN1gasQvJ9CP"
    "lZxFRajcSHX_8vi8JE_IHKB_BXJQlw5X3-8ISpTDtOw\",\"expiresIn\":10800,\"idToken\""
    ":\"eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhY"
    "XMubmludGVuZG8uY29tLzEuMC4wL2NlcnRpZmljYXRlcyIsImtpZCI6IjgwYzRjZGQxLTJlMzMtN"
    "DUzNi1iZTFlLWUwODdjYmU1NTc4ZCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJiNDMyNzFlNDJhZ"
    "jMxZTNmIiwiaWF0IjoxNDU4MjY4OTAyLCJzdWIiOiJlNmM5OGRjYjZjYzkzMWU2IiwiaXNzIjoia"
    "HR0cHM6Ly9lMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvL"
    "mNvbSIsInR5cCI6ImlkX3Rva2VuIiwiYnM6ZGlkIjoiMDI0NDFjODBlZDU3OWJmMSIsImp0aSI6I"
    "jIxYjQyNDdmLTA3OWQtNDZlMC04ZjZiLWRhZWFhMzRmNDI5OCIsImV4cCI6MTQ1ODI3OTcwMn0.U"
    "s-sVwXV9AtKIfrufjV4XP4d_MLmEbv8ryEFt6WniRoPKapiZSQLBBmwflG75wVWp8PD70oE2AvWg"
    "DmNePHc6emz2xGdvmydVPy3lP6yzbf5YxuAUQNAQaUkvZb8bsDU0t7Ah3M_zQCHnWG3YgQdkh0a9"
    "juri9Ybsdrmnizgpq_uJqcEebTtShdkHzvgO9ECXH5OSY_emR8dquFHEtKmWFuK0GxWLM6e_cZAr"
    "G3V5oIZzb9qh0d1hbe9y4p6BZI6nYApV0LDcb-guOUAHVrMR49ifa_zpc-HcTi8Aw-7S8tSTKOP5"
    "vqqBH9rtJUIMkHEgG8vI2oz6sISKYlWhSSbzg\",\"tokenType\":\"Bearer\"}";
const char ExpectedAccessToken[] =
    "eyJhbGciOiJSUzI1NiIsImtpZCI6ImViYWZlMDJjLTIwOWQtNGRiNC04MjQzLWIxYTBmMDhlMWFm"
    "MSIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFh"
    "cy5uaW50ZW5kby5jb20vMS4wLjAvaW50ZXJuYWxfY2VydGlmaWNhdGVzIn0.eyJqdGkiOiJhNTg5"
    "NTdkMC0yODBjLTQ2YjQtODlkMS0zZTFlNTRmYzA5YTYiLCJleHAiOjE0NTgyNzk3MDIsImlzcyI6"
    "Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5k"
    "by5jb20iLCJiczpncnQiOjIsInN1YiI6ImU2Yzk4ZGNiNmNjOTMxZTYiLCJiczpzdHMiOlsxMDQx"
    "NTY4NjUzNDg0NTQ4Niw0Mjk4MzUwMDAzMiwxMDk5NTYxOTY3NjE3LDQ2NDg2NjUwODgsODU5MDAw"
    "MDEyOSwyMzgxOTk1NzM0MzA5MzQ5MSwwLDAsMCwwLDAsMCwwLDAsMCwwXSwiaWF0IjoxNDU4MjY4"
    "OTAyLCJhdWQiOiJiNDMyNzFlNDJhZjMxZTNmIiwidHlwIjoidG9rZW4ifQ.WKOkPaFjO35dvB1RX"
    "dmgDG2EqD9L5yZWmCf7-W-fvN5nGasHLMEnPx6Us0TzkJcQs_hC9ZMe9hiX17w5I1kwb49ZKGcPb"
    "4Sc-gs81BPu00zKm58NFPUUwvuOFbwEIZiltV88Q__57OhNTnsnfECXwKSA1gKnOUsfwydUbn9zF"
    "PHXUbpxMBIAOl0L1ISwA25x1lIGQNV6vo5XZiP5MQgNZR-bVoXC0N-Xrpsdv-tSDYWbegdgXAaPL"
    "ure0m089WHnN1bJybWoHUqd5jHl8KfZY5UNH0iOVz6XN1gasQvJ9CPlZxFRajcSHX_8vi8JE_IHK"
    "B_BXJQlw5X3-8ISpTDtOw";
const char ExpectedIdToken[] =
    "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMu"
    "bmludGVuZG8uY29tLzEuMC4wL2NlcnRpZmljYXRlcyIsImtpZCI6IjgwYzRjZGQxLTJlMzMtNDUz"
    "Ni1iZTFlLWUwODdjYmU1NTc4ZCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJiNDMyNzFlNDJhZjMx"
    "ZTNmIiwiaWF0IjoxNDU4MjY4OTAyLCJzdWIiOiJlNmM5OGRjYjZjYzkzMWU2IiwiaXNzIjoiaHR0"
    "cHM6Ly9lMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvLmNv"
    "bSIsInR5cCI6ImlkX3Rva2VuIiwiYnM6ZGlkIjoiMDI0NDFjODBlZDU3OWJmMSIsImp0aSI6IjIx"
    "YjQyNDdmLTA3OWQtNDZlMC04ZjZiLWRhZWFhMzRmNDI5OCIsImV4cCI6MTQ1ODI3OTcwMn0.Us-s"
    "VwXV9AtKIfrufjV4XP4d_MLmEbv8ryEFt6WniRoPKapiZSQLBBmwflG75wVWp8PD70oE2AvWgDmN"
    "ePHc6emz2xGdvmydVPy3lP6yzbf5YxuAUQNAQaUkvZb8bsDU0t7Ah3M_zQCHnWG3YgQdkh0a9jur"
    "i9Ybsdrmnizgpq_uJqcEebTtShdkHzvgO9ECXH5OSY_emR8dquFHEtKmWFuK0GxWLM6e_cZArG3V"
    "5oIZzb9qh0d1hbe9y4p6BZI6nYApV0LDcb-guOUAHVrMR49ifa_zpc-HcTi8Aw-7S8tSTKOP5vqq"
    "BH9rtJUIMkHEgG8vI2oz6sISKYlWhSSbzg";
const a::NintendoAccountId ExpectedNaId = {0x2f71d35f47ea8b01ull};

// Application Authentication Token あり版
const char InputForApp[] =
    "{\"idToken\":\"eyJraWQiOiI4MGM0Y2RkMS0yZTMzLTQ1MzYtYmUxZS1lMDg3Y2JlNTU3OGQiL"
    "CJhbGciOiJSUzI1NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZ"
    "jI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20vMS4wLjAvY2VydGlmaWNhdGVzIn0.eyJqdGkiOiI1Z"
    "TlkM2RmMi0wOGNmLTRjYjYtYTQ5My00NmVhOTE0YWJiZTciLCJleHAiOjE0NTgyNzk3MDQsImlzc"
    "yI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50Z"
    "W5kby5jb20iLCJuaW50ZW5kbyI6eyJkaSI6IjYxMDAwMDAwMDAwMDAwMjIiLCJzbiI6IlhBVjAwM"
    "DAwMDAwMzQ1IiwicGMiOiJIQUMtUy1aWlpaVihVU1opIiwidGkiOiIwMTIzNDU2Nzg5MDEyMzQ1I"
    "iwiZHQiOiJOWCBWRGV2IDEifSwic3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsImF1ZCI6ImI0MzI3M"
    "WU0MmFmMzFlM2YiLCJpYXQiOjE0NTgyNjg5MDQsImJzOmRpZCI6IjAyNDQxYzgwZWQ1NzliZjEiL"
    "CJ0eXAiOiJpZF90b2tlbiJ9.PU2WJBqNrAhBtmO-jz6_p-Pv7pqc5bfg2X0y2kpShQbRV8OOBfHu"
    "LahIjtLgZcP29z6oZKauHVVS2WaLMPtPDDzMtyYOSl-44bPlsCLRV9bDQS6FTcKltvHKA6KtoDj1"
    "uuPbWdAGPxCMgcYP5Fevt_-TL4LBClsspX43Yg7hKoX79OHLhFhv7gdbtjGl0B43mCph7q5DQirA"
    "rql9lqOOVcfZ6fv2t2qUd0b25HXDjw6R2Ym1RQXZZa1jkBWmRZZqh3M9NxNsALd-zv8W5sqxuPp7"
    "QZ7UpBBw0hsaWsX_h3oAylqX_Ea-yePufC7dG2OXVfS4up7-7zWW8kQbixze1w\",\"tokenType"
    "\":\"Bearer\",\"accessToken\":\"eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5"
    "NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMubmludGVuZG8uY29tLzEuMC4wL2ludGVybmFsX2NlcnRp"
    "ZmljYXRlcyIsImtpZCI6ImViYWZlMDJjLTIwOWQtNGRiNC04MjQzLWIxYTBmMDhlMWFmMSIsImFs"
    "ZyI6IlJTMjU2In0.eyJiczpncnQiOjIsImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1"
    "MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20iLCJhdWQiOiJiNDMyNzFlNDJhZjMx"
    "ZTNmIiwiaWF0IjoxNDU4MjY4OTA0LCJiczpzdHMiOlsxMDQxNTY4NjUzNDg0NTQ4Niw0Mjk4MzUw"
    "MDAzMiwxMDk5NTYxOTY3NjE3LDQ2NDg2NjUwODgsODU5MDAwMDEyOSwyMzgxOTk1NzM0MzA5MzQ5"
    "MSwwLDAsMCwwLDAsMCwwLDAsMCwwXSwic3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsInR5cCI6InRv"
    "a2VuIiwianRpIjoiZjUzNWUyOTctZTMwNi00MGUxLTg2OGUtZTFlZDMyMjE3MzA4IiwiZXhwIjox"
    "NDU4Mjc5NzA0fQ.EWqK1CImKBjnslr7toovMEscPwpKdRRYIRdN3WULaJMYJh5MmPHub92SP5Pbi"
    "Hgi0E8f8WrEIXyKjrY9DTg9vZahW2nXL5aOeiQ66RYVTOR_ZTbeLY9UgSSAdMTZHo-zJcEVDWpFX"
    "7I2YmnIPDXtqj9yePtFyrrNHniVAiXPKt2mywI2PkWtoasxjoAc1c2JHNV6jFjvGkL6zfIWopben"
    "sGobJutTZUaS4lUOToY022ubD0FA9h05x0nY497e_dcA9sbSnYUiU2xy81PZKG6GLW755VsOrpbf"
    "Exc9Xic5FudpI5z7C17924u9pvVzKK-nryQ6blt8WaxMYZNX5UQFA\",\"expiresIn\":10800,"
    "\"user\":{\"country\":\"\",\"etag\":\"\\\"e6c98dcb6cc931e6-698679167c0089e7\\"
    "\"\",\"friendsUpdatedAt\":1458268901,\"thumbnailUrl\":\"\",\"createdAt\":145"
    "8268901,\"links\":{\"nintendoAccount\":{\"id\":\"47ea8b012f71d35f\",\"create"
    "dAt\":1453780220,\"updatedAt\":1453780220}},\"blocksUpdatedAt\":1458268901,\""
    "presence\":{\"state\":"
    "\"OFFLINE\",\"extras\":{\"everyone\":{},\"friends\":{},\"foaf\":{},\"self\":"
    "{},\"favoriteFriends\":{}},\"updatedAt\":1458268901},\"updatedAt\":145826890"
    "1,\"extras\":{\"everyone\":{},\"favoriteFriends\":{},\"self\":{},\"foaf\":{}"
    ",\"friends\":{}},\"birthday\":\"0000-00-00\",\"id\":\"e6c98dcb6cc931e6\",\"n"
    "ickname\":\"\",\"permissions\":{\"personalNotificationUpdatedAt\":1458268901"
    ",\"personalAnalytics\":true,\"friendRequestReception\":true,\"friends\":\"EV"
    "ERYONE\",\"presence\":\"SELF\",\"personalNotification\":true,\"personalAnaly"
    "ticsUpdatedAt\":1458268901},\"deviceAccounts\":[{\"id\":\"02441c80ed579bf1\""
    "}]}}";
const char ExpectedAccessTokenForApp[] =
    "eyJqa3UiOiJodHRwczovL2UwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLmJhYXMu"
    "bmludGVuZG8uY29tLzEuMC4wL2ludGVybmFsX2NlcnRpZmljYXRlcyIsImtpZCI6ImViYWZlMDJj"
    "LTIwOWQtNGRiNC04MjQzLWIxYTBmMDhlMWFmMSIsImFsZyI6IlJTMjU2In0.eyJiczpncnQiOjIs"
    "ImlzcyI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5u"
    "aW50ZW5kby5jb20iLCJhdWQiOiJiNDMyNzFlNDJhZjMxZTNmIiwiaWF0IjoxNDU4MjY4OTA0LCJi"
    "czpzdHMiOlsxMDQxNTY4NjUzNDg0NTQ4Niw0Mjk4MzUwMDAzMiwxMDk5NTYxOTY3NjE3LDQ2NDg2"
    "NjUwODgsODU5MDAwMDEyOSwyMzgxOTk1NzM0MzA5MzQ5MSwwLDAsMCwwLDAsMCwwLDAsMCwwXSwi"
    "c3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsInR5cCI6InRva2VuIiwianRpIjoiZjUzNWUyOTctZTMw"
    "Ni00MGUxLTg2OGUtZTFlZDMyMjE3MzA4IiwiZXhwIjoxNDU4Mjc5NzA0fQ.EWqK1CImKBjnslr7t"
    "oovMEscPwpKdRRYIRdN3WULaJMYJh5MmPHub92SP5PbiHgi0E8f8WrEIXyKjrY9DTg9vZahW2nXL"
    "5aOeiQ66RYVTOR_ZTbeLY9UgSSAdMTZHo-zJcEVDWpFX7I2YmnIPDXtqj9yePtFyrrNHniVAiXPK"
    "t2mywI2PkWtoasxjoAc1c2JHNV6jFjvGkL6zfIWopbensGobJutTZUaS4lUOToY022ubD0FA9h05"
    "x0nY497e_dcA9sbSnYUiU2xy81PZKG6GLW755VsOrpbfExc9Xic5FudpI5z7C17924u9pvVzKK-n"
    "ryQ6blt8WaxMYZNX5UQFA";
const char ExpectedIdTokenForApp[] =
    "eyJraWQiOiI4MGM0Y2RkMS0yZTMzLTQ1MzYtYmUxZS1lMDg3Y2JlNTU3OGQiLCJhbGciOiJSUzI1"
    "NiIsImprdSI6Imh0dHBzOi8vZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFh"
    "cy5uaW50ZW5kby5jb20vMS4wLjAvY2VydGlmaWNhdGVzIn0.eyJqdGkiOiI1ZTlkM2RmMi0wOGNm"
    "LTRjYjYtYTQ5My00NmVhOTE0YWJiZTciLCJleHAiOjE0NTgyNzk3MDQsImlzcyI6Imh0dHBzOi8v"
    "ZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYmFhcy5uaW50ZW5kby5jb20iLCJu"
    "aW50ZW5kbyI6eyJkaSI6IjYxMDAwMDAwMDAwMDAwMjIiLCJzbiI6IlhBVjAwMDAwMDAwMzQ1Iiwi"
    "cGMiOiJIQUMtUy1aWlpaVihVU1opIiwidGkiOiIwMTIzNDU2Nzg5MDEyMzQ1IiwiZHQiOiJOWCBW"
    "RGV2IDEifSwic3ViIjoiZTZjOThkY2I2Y2M5MzFlNiIsImF1ZCI6ImI0MzI3MWU0MmFmMzFlM2Yi"
    "LCJpYXQiOjE0NTgyNjg5MDQsImJzOmRpZCI6IjAyNDQxYzgwZWQ1NzliZjEiLCJ0eXAiOiJpZF90"
    "b2tlbiJ9.PU2WJBqNrAhBtmO-jz6_p-Pv7pqc5bfg2X0y2kpShQbRV8OOBfHuLahIjtLgZcP29z6"
    "oZKauHVVS2WaLMPtPDDzMtyYOSl-44bPlsCLRV9bDQS6FTcKltvHKA6KtoDj1uuPbWdAGPxCMgcY"
    "P5Fevt_-TL4LBClsspX43Yg7hKoX79OHLhFhv7gdbtjGl0B43mCph7q5DQirArql9lqOOVcfZ6fv"
    "2t2qUd0b25HXDjw6R2Ym1RQXZZa1jkBWmRZZqh3M9NxNsALd-zv8W5sqxuPp7QZ7UpBBw0hsaWsX"
    "_h3oAylqX_Ea-yePufC7dG2OXVfS4up7-7zWW8kQbixze1w";
const a::NintendoAccountId ExpectedNaIdForApp = {0x47ea8b012f71d35full};

void RunTest(
    const char* accessToken, const char* idToken, const a::NintendoAccountId& expectedNaId,
    const char* json, const a::detail::ApplicationInfo appInfo) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::baas::UserAccessTokenCache accessTokenCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(accessTokenCache.Initialize(s));
    a::baas::UserIdTokenCache idTokenCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(idTokenCache.Initialize(s));

    a::NetworkServiceAccountId current = {0xFEDCBA9876543210ull};
    a::NetworkServiceAccountId expected = {0xe6c98dcb6cc931e6ull};
    ASSERT_FALSE(accessTokenCache.IsAvailable(current));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(current, appInfo)));
    ASSERT_FALSE(accessTokenCache.IsAvailable(expected));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(expected, appInfo)));

    a::baas::UserLoginAdaptor adaptor(accessTokenCache, idTokenCache, appInfo);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));
    ASSERT_NE(current, adaptor.GetUserId());
    ASSERT_EQ(expected, adaptor.GetUserId());

    ASSERT_FALSE(accessTokenCache.IsAvailable(current));
    ASSERT_TRUE(accessTokenCache.IsAvailable(expected));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(current, appInfo)));
    if (appInfo)
    {
        ASSERT_TRUE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(expected, appInfo)));
    }
    else
    {
        ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(expected, appInfo)));
    }

    t::Buffer buffer(1024 * 32);
    size_t sizeActual;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(accessTokenCache.Load(&sizeActual, buffer.Get<char>(), buffer.GetSize(), expected));
    ASSERT_EQ(strlen(accessToken), sizeActual);
    ASSERT_EQ(0, std::memcmp(accessToken, buffer.GetAddress(), sizeActual));
    if (appInfo)
    {
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(idTokenCache.Load(&sizeActual, buffer.Get<char>(), buffer.GetSize(), expected, appInfo));
        ASSERT_EQ(strlen(idToken), sizeActual);
        ASSERT_EQ(0, std::memcmp(idToken, buffer.GetAddress(), sizeActual));
    }
    a::NintendoAccountId naId;
    ASSERT_TRUE(adaptor.TryGetNintendoAccountId(&naId));
    ASSERT_EQ(expectedNaId, naId);
}
} // ~namespace federation

TEST(AccountAdaptor, BaasFederationLogin)
{
    federation::RunTest(
        federation::ExpectedAccessToken,
        federation::ExpectedIdToken,
        federation::ExpectedNaId,
        federation::Input,
        a::detail::InvalidApplicationInfo);
}

TEST(AccountAdaptor, BaasFederationLoginForApp)
{
    const auto appInfo = a::detail::ApplicationInfo::Get(0x012345678ABCDEFull, 0, a::detail::ApplicationMediaType::GameCard);
    NN_ABORT_UNLESS(appInfo);

    federation::RunTest(
        federation::ExpectedAccessTokenForApp,
        federation::ExpectedIdTokenForApp,
        federation::ExpectedNaIdForApp,
        federation::InputForApp,
        appInfo);
}
#endif // NNT_ACCOUNT_ENABLE_BAAS_USER_LOGIN

#if defined(NNT_ACCOUNT_ENABLE_BAAS_CLIENT_LOGIN_EXPIRATION)
namespace baasClientLoginExpr
{
const char* Inputs[] =
    {
        "{\"accessToken\":\"________________\",\"expiresIn\":1200}",
        "{\"accessToken\":\"________________\",\"expiresIn\":1203}",
    };

uint32_t ExpirationWaits[] = { 0, 3 };

void RunTest(uint32_t expiresIn, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    a::baas::ClientAccessTokenCache cache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(cache.Initialize(s));

    ASSERT_FALSE(cache.IsAvailable());

    a::baas::ClientLoginAdaptor adaptor(cache);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    if (expiresIn > 0)
    {
        // 有効 → 無効
        ASSERT_TRUE(cache.IsAvailable());
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(expiresIn));
    ASSERT_FALSE(cache.IsAvailable());
}

} // ~namespace baasClientLoginExpr

TEST(AccountAdaptor, BaasClientLoginExpiration )
{
    for (auto i = 0; i < sizeof(baasClientLoginExpr::Inputs) / sizeof(baasClientLoginExpr::Inputs[0]); ++ i)
    {
        baasClientLoginExpr::RunTest(
            baasClientLoginExpr::ExpirationWaits[i],
            baasClientLoginExpr::Inputs[i]);
    }
}
#endif // NNT_ACCOUNT_ENABLE_BAAS_CLIENT_LOGIN_EXPIRATION

#if defined(NNT_ACCOUNT_ENABLE_BAAS_USER_LOGIN_EXPIRATION)
namespace baasUserLoginExpr
{
const char* Inputs[] =
    {
        "{\"user\":{\"id\":\"e6c98dcb6cc931e6\"},\"accessToken\":\"________________\",\"idToken\":\"^^^^^^^^^^^^^^^^\",\"expiresIn\":1200}",
        "{\"user\":{\"id\":\"e6c98dcb6cc931e6\"},\"accessToken\":\"________________\",\"idToken\":\"^^^^^^^^^^^^^^^^\",\"expiresIn\":1203}",
    };

uint32_t ExpirationWaits[] = { 0, 3 };

void RunTest(uint32_t expiresIn, const char* json, const a::detail::ApplicationInfo appInfo) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    a::baas::UserAccessTokenCache accessTokenCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(accessTokenCache.Initialize(s));
    a::baas::UserIdTokenCache idTokenCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(idTokenCache.Initialize(s));

    a::NetworkServiceAccountId id = {0xe6c98dcb6cc931e6ull};
    ASSERT_FALSE(accessTokenCache.IsAvailable(id));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(id, appInfo)));

    a::baas::UserLoginAdaptor adaptor(accessTokenCache, idTokenCache, appInfo);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));
    ASSERT_EQ(id, adaptor.GetUserId());

    if (expiresIn > 0)
    {
        // 有効 → 無効
        ASSERT_TRUE(accessTokenCache.IsAvailable(id));
        ASSERT_TRUE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(id, appInfo)));
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(expiresIn));
    ASSERT_FALSE(accessTokenCache.IsAvailable(id));
    ASSERT_FALSE(idTokenCache.IsAvailable(std::pair<a::NetworkServiceAccountId, a::detail::ApplicationInfo>(id, appInfo)));
}

} // ~namespace baasUserLoginExpr

TEST(AccountAdaptor, BaasUserLoginExpiration )
{
    const auto appInfo = a::detail::ApplicationInfo::Get(0x012345678ABCDEFull, 0, a::detail::ApplicationMediaType::GameCard);
    NN_ABORT_UNLESS(appInfo);

    for (auto i = 0; i < sizeof(baasUserLoginExpr::Inputs) / sizeof(baasUserLoginExpr::Inputs[0]); ++ i)
    {
        baasUserLoginExpr::RunTest(
            baasUserLoginExpr::ExpirationWaits[i],
            baasUserLoginExpr::Inputs[i],
            appInfo);
    }
}
#endif // NNT_ACCOUNT_ENABLE_BAAS_USER_LOGIN_EXPIRATION
