﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>

namespace nnt { namespace account {

class MultiWaitHolder
{
private:
    nn::os::MultiWaitHolderType m_Holder;

public:
    explicit MultiWaitHolder(nn::os::Event& e) NN_NOEXCEPT
    {
        nn::os::InitializeMultiWaitHolder(&m_Holder, e.GetBase());
    }
    explicit MultiWaitHolder(std::function<void(nn::os::MultiWaitHolderType*)> initializer) NN_NOEXCEPT
    {
        initializer(&m_Holder);
    }
    ~MultiWaitHolder() NN_NOEXCEPT
    {
        nn::os::FinalizeMultiWaitHolder(&m_Holder);
    }
    nn::os::MultiWaitHolderType* GetBase() NN_NOEXCEPT
    {
        return &m_Holder;
    }
};

class MultiWait
{
private:
    nn::os::MultiWaitType m_Waiter;

public:
    MultiWait() NN_NOEXCEPT
    {
        nn::os::InitializeMultiWait(&m_Waiter);
    }
    ~MultiWait() NN_NOEXCEPT
    {
        nn::os::UnlinkAllMultiWaitHolder(&m_Waiter);
        nn::os::FinalizeMultiWait(&m_Waiter);
    }
    nn::os::MultiWaitHolderType* WaitAny() NN_NOEXCEPT
    {
        return nn::os::WaitAny(&m_Waiter);
    }
    nn::os::MultiWaitType* GetBase() NN_NOEXCEPT
    {
        return &m_Waiter;
    }
    void Link(MultiWaitHolder* pMwh) NN_NOEXCEPT
    {
        nn::os::LinkMultiWaitHolder(&m_Waiter, pMwh->GetBase());
    }
};

class Thread
{
    NN_DISALLOW_COPY(Thread);
    NN_DISALLOW_MOVE(Thread);

private:
    bool m_Initialized;
    nn::os::ThreadType m_Thread;

    std::function<void(void)> m_Function;

    static void ThreadFunction(void* p)
    {
        reinterpret_cast<Thread*>(p)->Run();
    }

    void Run() NN_NOEXCEPT
    {
        m_Function();
    }

public:
    Thread() NN_NOEXCEPT
        : m_Initialized(false)
    {
    }

    void Initialize(std::function<void(void)> f, void* stack, size_t stackSize, int priority, const char* name) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_Initialized);
        m_Function = f;
        auto result = nn::os::CreateThread(&m_Thread, ThreadFunction, this, stack, stackSize, priority);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::os::SetThreadNamePointer(&m_Thread, name);
        m_Initialized = true;
    }
    ~Thread() NN_NOEXCEPT
    {
        if (m_Initialized)
        {
            Finalize();
        }
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Initialized);
        Wait();
        nn::os::DestroyThread(&m_Thread);
        m_Initialized = false;
    }

    void Start() NN_NOEXCEPT
    {
        nn::os::StartThread(&m_Thread);
    }
    void Wait() NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_Thread);
    }
};

class StackMemoryAllocator;

class StackMemory
{
    friend class StackMemoryAllocator;

    NN_DISALLOW_COPY(StackMemory);

private:
    StackMemoryAllocator* m_pAllocator;
    int m_Slot;
    void* m_Memory;
    size_t m_Size;

    void Swap(StackMemory& rhs) NN_NOEXCEPT
    {
        std::swap(m_pAllocator, rhs.m_pAllocator);
        std::swap(m_Slot, rhs.m_Slot);
        std::swap(m_Memory, rhs.m_Memory);
        std::swap(m_Size, rhs.m_Size);
    }

    StackMemory(StackMemoryAllocator& allocator, int slot, void* memory, size_t size) NN_NOEXCEPT
        : m_pAllocator(&allocator)
        , m_Slot(slot)
        , m_Memory(memory)
        , m_Size(size)
    {
    }

public:
    StackMemory() NN_NOEXCEPT
        : m_pAllocator(nullptr)
        , m_Slot(-1)
        , m_Memory(nullptr)
        , m_Size(0)
    {
    }
    StackMemory(StackMemory&& rhs) NN_NOEXCEPT
        : m_pAllocator(rhs.m_pAllocator)
        , m_Slot(rhs.m_Slot)
        , m_Memory(rhs.m_Memory)
        , m_Size(rhs.m_Size)
    {
        rhs.m_pAllocator = nullptr;
        rhs.m_Slot = -1;
        rhs.m_Memory = nullptr;
        rhs.m_Size = 0;
    }
    ~StackMemory() NN_NOEXCEPT;
    StackMemory& operator=(StackMemory&& rhs) NN_NOEXCEPT
    {
        StackMemory tmp(std::move(rhs));
        this->Swap(tmp);
        return *this;
    }
    void* GetAddress() const NN_NOEXCEPT
    {
        return m_Memory;
    }
    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }
};

class StackMemoryAllocator
{
    friend class StackMemory;

    mutable nn::os::Mutex m_Mutex;
    const uintptr_t m_Memory;
    const size_t m_MemorySize;
    const size_t m_Count;
    const size_t m_Unit;

    bool* m_Slots;

protected:
    void Deallocate(int slot) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
        NN_SDK_ASSERT(slot >= 0);
        NN_SDK_ASSERT(static_cast<size_t>(slot) < m_Count);
        NN_SDK_ASSERT(m_Slots[slot]);
        m_Slots[slot] = false;
    }

public:
    template <size_t Count, size_t Unit>
    StackMemoryAllocator(char (&memory)[Count][Unit]) NN_NOEXCEPT
        : m_Mutex(false)
        , m_Memory(reinterpret_cast<uintptr_t>(memory))
        , m_MemorySize(sizeof(memory))
        , m_Count(Count)
        , m_Unit(Unit)
    {
        m_Slots = new bool[m_Count];
        for (size_t i = 0; i < m_Count; ++i)
        {
            m_Slots[i] = false;
        }
    }
    ~StackMemoryAllocator() NN_NOEXCEPT
    {
        for (size_t i = 0; i < m_Count; ++i)
        {
            NN_SDK_ASSERT(!m_Slots[i]);
        }
        delete[] m_Slots;
    }
    StackMemory Allocate() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
        for (size_t i = 0; i < m_Count; ++ i)
        {
            if (!m_Slots[i])
            {
                m_Slots[i] = true;
                return StackMemory(*this, static_cast<int>(i), reinterpret_cast<void*>(m_Memory + i * m_Unit), m_Unit);
            }
        }
        NN_ABORT("Not allocatable");
    }
};

inline StackMemory::~StackMemory() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        m_pAllocator->Deallocate(m_Slot);
    }
}

}} // ~namespace nnt::account
