﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "WaveDataUtility.h"

#include <Windows.h>
#include <fstream>
#include <iostream>

namespace {

// -----------------------------------------------------------------------------
void* LoadFile(const char* path)
{
    std::ifstream ifs(path, std::ios::binary);
    if (ifs.fail())
    {
        std::cerr << "cannot open " << path << std::endl;
        return nullptr;
    }
    ifs.seekg(0, ifs.end);
    int size = static_cast<int>(ifs.tellg());
    ifs.seekg(0, ifs.beg);
    char *str = new char[size];
    ifs.read(str, size);
    return str;
}

// -----------------------------------------------------------------------------
void UnloadFile(void* data)
{
    char* tmp = reinterpret_cast<char*>(data);
    delete [] tmp;
}

// -----------------------------------------------------------------------------
void PrintUsage()
{
    std::fprintf(stderr, "WaveDataDecorder.exe foo.{bfwav,bfstm}\n=> foo.{bfwav,bfstm}.dump.wav");
}

} // namespace



// =============================================================================
// argv[1] として渡された bfwav や bfstm をデコードして同名の .dump.wav に出力する
int main(int argc, char** argv)
{
    if (argc == 1)
    {
        PrintUsage();
    }

    const char* DataPath = argv[1];
    void* data = LoadFile(DataPath);

    nn::atkTool::WaveDataUtility::Result result;
    nn::atkTool::WaveDataUtility instance;

    // 初期化
    {
        result = instance.Initialize(data);
        if (result != nn::atkTool::WaveDataUtility::Result_Success)
        {
            return result;
        }
    }

    // ダンプ
    {
        char filename[MAX_PATH] = {'\0'};
        _snprintf_s(filename, sizeof(filename) / sizeof(filename[0]),
                MAX_PATH, "%s.dump.wav", DataPath, 0);
        result = instance.DumpDecodedSamples(filename);
        if (result != nn::atkTool::WaveDataUtility::Result_Success)
        {
            return result;
        }
        std::printf("output %s\n", filename);
    }

    // メタ情報の出力
    {
        nn::atkTool::WaveDataUtility::WaveInfo info;
        result = instance.GetWaveInfo(&info);
        if (result == nn::atkTool::WaveDataUtility::Result_Success)
        {
            std::printf("ch(%d) sampleRate(%d) sampleCount(%d) loop?(%s) (%d...%d)\n",
                    info.channelCount, info.sampleRate, info.sampleCount,
                    info.loopFlag ? "true" : "false",
                    info.loopStart, info.loopEnd);
        }
    }

    // 終了処理
    {
        result = instance.Finalize();
        if (result != nn::atkTool::WaveDataUtility::Result_Success)
        {
            return result;
        }
    }

    UnloadFile(data);

    return nn::atkTool::WaveDataUtility::Result_Success;
}
