﻿/*---------------------------------------------------------------------------*
  Project:  NintendoWare

  Copyright (C)Nintendo/HAL Laboratory, Inc.  All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//! @brief 四角上の任意の位置を指定して、４つの角の色から補間した色を求めます。
//!
//! @param[in]
//!
//! @return 補間して求めた色です。
//---------------------------------------------------------------------------
vec4 InterporateQuadColor(const in mat4 colors, const in vec2 coord)
{
    vec2 revCoord = 1.0 - coord;
    vec4 colorL = colors[0] * revCoord.y + colors[2] * coord.y;
    vec4 colorR = colors[1] * revCoord.y + colors[3] * coord.y;
    return colorL * revCoord.x + colorR * coord.x;
}

//---------------------------------------------------------------------------
//! @brief 任意の四角の頂点位置を求めます。
//!
//! @param[in]
//!
//! @return 頂点位置です。
//---------------------------------------------------------------------------
vec4 ShapeQuad(const in vec2 vertexIndex, const in vec4 sizeAndPosition)
{
    vec4 position = vec4(vertexIndex.x, - vertexIndex.y, 0.0, 1.0);
    position.xy = position.xy * sizeAndPosition.xy + sizeAndPosition.zw;
    return position;
}

//---------------------------------------------------------------------------
//! @brief 座標変換して座標を求めます。
//!
//! @param[in]
//!
//! @return 座標変換後の位置です。
//---------------------------------------------------------------------------
vec4 Transform34(const in vec4 modelView[3], const in vec4 position)
{
    vec4 tmp0;
    tmp0.x = dot(position, modelView[0]);
    tmp0.y = dot(position, modelView[1]);
    tmp0.z = dot(position, modelView[2]);
    tmp0.w = position.w;
    return tmp0;
}

//---------------------------------------------------------------------------
//! @brief ウィンドウペインのフレーム各頂点のペイン内座標を求めます。
//!
//! @param[in]
//!
//! @return ペイン内座標です。
//---------------------------------------------------------------------------
vec2 GetPaneCoordOfFrame(const in int frame)
{
    vec2 paneCoord;

    vec4 weight = uFrameSize;
    weight.xy /= uPaneSize.x;
    weight.zw /= uPaneSize.y;

    if (frame == 0)
    { // LT
        paneCoord = aVertexIndex * vec2(weight.x, weight.z);
    }
    else if (frame == 1)
    { // RT
        paneCoord = aVertexIndex * vec2(weight.y, weight.z);
        paneCoord += vec2(1.0 - weight.y, 0.0);
    }
    else if (frame == 2)
    { // LB
        paneCoord = aVertexIndex * vec2(weight.x, weight.w);
        paneCoord += vec2(0.0, 1.0 - weight.w);
    }
    else if (frame == 3)
    { // RB
        paneCoord = aVertexIndex * vec2(weight.y, weight.w);
        paneCoord += vec2(1.0 - weight.y, 1.0 - weight.w);
    }
    else if (frame == 4)
    { // L
        paneCoord = aVertexIndex * vec2(weight.x, 1.0 - weight.z - weight.w);
        paneCoord += vec2(0.0, weight.z);
    }
    else if (frame == 5)
    { // R
        paneCoord = aVertexIndex * vec2(weight.y, 1.0 - weight.z - weight.w);
        paneCoord += vec2(1.0 - weight.y, weight.z);
    }
    else if (frame == 6)
    { // T
        paneCoord = aVertexIndex * vec2(1.0 - weight.x - weight.y, weight.z);
        paneCoord += vec2(weight.x, 0.0);
    }
    else if (frame == 7)
    { // B
        paneCoord = aVertexIndex * vec2(1.0 - weight.x - weight.y, weight.w);
        paneCoord += vec2(weight.x, 1.0 - weight.w);
    }
    else if (frame == 8)
    { // frame4 - LT
        paneCoord = aVertexIndex * vec2(1.0 - weight.y, weight.z);
    }
    else if (frame == 9)
    { // frame4 - RT
        paneCoord = aVertexIndex * vec2(weight.y, 1.0 - weight.w);
        paneCoord += vec2(1.0 - weight.y, 0.0);
    }
    else if (frame == 10)
    { // frame4 - LB
        paneCoord = aVertexIndex * vec2(weight.x, 1.0 - weight.z);
        paneCoord += vec2(0.0, weight.z);
    }
    else if (frame == 11)
    { // frame4 - RB
        paneCoord = aVertexIndex * vec2(1.0 - weight.x, weight.w);
        paneCoord += vec2(weight.x, 1.0 - weight.w);
    }
    else if (frame == 12)
    { // frame2 - L
        paneCoord = aVertexIndex * vec2(1.0 - weight.y, 1.0 - weight.z - weight.w);
    }
    else
    {
        // none
        paneCoord = vec2(0.0, 0.0);
    }

    return paneCoord;
}

//---------------------------------------------------------------------------
//! @brief ペインの情報から色とテクスチャ座標を求めます。
//!
//! @param[in]
//---------------------------------------------------------------------------
void ComputeColorAndTexCoord(
    out vec4 color
#if 0 < NW_MULTI_TEXTURE_QUANTITY
    , in mat4 vertexTexCoord0
    , out vec4 texCoord0
#endif
#if 1 < NW_MULTI_TEXTURE_QUANTITY
    , in mat4 vertexTexCoord1
    , out vec4 texCoord1
#endif
#if 2 < NW_MULTI_TEXTURE_QUANTITY
    , in mat4 vertexTexCoord2
    , out vec4 texCoord2
#endif
    )
{
    if ((uFrameSpec & 0x40000000) != 0)
    { // constant color
        if ((uFrameSpec & 2) != 0)
        {
            #if 0 < NW_MULTI_TEXTURE_QUANTITY
            texCoord0 = vec4(aVertexIndex.x, aVertexIndex.y, 0, 1);
            #endif
            #if 1 < NW_MULTI_TEXTURE_QUANTITY
            texCoord1 = texCoord0;
            #endif
            #if 2 < NW_MULTI_TEXTURE_QUANTITY
            texCoord2 = texCoord0;
            #endif
        }
        else
        {
            #if 0 < NW_MULTI_TEXTURE_QUANTITY
            int i = int(1 * int(aVertexIndex.x) + 2 * int(aVertexIndex.y));
            texCoord0 = vertexTexCoord0[i];
            #endif
            #if 1 < NW_MULTI_TEXTURE_QUANTITY
            texCoord1 = vertexTexCoord1[i];
            #endif
            #if 2 < NW_MULTI_TEXTURE_QUANTITY
            texCoord2 = vertexTexCoord2[i];
            #endif
        }

#if NW_COLOR_EFFECT_ENABLED
        vPaneCoord.xy = aVertexIndex;
        vPaneCoord.zw = 1.0 - aVertexIndex;
#endif
        color = vec4(255.0);
    }
    else
    {
        int frame = (uFrameSpec >> 8) & 0xf;

        vec2 paneCoord;
        if ((uFrameSpec & 2) != 0)
        { // for frame
            #if 0 < NW_MULTI_TEXTURE_QUANTITY
            texCoord0 = vec4(aVertexIndex.x, aVertexIndex.y, 0, 1);
            #endif
            #if 1 < NW_MULTI_TEXTURE_QUANTITY
            texCoord1 = texCoord0;
            #endif
            #if 2 < NW_MULTI_TEXTURE_QUANTITY
            texCoord2 = texCoord0;
            #endif

            paneCoord = GetPaneCoordOfFrame(frame);
#if !NW_COLOR_EFFECT_ENABLED
            color = InterporateQuadColor(uVertexColorVsh, paneCoord);
#endif
        }
        else
        { // for content
            int i = int(1 * int(aVertexIndex.x) + 2 * int(aVertexIndex.y));
            #if 0 < NW_MULTI_TEXTURE_QUANTITY
            texCoord0 = vertexTexCoord0[i];
            #endif
            #if 1 < NW_MULTI_TEXTURE_QUANTITY
            texCoord1 = vertexTexCoord1[i];
            #endif
            #if 2 < NW_MULTI_TEXTURE_QUANTITY
            texCoord2 = vertexTexCoord2[i];
            #endif

            if (frame == 13)
            {
                vec4 weight = uFrameSize;
                weight.xy /= uPaneSize.x;
                weight.zw /= uPaneSize.y;

                paneCoord = aVertexIndex * vec2(1.0 - weight.x - weight.y, 1.0 - weight.z - weight.w);
                paneCoord += weight.xz;
#if !NW_COLOR_EFFECT_ENABLED
                color = InterporateQuadColor(uVertexColorVsh, paneCoord);
#endif
            }
            else
            {
#if NW_COLOR_EFFECT_ENABLED
                paneCoord = aVertexIndex;
#else
                color = uVertexColorVsh[i];
#endif
            }
        }

#if NW_COLOR_EFFECT_ENABLED
        vPaneCoord.xy = paneCoord;
        vPaneCoord.zw = 1.0 - paneCoord;
        color = vec4(1.0);
#endif
    }
}

//---------------------------------------------------------------------------
//! @brief ペインの情報からテクスチャ座標を変形します。
//!
//! @param[in]
//!
//! @return 変形後の座標です。
//---------------------------------------------------------------------------
vec2 DeformTexCoord(const in vec2 srcTexCoord)
{
    vec2 texCoord = srcTexCoord;

    if ((uFrameSpec & 2) != 0)
    {
        vec4 texScale = uTransform;

        if ((uFrameSpec & 0x10000) != 0)
        {
            texScale.xy = texScale.xy * uRcpTexSize0.yx;
        }
        else
        {
            texScale.xy = texScale.xy * uRcpTexSize0.xy;
        }

        // DOT BY DOT
        if ((uFrameSpec & 16) != 0)
        {
            texCoord.x = texCoord.x * texScale.x;
        }
        if ((uFrameSpec & 32) != 0)
        {
            texCoord.y = texCoord.y * texScale.y;
        }

        // ALIGN
        texScale = 1 - texScale;

        if ((uFrameSpec & 4) != 0)
        {
            texCoord.x = texCoord.x + texScale.x;
        }
        if ((uFrameSpec & 8) != 0)
        {
            texCoord.y = texCoord.y + texScale.y;
        }
    }

    // FLIP / SWAP
    if ((uFrameSpec & 0x20000) != 0)
    {
        texCoord.x = 1 - texCoord.x;
    }
    if ((uFrameSpec & 0x40000) != 0)
    {
        texCoord.y = 1 - texCoord.y;
    }
    if ((uFrameSpec & 0x10000) != 0)
    {
        texCoord.xy = texCoord.yx;
    }

    return texCoord;
}
