﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * MovieAudioOutputHandler class for processing audio data from movie::decoder
 *
 * @details
 *
 * Purpose:
 *     The main purpose of MovieAudioOutputHandler class is get decoded PCM audio data from movie::decoder and render
 * it to an output device capable of playing it back. This class receives callbacks from  MovieDecoderEventHandler when
 * audio output buffer is available. The buffer index is stored in a list. To make handling of rendering easier a fixed
 * size audio frame is generated. MovieAudioOutputHandler has a thread which will submit audio data to a registered
 * audio renderer. Fixed size frame is generated opportunistically in the MovieDecoderEventHandler callback or in
 * MovieAudioOutputHandler thread. Three lists are maintained, one for storing buffers from decoder, second for empty
 * AudioRenderer buffers, and third one for filled audio buffers. MovieAudioOutputHandler can also convert mono channel
 * audio to dual channel if needed, by duplicating audio data from the single channel.
 *
 * Setup:
 *     MoviePlayer class will create an instance of MovieAudioOutputHandler. It is initialized with MediaClock and an
 * event to be notified when rendering is completed. During initialization, a thread is created for submitting audio
 * data to renderer in timely manner. An instance of AudioOutRenderer is created and initialized for the playback of
 * audio. A temporary movie buffers is created to get data from audio decoder. The client needs to call Open() API with movie
 * decoder handle, audio sample rate, number of audio channels and indication to generate fixed size audio frames.
 * Most movie decoders outputs fixed size audio frame. MovieAudioOutputHandler can generate fixed size audio frames if
 * needed. Buffer are allocated for mono to dual channel conversion if needed. Finally, AudioRenderer is opened and
 * prepared.
 *
 * Main Processing:
 *     When client call Start() API, AudioRenderer is started. Also, MovieAudioOutputHandler thread is started.
 * This will result in the thread calling Render() API. This API will check whether there are any filled PCM buffers
 * available. If there is a filled PCM buffer, it will try to get an empty audio buffer from AudioRenderer. If it
 * succeeds getting an empty buffer, a mono to dual channel conversion is done if needed. Resampling may be done if
 * sample rate is not supported. Media clock anchor time might be updated. Finally, the filled buffer is submitted for
 * rendering. The MovieAudioOutputHandler buffer is moved back to empty list. If the buffer has end of stream flags
 * and if an event is registered it will be signaled.
 *
 * Teardown:
 *    Client need to call Stop() to stop AudioOutputHandler. AudioRenderer is stopped, any audio decoders remaining
 * in the list will be released back to the decoder. Calling Close() API will close AudioRenderer, return any decoders
 * buffers back to decoder, free buffers allocated. When Finalize() API is called the main thread is stopped and
 * resources are released. AudioOutRenderer object is deleted.
 *
 */
#include <cstdlib>
#include <cmath>
#include <nns/mm/mm_MovieAudioOutputHandler.h>

MovieAudioOutputHandler::MovieAudioOutputHandler()
    : m_AudioDecoder(nullptr),
      m_AudioOutRenderer(nullptr),
      m_ThreadStarted(false),
      m_ThreadDone(false),
      m_ThreadCreated(false),
      m_AudioOutStarted(false),
      m_AudioOutOpened(false),
      m_SampleRate(48000),
      m_Channels(2),
      m_InputSampleRate(-1),
      m_InputChannels(-1),
      m_FrameSize(0),
      m_PresentationTimeUs(0),
      m_Threadstacksize(1024 * 128),
      m_Threadstack(nullptr),
      m_SampleSize(16),
      m_InAudioBuffer(nullptr),
      m_AudioDecoderBuffers(16),
      m_AudioDecoderBufferSize(1024 * 8),
      m_FixedAudioFrameSize(1024 * 4),
      m_GenerateFixedSizeAudioFrames(false),
      m_DecoderBuffer(nullptr),
      m_ThreadSleepTimeMs(5),
      m_BufferSize(8192 * 4),
      m_AllFramesRendered(false),
      m_MediaClock(nullptr),
      m_AudioFirstAnchorTimeMediaUs(-1),
      m_AudioStartAnchorTimeUs(0),
      m_FramesWritten(0),
      m_IsPaused(false),
      m_MonoToDualChannel(false),
      m_MonoToDualChannelBuffer(nullptr),
      m_PlaybackPositionUs(0),
      m_AudioOutputBuffersReceived(0)
{
    // Buffers are managed by a vector list, allocate one big chunk.
    m_InAudioBufferSize = m_AudioDecoderBufferSize * m_AudioDecoderBuffers;
}

MovieAudioOutputHandler::~MovieAudioOutputHandler()
{

}

movie::Status MovieAudioOutputHandler::Initialize(MediaClock *mediaClock, nn::os::EventType *audioRenderComplete)
{
    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;
    m_AudioRenderComplete = audioRenderComplete;
    m_Channels = 0;
    m_SampleRate = 0;
    m_FrameSize = sizeof(short);
    m_ThreadDone = false;
    m_ThreadStarted = false;
    m_PresentationTimeUs = 0ll;
    m_ThreadSleepTimeMs = 5;
    m_SampleSize = 16;
    m_AudioOutStarted = false;
    m_BufferSize = 8192 * 4;
    m_AudioOutOpened = false;
    m_ThreadCreated = false;
    m_Threadstack = nullptr;
    m_MediaClock = mediaClock;
    m_AudioFirstAnchorTimeMediaUs = -1;
    m_AudioStartAnchorTimeUs = -1;
    m_FramesWritten = 0;
    m_IsPaused = false;
    m_GenerateFixedSizeAudioFrames = false;
    m_MonoToDualChannel = false;
    m_MonoToDualChannelBufferSize = 1024 * 32;
    m_MonoToDualChannelBuffer = nullptr;
    m_PlaybackPositionUs = 0;
    m_AudioCodecBuffers.reserve(64);
    m_AudioBufferRenderEmptyList.reserve(64);
    m_AudioBufferRenderFilledList.reserve(64);
    nn::os::InitializeMutex(&m_AudioBufferVectorMutex, false, 0);
    m_AudioOutRenderer = new AudioOutRenderer();
    if( m_AudioOutRenderer == nullptr )
    {
        return movie::Status_OutOfMemory;
    }
    // Allocate memory for render thread stack
    m_Threadstack = aligned_alloc(nn::os::ThreadStackAlignment, m_Threadstacksize);
    if( m_Threadstack == nullptr )
    {
        return movie::Status_OutOfMemory;
    }
    // Create audio renderer thread
    nnResult = nn::os::CreateThread(&m_ThreadType,
        &MovieAudioOutputHandlerThreadFunction,
        ( void* )this,
        m_Threadstack,
        m_Threadstacksize,
        nn::os::DefaultThreadPriority);
    m_ThreadCreated = true;
    if( nnResult.IsSuccess() )
    {
        nn::os::SetThreadName(&m_ThreadType, "MovieAudioOutputHandlerThread");
        movieStatus = movie::Status_Success;
    }
    movieStatus = m_AudioOutRenderer->Initialize();
    return movieStatus;
}

movie::Status MovieAudioOutputHandler::Finalize()
{
    // Release all resources.
    // Destroy audio rendering thread.
    m_ThreadDone = true;
    if( m_ThreadCreated == true )
    {
        if( m_ThreadStarted == true )
        {
            nn::os::WaitThread(&m_ThreadType);
            m_ThreadStarted = false;
        }
        nn::os::DestroyThread(&m_ThreadType);
        m_ThreadCreated = false;
    }
     nn::os::FinalizeMutex( &m_AudioBufferVectorMutex );
     if( m_Threadstack != nullptr )
     {
         free(m_Threadstack);
         m_Threadstack = nullptr;
     }
     m_AudioCodecBuffers.clear();
     m_AudioBufferRenderEmptyList.clear();
     m_AudioBufferRenderFilledList.clear();
     if( m_AudioOutRenderer != nullptr )
     {
         delete m_AudioOutRenderer;
         m_AudioOutRenderer = nullptr;
     }
     return movie::Status_Success;
}

int64_t MovieAudioOutputHandler::GetThreadSleepTimeMs()
{
    return m_ThreadSleepTimeMs;
}

movie::Status MovieAudioOutputHandler::Open(movie::Decoder* audioDecoder, uint32_t sampleRate, int channelCount, bool variableSizeAudioFrames)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    m_InputSampleRate = sampleRate;
    m_InputChannels   = channelCount;
    m_Channels = channelCount;
    m_AudioDecoder    = audioDecoder;
    m_GenerateFixedSizeAudioFrames = variableSizeAudioFrames;
    if( audioDecoder == nullptr )
    {
        nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
        return movieStatus;
    }
    if( channelCount > 2 )
    {
        m_AudioDecoderBufferSize = 1024 * 32;
        m_InAudioBufferSize = m_AudioDecoderBufferSize * m_AudioDecoderBuffers;
        m_FixedAudioFrameSize = 1024 * 12;
    }
    // Allocate one movie buffer to read decoded data from decoder.
    // Data is owned by this movie buffer.
    m_DecoderBuffer = new movie::Buffer(m_AudioDecoderBufferSize);
    if( m_DecoderBuffer == nullptr )
    {
        return movie::Status_OutOfMemory;
    }
    m_DecoderBuffer->SetRange(0, 0);
    // Allocate one big chunk of data to cache audio output data
    m_InAudioBuffer = new char[ m_InAudioBufferSize ];
    if( m_InAudioBuffer == nullptr )
    {
        return movie::Status_OutOfMemory;
    }
    // Distribute the memory into multiple Movie buffers.
    // These move buffers don’t own the data. Buffers are passed between
    // filled and empty list. Same buffer is moved to render list for
    // final rendering by AudioOut. Only data pointer is passed around.
    // Audio data is copied into AudioOut buffers.
    int memoryOffset = 0;
    for( int j = 0; j < m_AudioDecoderBuffers; ++j )
    {
        movie::Buffer buffer(&m_InAudioBuffer[ memoryOffset ], m_AudioDecoderBufferSize);
        memoryOffset += m_AudioDecoderBufferSize;
        m_AudioBufferRenderEmptyList.push_back(buffer);
    }

    if( m_InputChannels == 1 )
    {
        m_MonoToDualChannel = true;
        m_Channels = 2;
        m_MonoToDualChannelBuffer = aligned_alloc(nn::audio::AudioOutBuffer::AddressAlignment, m_MonoToDualChannelBufferSize);
        if( m_MonoToDualChannelBuffer == nullptr )
        {
            nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
            return movie::Status_OutOfMemory;
        }
        memset(m_MonoToDualChannelBuffer, 0, m_MonoToDualChannelBufferSize);
    }
    m_SampleRate = 48000;
    m_FrameSize = m_Channels * sizeof(short);
    movieStatus = m_AudioOutRenderer->Open(m_InputSampleRate, m_Channels, &m_ResampleAudio, m_FixedAudioFrameSize);
    if( movieStatus == movie::Status_Success )
    {
        m_AudioOutOpened = true;
    }
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
    return movieStatus;
}

movie::Status MovieAudioOutputHandler::Start()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    m_AllFramesRendered = false;
    m_AudioStartAnchorTimeUs = -1;
    m_MediaClock->SetMediaStartTime(m_AudioStartAnchorTimeUs);
    movieStatus = m_AudioOutRenderer->Start();
    if( movieStatus == movie::Status_Success )
    {
        m_AudioOutStarted = true;
    }
    if( m_ThreadStarted == false )
    {
        nn::os::StartThread(&m_ThreadType);
        m_ThreadStarted = true;
    }
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
    return movieStatus;
}

movie::Status MovieAudioOutputHandler::Render()
{
    movie::Status movieStatus = movie::Status_Success;
    void* buffer = nullptr;
    size_t size = 0;
    size_t dataSize = 0;
    movie::Buffer* audioBufferData;
    // Try to generate a fixed size frame from audio decoder output.
    GenerateFixedSizeAudioFrame();
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    if( m_AudioBufferRenderFilledList.size() > 0 )
    {
        audioBufferData = &m_AudioBufferRenderFilledList.front();
        buffer = audioBufferData->Base();
        size = audioBufferData->Size();
    }
    else
    {
        nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
        return movieStatus;
    }
    uint32_t flags = audioBufferData->GetUint32Data();
    int64_t audioPresentationTimeUs = audioBufferData->GetInt64Data();
    void* pcmDataBuffer = audioBufferData->Base();
    size_t pcmDataBufferCapacity = audioBufferData->Capacity();
    // If there is a valid audio buffer try to submit to AudioOut
    if( ( buffer != nullptr ) && ( size > 0 ) )
    {
        // Check whether AudioOut has any empty buffer available
        nn::audio::AudioOutBuffer* audioOutBuffer = nullptr;
        void *dataBuffer = nullptr;
        size_t outSize = 0;
        movieStatus = m_AudioOutRenderer->GetEmptyAudioOutBuffer(&audioOutBuffer, &dataBuffer, &outSize);
        if( movieStatus != movie::Status_Success )
        {
            nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
            return movieStatus;
        }
        m_PresentationTimeUs = audioPresentationTimeUs;
        if( m_AudioFirstAnchorTimeMediaUs == -1 )
        {
            m_MediaClock->SetMediaStartTime(m_PresentationTimeUs);
            m_AudioFirstAnchorTimeMediaUs = m_PresentationTimeUs;
        }
        // Update audio time stamp with mediaclock for AV sync
        if( ( m_AudioFirstAnchorTimeMediaUs >= 0 ) && ( m_MediaClock != nullptr ) )
        {
            int64_t nowUs = Clock::GetNowUs();
            int64_t nowMediaUs = m_AudioFirstAnchorTimeMediaUs + GetPlayedOutAudioDurationUs(nowUs);
            m_MediaClock->UpdateAnchorTime(nowMediaUs, nowUs, m_PresentationTimeUs);
            if( m_AudioStartAnchorTimeUs == -1 )
            {
                m_AudioStartAnchorTimeUs = Clock::GetNowUs();
                m_MediaClock->SetMediaStartTime(m_AudioFirstAnchorTimeMediaUs);
            }
        }
        bool copyDataToAudioOutBuffer = true;
        // Convert audio data from single channel to two channels if needed.
        // AudioOut doesn't support single channel playback.
        // If resampling is needed, use a temporary buffer. Otherwise use AudioOut buffer.
        if( m_MonoToDualChannel == true )
        {
            int16_t* pIn = static_cast< int16_t* >( buffer );
            int16_t* pOut = static_cast< int16_t* >( dataBuffer );
            int32_t outSamples = size / 2;
            if( m_ResampleAudio == true )
            {
                pOut = static_cast< int16_t* >( m_MonoToDualChannelBuffer );
            }
            for( int32_t index = 0; index < outSamples; index++ )
            {
                for( int32_t channel = 0; channel < m_Channels; channel++ )
                {
                    pOut[ channel ] = pIn[ index ];
                }
                pOut = &pOut[ m_Channels ];
            }
            dataSize = size * m_Channels;
            copyDataToAudioOutBuffer = false;
            if( m_ResampleAudio == true )
            {
                buffer = m_MonoToDualChannelBuffer;
                // Update data size due to mono to dual channel conversion.
                size = dataSize;
            }
        }
        size_t resampledSize = 0;
        size_t bufferSize = m_BufferSize;
        if( m_ResampleAudio == true )
        {
            m_AudioOutRenderer->ResampleAudio(dataBuffer, buffer, size, &resampledSize);
            dataSize = resampledSize;
            m_AudioOutRenderer->GetResampledOutBufferSize(&bufferSize);
        }
        else
        {
            if( copyDataToAudioOutBuffer == true )
            {
                dataSize = size;
                memcpy(dataBuffer, buffer, dataSize);
            }
        }
        // Update number of bytes sent to AudioOut to calculate audio consumption
        // TODO: Get accurate consumption information from AudioOut
        m_FramesWritten += dataSize / m_FrameSize;
        int renderduration = ( dataSize * 1000 / m_FrameSize ) / m_SampleRate;
        int granulaUpdateFactor = ceil(double(renderduration / 8.3333));
        m_ThreadSleepTimeMs = renderduration / granulaUpdateFactor;
        m_ThreadSleepTimeMs /= 2;
        m_PlaybackPositionUs = m_PresentationTimeUs;
        // Set new audio buffer to AudioOut for playback
        m_AudioOutRenderer->AppendFilledAudioOutBuffer(audioOutBuffer, dataBuffer, bufferSize, dataSize);
        // Check whether we got last audio frame.
        // Set playback complete
        if( flags == movie::BufferFlags_EndOfStream )
        {
            m_AllFramesRendered = true;
            if( m_AudioRenderComplete != nullptr )
            {
                nn::os::SignalEvent(m_AudioRenderComplete);
            }
        }
        // Move the buffer back to empty list.
        movie::Buffer emptyBuffer(pcmDataBuffer, pcmDataBufferCapacity);
        m_AudioBufferRenderEmptyList.push_back(emptyBuffer);
        m_AudioBufferRenderFilledList.erase(m_AudioBufferRenderFilledList.begin());
    }
    else
    {
        // There is no valid audio data, check whehter we got a EOS buffer
        if( flags == movie::BufferFlags_EndOfStream )
        {
            m_AllFramesRendered = true;
            if( m_AudioRenderComplete != nullptr )
            {
                nn::os::SignalEvent(m_AudioRenderComplete);
            }
        }
    }
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
    return movieStatus;
}//NOLINT(impl/function_size)

void MovieAudioOutputHandler::Stop()
{
    // Send any remaining decoder buffers, back to decoder
    nn::os::LockMutex( &m_AudioBufferVectorMutex );
    audioData *pcmData;
    int pcmListSize = m_AudioCodecBuffers.size();
    for(int i = 0; i < pcmListSize; i ++)
    {
        pcmData = &m_AudioCodecBuffers.front();
        if( m_AudioDecoder != nullptr )
        {
            m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        }
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
    }
    m_AudioOutRenderer->Stop();
    m_AudioOutputBuffersReceived = 0;
    nn::os::UnlockMutex( &m_AudioBufferVectorMutex );
}

bool MovieAudioOutputHandler::ExitThread()
{
    return m_ThreadDone;
}

bool MovieAudioOutputHandler::IsAudioOutStarted()
{
    return m_AudioOutStarted;
}

void MovieAudioOutputHandler::Close()
{
    // Close AudioOut.
    m_AudioOutRenderer->Close();
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    audioData *pcmData;
    int pcmListSize = m_AudioCodecBuffers.size();
    for(int i = 0; i < pcmListSize; i ++)
    {
        pcmData = &m_AudioCodecBuffers.front();
        m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
    }
    if( m_MonoToDualChannelBuffer != nullptr )
    {
        free(m_MonoToDualChannelBuffer);
        m_MonoToDualChannelBuffer = nullptr;
    }

    if( m_InAudioBuffer != nullptr )
    {
        delete[] m_InAudioBuffer;
        m_InAudioBuffer = nullptr;
    }
    if( m_DecoderBuffer != nullptr )
    {
        delete m_DecoderBuffer;
        m_DecoderBuffer = nullptr;
    }

    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
}

void MovieAudioOutputHandler::OnOutputAvailable(int index, int64_t presentationTimeUs, uint32_t flags)
{
    // Callback from decoder when output is available
    audioData pcmData;
    pcmData.index = index;
    pcmData.presentationTimeUs = presentationTimeUs;
    pcmData.flags = flags;
    nn::os::LockMutex( &m_AudioBufferVectorMutex );
    m_AudioCodecBuffers.push_back(pcmData);
    nn::os::UnlockMutex( &m_AudioBufferVectorMutex );
    GenerateFixedSizeAudioFrame();
}

void MovieAudioOutputHandler::GenerateFixedSizeAudioFrame()
{
    nn::os::LockMutex(&m_AudioBufferVectorMutex);

    int decoderBufferIndicesListSize = m_AudioCodecBuffers.size();
    int audioBufferRenderEmptyListSize = m_AudioBufferRenderEmptyList.size();
    movie::Buffer* audioBuffer = nullptr;
    audioData *pcmData = nullptr;
    if( ( decoderBufferIndicesListSize > 0 ) && ( audioBufferRenderEmptyListSize > 0 ) )
    {
        audioBuffer = &m_AudioBufferRenderEmptyList.front();
        pcmData = &m_AudioCodecBuffers.front();
        // If decoder frame size varies try to generate fixed size frame.
        if( m_GenerateFixedSizeAudioFrames == true )
        {
            if( ( m_DecoderBuffer->Size() > 0 ) && ( m_DecoderBuffer->Size() <= audioBuffer->Capacity() ) )
            {
                memcpy(( char* ) audioBuffer->Base() + audioBuffer->Offset(), m_DecoderBuffer->Base() + m_DecoderBuffer->Offset(), m_DecoderBuffer->Size());
                audioBuffer->SetRange(0, audioBuffer->Offset() + m_DecoderBuffer->Size());
                audioBuffer->SetInt64Data(m_DecoderBuffer->GetInt64Data());
            }
            else
            {
                m_DecoderBuffer->SetInt64Data(pcmData->presentationTimeUs);
            }
        }
        if( m_AudioDecoder != nullptr )
        {
            movie::Status movieStatus = m_AudioDecoder->GetOutputBuffer(pcmData->index, m_DecoderBuffer);
            if( ( ( movieStatus == movie::Status_Success ) && ( m_DecoderBuffer->Size() > 0 ) ) ||
                ( pcmData->flags == movie::BufferFlags_EndOfStream ) )
            {
                audioBuffer->SetUint32Data(pcmData->flags);
                audioBuffer->SetInt64Data(pcmData->presentationTimeUs);
                if( m_GenerateFixedSizeAudioFrames == true )
                {
                    if( audioBuffer->Size() == 0 )
                    {
                        audioBuffer->SetInt64Data(m_DecoderBuffer->GetInt64Data());
                    }
                    size_t maxFramesizeSize = m_FixedAudioFrameSize;
                    size_t renderBufferSize = audioBuffer->Size();
                    size_t renderBufferCapacity = audioBuffer->Capacity();
                    size_t dataAvailableInDecoderBuffer = m_DecoderBuffer->Size();
                    if( maxFramesizeSize > renderBufferCapacity )
                    {
                        maxFramesizeSize = renderBufferCapacity;
                    }
                    int32_t dataNeededForRenderBuffer = 0;
                    if( renderBufferSize < maxFramesizeSize )
                    {
                        dataNeededForRenderBuffer = maxFramesizeSize - renderBufferSize;
                    }
                    int32_t dataToCopy = 0;
                    if( ( dataNeededForRenderBuffer > 0 ) && ( dataAvailableInDecoderBuffer > 0 ) )
                    {
                        if( dataAvailableInDecoderBuffer < dataNeededForRenderBuffer )
                        {
                            dataToCopy = dataAvailableInDecoderBuffer;
                        }
                        else
                        {
                            dataToCopy = dataNeededForRenderBuffer;
                        }
                    }
                    // Copy data from decoder buffer.
                    // If partial data is copied from decoder, re-calculate and update the timestamp  for decoder buffer.
                    if( dataToCopy > 0 )
                    {
                        memcpy(( char* ) audioBuffer->Base() + renderBufferSize,
                            m_DecoderBuffer->Base() + m_DecoderBuffer->Offset(),
                            dataToCopy);
                        audioBuffer->SetRange(0, renderBufferSize + dataToCopy);
                        if( dataAvailableInDecoderBuffer > dataToCopy )
                        {
                            // Update decoder buffer with new data offset, time stamp.
                            m_DecoderBuffer->SetRange(dataToCopy + m_DecoderBuffer->Offset(), dataAvailableInDecoderBuffer - dataToCopy);
                            int64_t ts = pcmData->presentationTimeUs;
                            int64_t deltaTs = 0ll;
                            if( ( m_FrameSize > 0 ) && ( m_SampleRate > 0 ) )
                            {
                                deltaTs = ( ( dataToCopy * 1000 / m_FrameSize ) / m_SampleRate ) * 1000;
                            }
                            m_DecoderBuffer->SetInt64Data(ts + deltaTs);
                            // This decoder buffer data is not consumed completely.
                        }
                        else
                        {
                            // This decoder buffer data is consumed, reset m_DecoderBuffer fileds.
                            m_DecoderBuffer->SetInt64Data(-1);
                            m_DecoderBuffer->SetInt32Data(-1);
                            m_DecoderBuffer->SetUint32Data(0);
                            m_DecoderBuffer->SetRange(0, 0);
                        }
                    }
                }
                else
                {
                    memcpy(( char* ) audioBuffer->Base() + audioBuffer->Offset(),
                        m_DecoderBuffer->Base() + m_DecoderBuffer->Offset(),
                        m_DecoderBuffer->Size());
                    audioBuffer->SetRange(0, audioBuffer->Offset() + m_DecoderBuffer->Size());
                    m_AudioOutputBuffersReceived += 1;
                }

                bool fixedSizeAudioFrameReady = false;
                if( m_GenerateFixedSizeAudioFrames == true )
                {
                    if( ( audioBuffer->Size() == m_FixedAudioFrameSize ) ||
                        ( audioBuffer->GetUint32Data() == movie::BufferFlags_EndOfStream ) )
                    {
                        fixedSizeAudioFrameReady = true;
                    }
                }
                else
                {
                    fixedSizeAudioFrameReady = true;
                }
                if( fixedSizeAudioFrameReady == true )
                {
                    m_AudioBufferRenderFilledList.emplace_back(*audioBuffer);
                    m_AudioBufferRenderEmptyList.erase(m_AudioBufferRenderEmptyList.begin());
                    m_AudioOutputBuffersReceived += 1;
                }
            }
            else
            {
                m_DecoderBuffer->SetInt64Data(-1);
                m_DecoderBuffer->SetInt32Data(-1);
                m_DecoderBuffer->SetUint32Data(0);
                m_DecoderBuffer->SetRange(0, 0);
            }
        }
        if( m_AudioDecoder != nullptr )
        {
            m_AudioDecoder->ReleaseOutputBufferIndex(pcmData->index);
        }
        m_AudioCodecBuffers.erase(m_AudioCodecBuffers.begin());
    }
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
}//NOLINT(impl/function_size)

int64_t MovieAudioOutputHandler::GetPlayedOutAudioDurationUs(int64_t nowUs)
{
    if( m_AudioStartAnchorTimeUs == -1 )
    {
        m_AudioStartAnchorTimeUs = nowUs;
    }
    int64_t timeDuration = nowUs - m_AudioStartAnchorTimeUs;
    int64_t frameDuration = ( m_FramesWritten * 1000000L) / m_SampleRate;
    if( timeDuration > frameDuration )
    {
        return frameDuration;
    }
    return timeDuration;
}

void MovieAudioOutputHandler::Pause()
{
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    m_IsPaused = true;
    m_MediaClock->ClearAnchorTime();
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
}

void MovieAudioOutputHandler::Flush()
{
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    m_AudioCodecBuffers.clear();
    m_AudioBufferRenderEmptyList.clear();
    m_AudioBufferRenderFilledList.clear();
    int memoryOffset = 0;
    for( int j = 0; j < m_AudioDecoderBuffers; ++j )
    {
        movie::Buffer buffer(&m_InAudioBuffer[ memoryOffset ], m_AudioDecoderBufferSize);
        memoryOffset += m_AudioDecoderBufferSize;
        m_AudioBufferRenderEmptyList.push_back(buffer);
    }
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
}

void MovieAudioOutputHandler::Resume()
{
    nn::os::LockMutex(&m_AudioBufferVectorMutex);
    m_AudioStartAnchorTimeUs = -1;
    m_AudioFirstAnchorTimeMediaUs = -1;
    m_FramesWritten = 0;
    m_IsPaused = false;
    nn::os::UnlockMutex(&m_AudioBufferVectorMutex);
}

movie::Status MovieAudioOutputHandler::GetPlaybackPosition(int64_t *playbackPositionUs) const
{
    if( playbackPositionUs == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    else
    {
        *playbackPositionUs = m_PlaybackPositionUs;
        return movie::Status_Success;
    }
}

bool MovieAudioOutputHandler::CanStartRendering()
{
    int audioOutputBuffersNeededForStart = 8;
    // Vorbis needs more time to stabilize at start. Vorbis decoder generates variables size frames
    if( m_GenerateFixedSizeAudioFrames == true )
    {
        audioOutputBuffersNeededForStart = 16;
    }
    if( m_AudioOutputBuffersReceived > audioOutputBuffersNeededForStart )
    {
        return true;
    }
    else
    {
        return false;
    }
}

void MovieAudioOutputHandlerThreadFunction(void *arg)
{
    MovieAudioOutputHandler *movieAudioOutputHandler = ( MovieAudioOutputHandler*) arg;
    if( movieAudioOutputHandler != nullptr )
    {
        for( ;; )
        {
            if( movieAudioOutputHandler->ExitThread() )
            {
                break;
            }
            if( true == movieAudioOutputHandler->IsAudioOutStarted() )
            {
                if( false == movieAudioOutputHandler->IsPaused() )
                {
                    movieAudioOutputHandler->Render();
                }
            }
            int64_t threadSleepTimeMs = movieAudioOutputHandler->GetThreadSleepTimeMs();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(threadSleepTimeMs));
        }
    }
}
