﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_DebugPad.h>
#include <nns/hid/hid_DebugPadAsset.h>
#include <nns/hid/hid_GamePad.h>
#include <nns/hid/hid_GamePadAsset.h>
#include <nns/hid/hid_KeyboardAndMouse.h>
#include <nns/hid/hid_KeyboardAsset.h>
#include <nns/hid/hid_MouseAsset.h>
#include <nns/hid/hid_TouchScreen.h>
#include <nns/hid/hid_TouchScreenAsset.h>

namespace nns { namespace hid {

ControllerManager::ControllerManager() NN_NOEXCEPT
{
    // 何もしません。
}

ControllerManager::~ControllerManager() NN_NOEXCEPT
{
    // 管理対象のコントローラを開放します。
    for (std::vector<Controller*>::iterator it = m_Controllers.begin();
         it != m_Controllers.end();
         ++it)
    {
        delete *it;
    }

    m_Controllers.clear();

    // 管理対象のデバイスアセットを開放します。
    for (std::vector<DeviceAsset*>::iterator it = m_DeviceAssets.begin();
         it != m_DeviceAssets.end();
         ++it)
    {
        delete *it;
    }

    m_DeviceAssets.clear();
}

void ControllerManager::Update() NN_NOEXCEPT
{
    // 管理対象のデバイスアセットを更新します。
    for (std::vector<DeviceAsset*>::iterator it = m_DeviceAssets.begin();
         it != m_DeviceAssets.end();
         ++it)
    {
        (*it)->Update();
    }

    // 管理対象のコントローラを更新します。
    for (std::vector<Controller*>::iterator it = m_Controllers.begin();
         it != m_Controllers.end();
         ++it)
    {
        (*it)->Update();
    }
}

DeviceAsset *ControllerManager::GetDeviceAsset(int index) const NN_NOEXCEPT
{
    // 添字が範囲外ならば NULL を返します。
    if (index < 0 && m_DeviceAssets.size() <= static_cast<size_t>(index))
    {
        return 0;
    }

    return m_DeviceAssets.at(index);
}

DeviceAsset *ControllerManager::GetDeviceAsset(DeviceAssetId id,
                                               int index) const NN_NOEXCEPT
{
    // 添字が範囲外ならば NULL を返します。
    if (index < 0 && m_DeviceAssets.size() <= static_cast<size_t>(index))
    {
        return 0;
    }

    for (std::vector<DeviceAsset*>::const_iterator it = m_DeviceAssets.begin();
         it != m_DeviceAssets.end();
         ++it)
    {
        DeviceAsset* pDeviceAsset = *it;

        // 指定のデバイスアセット識別子を持つか判定します。
        if (pDeviceAsset->GetDeviceAssetId() == id)
        {
            // 目的の添字に到達したならばデバイスアセットを返します。
            if (index == 0)
            {
                return pDeviceAsset;
            }

            // デバイスアセットが見つかったので添字を進めます。
            --index;
        }
    }

    // 条件を満たすデバイスアセットが見つからなかった場合は NULL を返します。
    return 0;
}

Controller *ControllerManager::GetController(int index) const NN_NOEXCEPT
{
    // 添字が範囲外ならば NULL を返します。
    if (index < 0 && m_Controllers.size() <= static_cast<size_t>(index))
    {
        return 0;
    }

    return m_Controllers.at(index);
}

Controller *ControllerManager::GetController(ControllerId id,
                                             int index) const NN_NOEXCEPT
{
    // 添字が範囲外ならば NULL を返します。
    if (index < 0 && m_Controllers.size() <= static_cast<size_t>(index))
    {
        return 0;
    }

    for (std::vector<Controller*>::const_iterator it = m_Controllers.begin();
         it != m_Controllers.end();
         ++it)
    {
        Controller* pController = *it;

        // 指定のコントローラ識別子を持つか判定します。
        if (pController->GetControllerId() == id)
        {
            // 目的の添字に到達したならばコントローラを返します。
            if (index == 0)
            {
                return pController;
            }

            // コントローラが見つかったので添字を進めます。
            --index;
        }
    }

    // 条件を満たすコントローラが見つからなかった場合は NULL を返します。
    return 0;
}

namespace util {

void SetControllerManagerWithDefault(ControllerManager* pManager) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pManager);

    // デバッグパッドのデバイスアセットを管理対象に追加します。
    DebugPadAsset::Initialize();
    pManager->GetDeviceAssetList().push_back(new DebugPadAsset(pManager));

    // デバッグパッドを管理対象に追加します。
    pManager->GetControllerList().push_back(new DebugPad(pManager));

    // ゲームパッドのデバイスアセットを管理対象に追加します。
    GamePadAsset::Initialize();
    pManager->GetDeviceAssetList().push_back(new GamePadAsset(pManager));

    // ゲームパッドを管理対象に追加します。
    for (int i = 0; i < 4; ++i)
    {
        GamePad* pGamePad = new GamePad(pManager);
        pGamePad->SetPlayerNumber(i);
        pManager->GetControllerList().push_back(pGamePad);
    }

    // キーボードのデバイスアセットを管理対象に追加します。
    KeyboardAsset::Initialize();
    pManager->GetDeviceAssetList().push_back(new KeyboardAsset(pManager));

    // マウスのデバイスアセットを管理対象に追加します。
    MouseAsset::Initialize();
    pManager->GetDeviceAssetList().push_back(new MouseAsset(pManager));

    // キーボードとマウスを管理対象に追加します。
    pManager->GetControllerList().push_back(new KeyboardAndMouse(pManager));

    // タッチスクリーンのデバイスアセットを管理対象に追加します。
    TouchScreenAsset::Initialize();
    pManager->GetDeviceAssetList().push_back(new TouchScreenAsset(pManager));

    // タッチスクリーンを管理対象に追加します。
    pManager->GetControllerList().push_back(new TouchScreen(pManager));
}

} // namespace util

}} // namespace nns::hid
