﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <nn/gfx.h>
#include <nn/nn_Allocator.h>

#include "gfxLogPrimitiveRendererMeshRes.h"
#include "gfxLogPrimitiveRendererShaderRes.h"


namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

//---------------------------------------------------------------------------
//  前方宣言
//---------------------------------------------------------------------------
class Renderer;
class MeshBuffer;

//------------------------------------------------------------------------------
//! @brief 定数です。
//------------------------------------------------------------------------------
enum BlendType
{
    //! @brief  通常ブレンド
    //!         BlendFunction          : nn::gfx::BlendFunction_Add
    //!         SourceBlendFactor      : nn::gfx::BlendFactor_SourceAlpha
    //!         DestinationBlendFactor : nn::gfx::BlendFactor_OneMinusSourceAlpha
    BlendType_Normal = 0,

    //! @brief  加算ブレンド
    //!         BlendFunction          : nn::gfx::BlendFunction_Add
    //!         SourceBlendFactor      : nn::gfx::BlendFactor_SourceAlpha
    //!         DestinationBlendFactor : nn::gfx::BlendFactor_One
    BlendType_Add,

    //! @brief  減算ブレンド
    //!         BlendFunction          : nn::gfx::BlendFunction_ReverseSubtract
    //!         SourceBlendFactor      : nn::gfx::BlendFactor_SourceAlpha
    //!         DestinationBlendFactor : nn::gfx::BlendFactor_One
    BlendType_Sub,

    //! @brief  乗算ブレンド
    //!         BlendFunction          : nn::gfx::BlendFunction_Add
    //!         SourceBlendFactor      : nn::gfx::BlendFactor_Zero
    //!         DestinationBlendFactor : nn::gfx::BlendFactor_SourceColor
    BlendType_Mul,

    //! @brief  不透明（ブレンドなし）
    BlendType_Opacity,

    BlendType_CountMax
};

enum DepthStencilType
{
    //! @brief 深度テストと深度値の書き込みを共に有効にします。
    //!        DepthWrite : true
    //!        DepthTest  : true
    DepthStencilType_DepthWriteTest = 0,

    //! @brief 深度値の書き込みのみ有効にします。
    //!        DepthWrite : true
    //!        DepthTest  : false
    DepthStencilType_DepthWrite,

    //! @brief 深度テストのみ有効にします。
    //!        DepthWrite : false
    //!        DepthTest  : true
    DepthStencilType_DepthTest,

    //! @brief 深度テストと深度値の書き込みを共に無効にします。
    //!        DepthWrite : false
    //!        DepthTest  : false
    DepthStencilType_DepthNoWriteTest,

    DepthStencilType_CountMax
};

//! @brief グラフィックスリソースクラスです。
class GraphicsResource
{
    NN_DISALLOW_COPY(GraphicsResource);
    NN_DISALLOW_MOVE(GraphicsResource);

public:
    //---------------------------------------------------------------------------
    //! @brief プリミティブレンダラが利用するグラフィックスリソースを初期化します。
    //!
    //! @param[in] pGfxDevice                  GPU デバイスです。
    //! @param[in] pMemory                     利用するメモリのポインタです。
    //! @param[in] memorySize                  pMemory が指すメモリ領域のサイズです。
    //! @param[in] pMemoryPool                 利用するメモリプールのポインタです。
    //! @param[in] memoryPoolOffset            pMemoryPool のオフセット値です。
    //! @param[in] memoryPoolSize              pMmoeryPool のサイズです。
    //! @return グラフィックスリソースの初期化に成功したら、trueを返します。
    //---------------------------------------------------------------------------
    bool Initialize(
        nn::gfx::Device*                        pGfxDevice,
        void*                                   pMemory,
        size_t                                  memorySize,
        nn::gfx::MemoryPool*                    pMemoryPool,
        ptrdiff_t                               memoryPoolOffset,
        size_t                                  memoryPoolSize ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief プリミティブレンダラが利用するグラフィックスリソースを破棄します。
    //!
    //! @param[in] pGfxDevice              GPU デバイスです。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief システムのデフォルトシェーダを取得します。
    //! @param[in] shaderVariation         シェーダバリエーションです。
    //! @return シェーダです。
    //---------------------------------------------------------------------------
    Shader* GetDefaultShader( const ShaderVariation shaderVariation ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief システムのデフォルト描画バッファリストを取得します。
    //! @return メッシュ集合です。
    //---------------------------------------------------------------------------
    MeshSet* GetDefaultMeshSet() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief GraphicsResourceが必要とするMemoryPoolのサイズを取得します。
    //! @param[in] pGfxDevice              GPU デバイスです。
    //! @return                            MemoryPoolのサイズです。
    //---------------------------------------------------------------------------
    static size_t GetRequiredMemoryPoolSize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief GraphicsResourceで利用するMemoryPoolのアライメントを取得します。
    //! @param[in] pGfxDevice              GPU デバイスです。
    //! @return                            MemoryPoolのアライメントです。
    //---------------------------------------------------------------------------
    static size_t GetMemoryPoolAlignment( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief GraphicsResourceで利用するメモリのサイズを取得します。
    //! @param[in] pGfxDevice              GPU デバイスです。
    //! @return                            メモリサイズです。
    //---------------------------------------------------------------------------
    static size_t GetRequiredMemorySize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief GraphicsResourceで利用するメモリのアライメントを取得します。
    //! @param[in] pGfxDevice              GPU デバイスです。
    //! @return                            メモリのアライメントです。
    //---------------------------------------------------------------------------
    static size_t GetRequiredMemoryAlignment( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;

    //! @brief コンストラクタです。
    GraphicsResource()  NN_NOEXCEPT
        : m_pResShaderFile(NULL)
        , m_DefaultMeshSet()
        , m_ShaderCodeType(nn::gfx::ShaderCodeType_Source)
        , m_GpuBuffer()
    {
    }

private:
    bool InitializeRenderState(
        nn::gfx::Device*   pGfxDevice,
        nn::util::BytePtr& pMemory,
        size_t             memorySize ) NN_NOEXCEPT;
    static void InitializeRenderStateInfo(
        nn::gfx::BlendState::InfoType& info,
        nn::gfx::BlendTargetStateInfo& targetInfo,
        int                            blendType ) NN_NOEXCEPT;
    void FinalizeRenderState( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;
    bool InitializeShader(
        nn::gfx::Device*   pGfxDevice,
        void*              pShaderBinaryFile,
        nn::util::BytePtr& pMemory,
        size_t             memorySize ) NN_NOEXCEPT;
    void FinalizeShader( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;
    bool InitializeDefaultMeshSet(
        nn::gfx::Device*     pGfxDevice,
        nn::gfx::MemoryPool* pMemoryPool,
        ptrdiff_t            memoryPoolOffset,
        size_t               memoryPoolSize ) NN_NOEXCEPT;
    void FinalizeDefaultMeshSet( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;
    static size_t GetRequiredMemorySizeForBlendState() NN_NOEXCEPT;
    static size_t GetRequiredMemorySizeForShader() NN_NOEXCEPT;

private:
    nn::gfx::BlendTargetStateInfo   m_BlendTargetStateInfo[ BlendType_CountMax ];
    nn::gfx::BlendState             m_BlendState[ BlendType_CountMax ];
    nn::gfx::DepthStencilState      m_DepthStencilState[ DepthStencilType_CountMax ];
    nn::gfx::RasterizerState        m_RasterizerState[ nn::gfx::PrimitiveTopologyType_End ][ nn::gfx::CullMode_End ][ nn::gfx::FillMode_End ];
    nn::gfx::ResShaderFile*         m_pResShaderFile;
    MeshSet                         m_DefaultMeshSet;
    Shader                          m_DefaultShader[ ShaderVariation_CountMax ];
    nn::gfx::ShaderCodeType         m_ShaderCodeType;
    GpuBuffer                       m_GpuBuffer;
};



} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
