﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/nn_SdkLog.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn_FuncPtrInline.h>
#endif

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! 定数です。
//------------------------------------------------------------------------------

namespace
{

//! @brief 初期値です。
const nn::util::Float4 Float4One  = NN_UTIL_FLOAT_4_INITIALIZER( 1.f, 1.f, 1.f, 1.f );
const nn::util::Float4 Float4Zero = NN_UTIL_FLOAT_4_INITIALIZER( 0.f, 0.f, 0.f, 0.f );

const nn::util::Float2 Float2One  = NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 1.f );
const nn::util::Float2 Float2Zero = NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 0.f );

const nn::util::Float4  Float4White = NN_UTIL_FLOAT_4_INITIALIZER( 1.f, 1.f, 1.f, 1.f );

/*
const nn::util::Uint8x4 Uint8Black = {{ 0, 0, 0, 0 }};
const nn::util::Uint8x4 Uint8Red   = {{ 255, 0, 0, 255 }};
const nn::util::Uint8x4 Uint8Green = {{ 0, 255, 0, 255 }};
const nn::util::Uint8x4 Uint8Blue  = {{ 0, 0, 255, 255 }};
*/

static const nn::util::MatrixT4x4fType MatrixT4x4Unit =
    NN_UTIL_MATRIX_T4X4F_INITIALIZER(
    1.f, 0.f, 0.f, 0.f,
    0.f, 1.f, 0.f, 0.f,
    0.f, 0.f, 1.f, 0.f,
    0.f, 0.f, 0.f, 1.f );

static const nn::gfx::PrimitiveTopology TopologyTable[] =
{
    nn::gfx::PrimitiveTopology_TriangleList,  // Quad
    nn::gfx::PrimitiveTopology_TriangleList,  // QuadTextured
    nn::gfx::PrimitiveTopology_LineList,      // QuadWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Cube
    nn::gfx::PrimitiveTopology_TriangleList,  // CubeTextured
    nn::gfx::PrimitiveTopology_LineList,      // CubeWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Sphere
    nn::gfx::PrimitiveTopology_TriangleList,  // SphereTextured
    nn::gfx::PrimitiveTopology_LineList,      // SphereWired
    nn::gfx::PrimitiveTopology_TriangleList,  // SphereCoarse
    nn::gfx::PrimitiveTopology_TriangleList,  // SphereTexturedCoarse
    nn::gfx::PrimitiveTopology_LineList,      // SphereWiredCoarse
    nn::gfx::PrimitiveTopology_LineList,      // Line
    nn::gfx::PrimitiveTopology_PointList,     // Point
    nn::gfx::PrimitiveTopology_TriangleList,  // Triangle
    nn::gfx::PrimitiveTopology_LineStrip,     // TriangleWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Circle
    nn::gfx::PrimitiveTopology_TriangleList,  // CircleTextured
    nn::gfx::PrimitiveTopology_LineStrip,     // CircleWired
    nn::gfx::PrimitiveTopology_TriangleList,  // CircleCoarse
    nn::gfx::PrimitiveTopology_TriangleList,  // CircleTexturedCoarse
    nn::gfx::PrimitiveTopology_LineStrip,     // CircleWiredCoarse
    nn::gfx::PrimitiveTopology_TriangleList,  // ScreenQuad
    nn::gfx::PrimitiveTopology_TriangleList,  // ScreenYFlipQuad
    nn::gfx::PrimitiveTopology_TriangleList,  // UpperHalfSphere
    nn::gfx::PrimitiveTopology_TriangleList,  // UpperHalfSphereTextured
    nn::gfx::PrimitiveTopology_LineList,      // UpperHalfSphereWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Pipe
    nn::gfx::PrimitiveTopology_TriangleList,  // PipeTextured
    nn::gfx::PrimitiveTopology_LineList,      // PipeWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Cone
    nn::gfx::PrimitiveTopology_TriangleList,  // ConeTextured
    nn::gfx::PrimitiveTopology_LineList,      // ConeWired
    nn::gfx::PrimitiveTopology_TriangleList,  // User
};

}; // namespace




// グラフィックスリソース
GraphicsResource    g_GraphicsResource;
static nn::gfx::MemoryPool g_MemoryPool;
static void* g_pMemoryPoolBuffer = nullptr;
static void* g_pMemoryBuffer = nullptr;
static bool g_PrimitiveRendererInitialized = false;

//  プリミティブレンダラを初期化します。
Renderer* CreateRenderer( nn::gfx::Device*                        pGfxDevice,
                          RendererInfo&                           info )
{
    if ( g_PrimitiveRendererInitialized == true )
    {
        return nullptr;
    }

    size_t memoryPoolSize = 0;
    ptrdiff_t memoryPoolOffset = 0;
    size_t memoryBufferSize = 0;

    // メモリプールを初期化
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuCached );

    // Renderer1つ分のMemoryPoolのサイズを計算
    memoryPoolSize =  nns::gfx::PrimitiveRenderer::Renderer::GetRequiredMemoryPoolSize( pGfxDevice, info );
    memoryPoolSize += nns::gfx::PrimitiveRenderer::Renderer::GetMemoryPoolAlignment( pGfxDevice, info );
    // GraphicsResourceに必要なMemoryPoolのサイズを計算して追加
    memoryPoolSize += nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemoryPoolSize( pGfxDevice );
    memoryPoolSize += nns::gfx::PrimitiveRenderer::GraphicsResource::GetMemoryPoolAlignment( pGfxDevice );
    memoryPoolSize = nn::util::align_up( memoryPoolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( pGfxDevice, memoryPoolInfo ));
    g_pMemoryPoolBuffer = info.GetAllocateFunction()( memoryPoolSize,
                                                      nn::gfx::MemoryPool::GetPoolMemoryAlignment( pGfxDevice, memoryPoolInfo ),
                                                      info.GetAllocateFunctionUserData() );
    memoryPoolInfo.SetPoolMemory( g_pMemoryPoolBuffer, memoryPoolSize );
    g_MemoryPool.Initialize( pGfxDevice, memoryPoolInfo );

    // グラフィックスリソースに必要なメモ領域を確保
    memoryBufferSize = nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemorySize( pGfxDevice );
    g_pMemoryBuffer = info.GetAllocateFunction()( memoryBufferSize,
                                                  nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemoryAlignment( pGfxDevice ),
                                                  info.GetAllocateFunctionUserData() );

    // グラフィックスリソースを初期化
    if ( g_GraphicsResource.Initialize( pGfxDevice,
                                        g_pMemoryBuffer,
                                        memoryBufferSize,
                                        &g_MemoryPool,
                                        memoryPoolOffset,
                                        memoryPoolSize ) == false )
    {
        return nullptr;
    }

    // メモリプールのオフセットを、GraphicsResource分ずらす
    memoryPoolOffset += nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemoryPoolSize( pGfxDevice );
    memoryPoolOffset = nn::util::align_up( memoryPoolOffset, nns::gfx::PrimitiveRenderer::Renderer::GetMemoryPoolAlignment( pGfxDevice, info ) );

    // Rendererを1つ生成
    Renderer* pRenderer = nullptr;
    const size_t size = sizeof(Renderer);
    void* ptr = info.GetAllocateFunction()( size, 16, info.GetAllocateFunctionUserData() );
    pRenderer = new ( ptr ) Renderer;
    NN_SDK_ASSERT_NOT_NULL( pRenderer );
    if ( pRenderer == nullptr )
    {
        return nullptr;
    }

    // Rendererを初期化する
    if ( pRenderer->Initialize( pGfxDevice,
                                info,
                                &g_GraphicsResource,
                                &g_MemoryPool,
                                memoryPoolOffset,
                                memoryPoolSize ) == false )
    {
        return nullptr;
    }

    g_PrimitiveRendererInitialized = true;

    return pRenderer;
}


//  プリミティブレンダラを破棄します。
void DestroyRenderer( Renderer* pRenderer,
                      nn::gfx::Device* pGfxDevice,
                      nn::FreeFunctionWithUserData pFreeFunction,
                      void* pFreeFunctionUserData )
{
    NN_SDK_ASSERT_NOT_NULL( pRenderer );

    if ( g_PrimitiveRendererInitialized == false )
    {
        return;
    }

    if( pRenderer )
    {
        pRenderer->Finalize( pGfxDevice );
        pFreeFunction( pRenderer, pFreeFunctionUserData );
    }

    // グラフィックスリソースを初期化
    g_GraphicsResource.Finalize( pGfxDevice );
    pFreeFunction( g_pMemoryBuffer, pFreeFunctionUserData );
    g_pMemoryBuffer = nullptr;
    g_MemoryPool.Finalize( pGfxDevice );
    pFreeFunction( g_pMemoryPoolBuffer, pFreeFunctionUserData );

    g_PrimitiveRendererInitialized = false;
}

GraphicsResource* GetGraphicsResource()
{
    if ( g_PrimitiveRendererInitialized == true )
    {
        return &g_GraphicsResource;
    }
    else
    {
        return nullptr;
    }
}

void Renderer::SetDefault( Model* pModel )
{
    NN_SDK_ASSERT_NOT_NULL(pModel);
    pModel->rate = 0.f;
    pModel->u_color  = Float4One;
    pModel->u_layer  = Float4Zero;
    pModel->u_uv_src = Float2Zero;
    pModel->u_uv_size = Float2One;
    nn::util::MatrixStore( &pModel->u_userMatrix, MatrixT4x4Unit );
}

void Renderer::SetDefault( View* pView )
{
    NN_SDK_ASSERT_NOT_NULL(pView);
    nn::util::MatrixStore( &pView->u_mvp, MatrixT4x4Unit );
}

size_t Renderer::GetRequiredMemoryPoolSize( nn::gfx::Device* pGfxDevice,
                                            const RendererInfo& info )
{
    // １つのコンスタントバッファのサイズ
    size_t constantBufferSize = info.CalculateConstantBufferSize( pGfxDevice );

    nns::gfx::GpuBuffer::InitializeArg gpuBufferArg;
    gpuBufferArg.SetBufferCount(info.m_MultiBufferQuantity);
    gpuBufferArg.SetBufferSize(constantBufferSize + info.m_AdditionalBufferSize);
    gpuBufferArg.SetGpuAccessFlag(nn::gfx::GpuAccess::GpuAccess_ConstantBuffer |
        nn::gfx::GpuAccess::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

    return nns::gfx::GpuBuffer::CalculateGpuBufferPoolSize( pGfxDevice, gpuBufferArg );

}

size_t Renderer::GetMemoryPoolAlignment( nn::gfx::Device* pGfxDevice,
                                          const RendererInfo& info )
{
    // １つのコンスタントバッファのサイズ
    size_t constantBufferSize = info.CalculateConstantBufferSize( pGfxDevice );

    nns::gfx::GpuBuffer::InitializeArg gpuBufferArg;
    gpuBufferArg.SetBufferCount(info.m_MultiBufferQuantity);
    gpuBufferArg.SetBufferSize(constantBufferSize);
    gpuBufferArg.SetGpuAccessFlag(nn::gfx::GpuAccess::GpuAccess_ConstantBuffer |
        nn::gfx::GpuAccess::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

    return nns::gfx::GpuBuffer::GetGpuBufferAlignement( pGfxDevice, gpuBufferArg );
}

Renderer::Renderer():
m_pView(nullptr),
m_pModel(nullptr),
m_Color(),
m_pUserPixelShader(nullptr),
m_UserConstatnElementSize(0),
m_pMeshSet(nullptr),
m_MultiBufferQuantity(0),
m_DrawCallCount(0),
m_DrawCallCountMax(0),
m_ScreenWidth(1280),
m_ScreenHeight(720),
m_SetUserPixelModelSlot(nns::gfx::PrimitiveRenderer::ShaderSlotNone)
{
    m_Param.SetDefault();
}

Renderer::~Renderer() {}

bool Renderer::Initialize( nn::gfx::Device* pGfxDevice,
                           const RendererInfo& info,
                           GraphicsResource* pGraphicsResource,
                           nn::gfx::MemoryPool *pMemoryPool,
                           ptrdiff_t memoryPoolOffset,
                           size_t memoryPoolSize )
{
    m_pGraphicsResource = pGraphicsResource;

    // デフォルトメッシュ
    m_pMeshSet = m_pGraphicsResource->GetDefaultMeshSet();

    // 内部の定数バッファの状態をリセットします。
    ResetConstantBufferStatus();

    // デフォルトの定数バッファ
    m_MultiBufferQuantity = info.m_MultiBufferQuantity;
    m_DrawCallCountMax = info.m_DrawCallCountMax;
    m_ConstantUpdateCallCountMax[ConstantBufferType_Model] = info.m_DrawCallCountMax;
    m_ConstantUpdateCallCountMax[ConstantBufferType_View]  = info.m_ViewFunctionCallCountMax;
    m_ConstantUpdateCallCountMax[ConstantBufferType_User]  = info.m_UserFunctionCallCountMax;

    // 定数バッファを初期化します。
    // 描画コールごとに内容を書き換えるので、最初に最大描画コール分だけ確保します。
    // また、このサイズに追加バッファのサイズも足しています。
    {
        nns::gfx::GpuBuffer::InitializeArg arg;
        arg.SetBufferCount(m_MultiBufferQuantity);
        arg.SetBufferSize(info.CalculateConstantBufferSize( pGfxDevice ) + info.m_AdditionalBufferSize);
        arg.SetGpuAccessFlag(nn::gfx::GpuAccess::GpuAccess_ConstantBuffer |
            nn::gfx::GpuAccess::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

        ptrdiff_t alignedMemoryPoolOffset = nn::util::align_up( memoryPoolOffset, GetMemoryPoolAlignment( pGfxDevice, info ) );

        // メモリプールのサイズが足りない場合は、何もせずにreturnする。
        if ( memoryPoolSize < GetRequiredMemoryPoolSize( pGfxDevice, info ) )
        {
            return false;
        }

        m_GpuBuffer.Initialize(pGfxDevice, arg, pMemoryPool, alignedMemoryPoolOffset);
    }

    return true;
}

void Renderer::Finalize( nn::gfx::Device* pGfxDevice )
{

    m_DrawCallCount = 0;
    m_pMeshSet = nullptr;
    ResetConstantBufferStatus();

    m_GpuBuffer.Finalize( pGfxDevice );
}

void Renderer::SetDefaultParameters()
{
    m_Param.SetDefault();
    m_Color = Float4White;
    m_pUserPixelShader = nullptr;
    m_SetUserPixelModelSlot = nns::gfx::PrimitiveRenderer::ShaderSlotNone;
}

void Renderer::SetProjectionMatrix( const nn::util::Matrix4x4fType* pProjectionMtx )
{
    m_Param.projMatrix = *pProjectionMtx;
}

void Renderer::SetViewMatrix( const nn::util::Matrix4x3fType* pViewMtx )
{
    nn::util::MatrixConvert( &m_Param.viewMatrix, *pViewMtx );
}

void Renderer::SetModelMatrix( const nn::util::Matrix4x3fType* pModelToWorldMtx ) NN_NOEXCEPT
{
    nn::util::MatrixConvert( &m_Param.modelMatrix, *pModelToWorldMtx );
}

void Renderer::SetModelMatrix( const nn::util::Matrix4x4fType* pModelToWorldMtx ) NN_NOEXCEPT
{
    m_Param.modelMatrix = *pModelToWorldMtx;
}

void Renderer::SetLineWidth(float width)
{
    m_Param.lineWidth = width;
}

void Renderer::SetColor( const nn::util::Color4u8& color ) NN_NOEXCEPT
{
    m_Color.v[ 0 ] = static_cast<float>( color.GetR() ) / 255.f;
    m_Color.v[ 1 ] = static_cast<float>( color.GetG() ) / 255.f;
    m_Color.v[ 2 ] = static_cast<float>( color.GetB() ) / 255.f;
    m_Color.v[ 3 ] = static_cast<float>( color.GetA() ) / 255.f;
}

void Renderer::SetColor( const nn::util::Uint8x4& color ) NN_NOEXCEPT
{
    ConvertUint8x4ToFloat4( &m_Color, color );
}

void Renderer::SetUserPixelShader( nn::gfx::Shader* pPixelShader ) NN_NOEXCEPT
{
    m_pUserPixelShader = pPixelShader;
    if ( pPixelShader != NULL  )
    {
        m_SetUserPixelModelSlot = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage::ShaderStage_Pixel,
                                                                  nn::gfx::ShaderInterfaceType_ConstantBuffer,
                                                                  "Model" );
    }
    else
    {
        m_SetUserPixelModelSlot = nns::gfx::PrimitiveRenderer::ShaderSlotNone;
    }

}

void Renderer::SetViewConstantBuffer( View* pPodData ) NN_NOEXCEPT
{
    ConstantBufferType type = ConstantBufferType_View;
    m_pView = UpdateConstantBuffer<View>( type, pPodData );
}

void Renderer::SetModelConstantBuffer( Model* pPodData ) NN_NOEXCEPT
{
    ConstantBufferType type = ConstantBufferType_Model;
    m_pModel = UpdateConstantBuffer<Model>( type, pPodData );
}

void Renderer::SetUserConstantBuffer( nn::gfx::GpuAddress* pGpuAddress, void* pPodData, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pGpuAddress );

    ConstantBufferType type = ConstantBufferType_User;
    NN_SDK_ASSERT(m_ConstantUpdateCallCount[type] < m_ConstantUpdateCallCountMax[type],
                    "[type %d] UpdateCall(%d) >= UpdateCallCountMax(%d)",
                    m_ConstantUpdateCallCount[type], m_ConstantUpdateCallCountMax[type]);

    if( void* pData = CreateUniformBlock(size) )
    {
        std::memcpy(pData, pPodData, size);
        UpdateConstantBufferPointer(type, pData);
        m_ConstantUpdateCallCount[type]++;
    }

    m_GpuBuffer.GetGpuAddress( pGpuAddress, m_ConstantMemoryPointer[type] );
}

void Renderer::ResetConstantBufferStatus() NN_NOEXCEPT
{
    for( int idxBuffer = 0; idxBuffer < ConstantBufferType_CountMax; ++idxBuffer )
    {
        m_ConstantMemoryPointer[ idxBuffer ] = nullptr;
        m_ConstantUpdateCallCount[ idxBuffer ] = 0;
    }
    m_pView = nullptr;
    m_pModel = nullptr;
}

void* Renderer::CreateUniformBlock(size_t size) NN_NOEXCEPT
{
    if ( m_GpuBuffer.GetAllocatedSize() + size > m_GpuBuffer.GetBufferSize() )
    {
        return nullptr;
    }

    void* pointer = m_GpuBuffer.Allocate(size);
    return pointer;
}

void Renderer::UpdateConstantBufferPointer(ConstantBufferType type, void* pointer)
{
    m_ConstantMemoryPointer[type] = pointer;
}

void Renderer::SetConstantBuffer( nn::gfx::CommandBuffer* pCommandBuffer,
                                  int slot,
                                  nn::gfx::ShaderStage shaderStage,
                                  void* pointer,
                                  size_t memorySize )
{
    NN_SDK_ASSERT(pointer != nullptr);
    if( slot == ShaderSlotNone )
    {
        return;
    }

    nn::gfx::GpuAddress gpuAddress;
    m_GpuBuffer.GetGpuAddress( &gpuAddress, pointer );

    // データサイズは、実際のPODを設定します。
    pCommandBuffer->SetConstantBuffer( slot, shaderStage, gpuAddress, memorySize );
}

void Renderer::MakeModelConstantBuffer(
                    Model* pOutModel,
                    const nn::util::Vector3fType& scale,
                    const nn::util::Vector3fType& trans )
{
    MakeModelConstantBuffer(pOutModel, scale, trans, m_Color );
}

void Renderer::MakeModelConstantBuffer(
                    Model* pOutModel,
                    const nn::util::Vector3fType& scale,
                    const nn::util::Vector3fType& trans,
                    const nn::util::Float4& color )
{
    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixSetScale( &modelMatrix, scale );
    nn::util::MatrixSetTranslate( &modelMatrix, trans );

    nn::util::Matrix4x4fType modelMtx4x4f;
    nn::util::MatrixConvert(&modelMtx4x4f, modelMatrix);
    nn::util::MatrixStore(&pOutModel->u_userMatrix, modelMtx4x4f);

    pOutModel->u_color = color;
}

void Renderer::MakeModelConstantBufferForQuad(
                    Model* pOutModel,
                    const nn::util::Vector3fType& scale,
                    const nn::util::Vector3fType& trans,
                    const nn::util::Float4& color )
{
    // プリミティブを nn::util::PrimitiveShape に乗り換えた際に原点が変わっており
    // 互換性維持のため原点調整行列を設定します。
    nn::util::Matrix4x3fType arrangeMatrix;

    nn::util::Vector3fType  arrangeScale
        = NN_UTIL_VECTOR_3F_INITIALIZER( 0.5f, 0.5f, 1.0f );
    nn::util::Vector3fType  arrangeTranslate
        = NN_UTIL_VECTOR_3F_INITIALIZER( 0.5f, 0.5f, 0.0f );

    nn::util::MatrixSetScale(&arrangeMatrix, arrangeScale);
    nn::util::MatrixSetTranslate(&arrangeMatrix, arrangeTranslate);

    nn::util::Matrix4x3fType originalMatrix;
    nn::util::MatrixSetScale( &originalMatrix, scale );
    nn::util::MatrixSetTranslate( &originalMatrix, trans );

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixMultiply( &modelMatrix, arrangeMatrix, originalMatrix);

    nn::util::Matrix4x4fType modelMtx4x4f;
    nn::util::MatrixConvert(&modelMtx4x4f, modelMatrix);

    nn::util::MatrixStore(&pOutModel->u_userMatrix, modelMtx4x4f);

    pOutModel->u_color = color;
}


void Renderer::SetupShader( nn::gfx::CommandBuffer* pCommandBuffer,
                            Shader*                 pDefaultShader,
                            const nn::gfx::Shader*  pUserPixelShader,
                            bool interleaved )
{
    // 頂点シェーダ
    pCommandBuffer->SetShader( pDefaultShader->GetVertexShader(), nn::gfx::ShaderStageBit_Vertex );

    // フラグメントシェーダ
    if( pUserPixelShader != nullptr  )
    {
        pCommandBuffer->SetShader( pUserPixelShader, nn::gfx::ShaderStageBit_Pixel );
    }
    else
    {
        pCommandBuffer->SetShader( pDefaultShader->GetPixelShader(), nn::gfx::ShaderStageBit_Pixel );
    }

    // 頂点ステート
    pCommandBuffer->SetVertexState( interleaved ? pDefaultShader->GetInterleavedVertexState() : pDefaultShader->GetVertexState() );
}

void Renderer::Update(int bufferIndex) NN_NOEXCEPT
{
    // 描画コール数を初期化します。
    m_DrawCallCount = 0;

    // 定数バッファの状態をリセットします。
    ResetConstantBufferStatus();

    m_GpuBuffer.Unmap();
    m_GpuBuffer.Map(bufferIndex);
}

void Renderer::UpdateGpuView() NN_NOEXCEPT
{
    View view;
    nn::util::Matrix4x4fType mvp;
    nn::util::MatrixMultiply( &mvp, m_Param.modelMatrix, m_Param.viewMatrix);
    nn::util::MatrixMultiply( &mvp, mvp,  m_Param.projMatrix );
    nn::util::MatrixStore( &view.u_mvp, mvp );
    nn::util::MatrixStore( &view.u_view, m_Param.viewMatrix );
    nn::util::MatrixStore( &view.u_model, m_Param.modelMatrix );

    SetViewConstantBuffer( &view );
}

void Renderer::DrawPrimitiveInternal( nn::gfx::PrimitiveTopology topology,
                                      ShaderVariation shaderVariation,
                                      nn::gfx::CommandBuffer* pCommandBuffer,
                                      const PrimitiveMesh* pMesh,
                                      Model* pModel,
                                      const int indexBase,
                                      const int numUserIndices,
                                      const int userBaseVertex,
                                      const SamplerType samplerType,
                                      const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                      const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pMesh);

    // pMeshがnullだったら、何もせずにreturnする。
    if (pMesh == nullptr)
    {
        return;
    }

    // インデックスバッファ
    NN_SDK_ASSERT(pMesh->IsValidIndexBuffer());

    // インデックスバッファが不正だったら、何もせずにreturnする。
    if (pMesh->IsValidIndexBuffer() == false)
    {
        return;
    }

    // ドローコール数をチェックします。
    if (m_DrawCallCount >= m_DrawCallCountMax)
    {
        NN_SDK_LOG("Draw Call(%d) >= limit(%d), Please Call Update()",
            m_DrawCallCount, m_DrawCallCountMax);
        return;
    }
    UpdateGpuView();

    // 定数バッファ
    SetModelConstantBuffer(pModel);

    // シェーダバリエーションからデフォルトシェーダを取得します。
    Shader* pDefaultShader = m_pGraphicsResource->GetDefaultShader(shaderVariation);

    // シェーダ
    SetupShader(pCommandBuffer, pDefaultShader, m_pUserPixelShader, false);

    // 頂点バッファ
    nn::gfx::GpuAddress vertexGpuAddress;

    int bufferIndex = 0;
    uint32_t vertexFormat = Shader::GetDefaultVertexFormat(shaderVariation);

    for (int idx = 0; idx < VertexAttribute_CountMax; ++idx)
    {
        // 使用されていないアトリビュートはスキップして頂点バッファを前から詰める
        if ((vertexFormat & (1 << idx)) == 0)
        {
            continue;
        }

        VertexAttribute attrib = static_cast<VertexAttribute>(idx);
        if (!pMesh->IsValidVertexBuffer(attrib))
        {
            continue;
        }
        const size_t stride = pMesh->GetVertexStride(attrib);
        const size_t size = pMesh->GetVertexMemorySize(attrib);
        pMesh->GetVertexBufferGpuAddress(attrib, &vertexGpuAddress);
        pCommandBuffer->SetVertexBuffer(bufferIndex, vertexGpuAddress, stride, size);
        bufferIndex++;
    }

    // 定数バッファのロード
    int slotView = pDefaultShader->slotView;
    int slotModelVs = pDefaultShader->slotModel[0];
    int slotModelPs = pDefaultShader->slotModel[1];
    if ( m_pUserPixelShader )
    {
        if ( m_SetUserPixelModelSlot != nns::gfx::PrimitiveRenderer::ShaderSlotNone )
        {
            slotModelPs = m_SetUserPixelModelSlot;
        }
        else
        {
            slotModelPs = nns::gfx::PrimitiveRenderer::ShaderSlotNone;
        }
    }

    // SetViewConstantBuffer() / UpdateGpuView()で、更新済みとします。
    SetConstantBuffer(pCommandBuffer,
        slotView,
        nn::gfx::ShaderStage_Vertex,
        m_ConstantMemoryPointer[ConstantBufferType_View],
        sizeof(View));

    // SetModelConstantBuffer() で、更新済みとします。
    SetConstantBuffer(pCommandBuffer,
        slotModelVs,
        nn::gfx::ShaderStage_Vertex,
        m_ConstantMemoryPointer[ConstantBufferType_Model],
        sizeof(Model));

    // UserPixelShaderにModelが無い場合はコンスタントバッファを設定しない
    if ( slotModelPs != nns::gfx::PrimitiveRenderer::ShaderSlotNone )
    {
        SetConstantBuffer(pCommandBuffer,
            slotModelPs,
            nn::gfx::ShaderStage_Pixel,
            m_ConstantMemoryPointer[ConstantBufferType_Model],
            sizeof(Model));
    }
    /*
    // ユーザ定義の定数バッファを設定します。
    // SetUserConstantBuffer()で、更新済みとします。
    if( m_ConstantMemoryOffset[ConstantBufferType_User] != InvalidOffset )
    {
    int slotUserVs = pDefaultShader->slotUser[0];
    int slotUserPs = pDefaultShader->slotUser[1];
    SetConstantBuffer( pCommandBuffer,
    slotUserVs,
    nn::gfx::ShaderStage_Vertex,
    m_ConstantMemoryOffset[ConstantBufferType_User],
    Constants_UserConstantBufferDataSizeMax);

    SetConstantBuffer( pCommandBuffer,
    slotUserPs,
    nn::gfx::ShaderStage_Pixel,
    m_ConstantMemoryOffset[ConstantBufferType_User],
    Constants_UserConstantBufferDataSizeMax);
    }
    */
    // サンプラ
    int idxSampler = ShaderSlotNone;
    if (SamplerType_CountMax != samplerType &&
        pTextureViewSlot != nullptr && pSamplerSlot != nullptr)
    {
        idxSampler = pDefaultShader->slotSamplerPs[samplerType];
    }

    // テクスチャとサンプラ
    if (ShaderSlotNone != idxSampler)
    {
        pCommandBuffer->SetTextureAndSampler(idxSampler,
            nn::gfx::ShaderStage_Pixel,
            *pTextureViewSlot, *pSamplerSlot);
    }

    // ドローコール
    const int numIndices = (0 < numUserIndices) ? numUserIndices : pMesh->GetIndexCount();
    NN_SDK_ASSERT(0 < numIndices);

    nn::gfx::GpuAddress indexGpuAddress;
    pMesh->GetIndexBufferGpuAddress(&indexGpuAddress);
    ptrdiff_t indexOffset = indexBase * sizeof(uint32_t);
    indexGpuAddress.Offset(indexOffset);
    pCommandBuffer->DrawIndexed(topology,
        nn::gfx::IndexFormat_Uint32, indexGpuAddress, numIndices, userBaseVertex);

    // 描画コール数をカウントします。
    m_DrawCallCount++;
}

void Renderer::DrawPrimitiveInternal(nn::gfx::PrimitiveTopology topology,
                                    ShaderVariation shaderVariation,
                                    nn::gfx::CommandBuffer * pCommandBuffer,
                                    const nn::gfx::util::PrimitiveShape* pShape,
                                    Model * pModel,
                                    const int indexBase,
                                    const int numUserIndices,
                                    const int userBaseVertex,
                                    const SamplerType samplerType,
                                    const nn::gfx::DescriptorSlot * pTextureViewSlot,
                                    const nn::gfx::DescriptorSlot * pSamplerSlot) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pShape);

    // ドローコール数をチェックします。
    if (m_DrawCallCount >= m_DrawCallCountMax)
    {
        NN_SDK_LOG("Draw Call(%d) >= limit(%d), Please Call Update()",
            m_DrawCallCount, m_DrawCallCountMax);
        return;
    }
    UpdateGpuView();

    // 定数バッファ
    SetModelConstantBuffer(pModel);

    // シェーダバリエーションからデフォルトシェーダを取得します。
    Shader* pDefaultShader = m_pGraphicsResource->GetDefaultShader(shaderVariation);

    // シェーダ
    SetupShader(pCommandBuffer, pDefaultShader, m_pUserPixelShader, true);

    GpuBuffer*  pGpuBuffer = m_pGraphicsResource->GetGpuBuffer();

    // 頂点バッファ
    nn::gfx::GpuAddress vertexGpuAddress;
    const size_t stride = pShape->GetStride();
    const size_t size = pShape->GetVertexBufferSize();
    pGpuBuffer->GetGpuAddress(&vertexGpuAddress, pShape->GetVertexBuffer());
    pCommandBuffer->SetVertexBuffer(0, vertexGpuAddress, stride, size);

    // 定数バッファのロード
    int slotView = pDefaultShader->slotView;
    int slotModelVs = pDefaultShader->slotModel[0];
    int slotModelPs = pDefaultShader->slotModel[1];
    if ( m_pUserPixelShader )
    {
        if ( m_SetUserPixelModelSlot != nns::gfx::PrimitiveRenderer::ShaderSlotNone )
        {
            slotModelPs = m_SetUserPixelModelSlot;
        }
        else
        {
            slotModelPs = nns::gfx::PrimitiveRenderer::ShaderSlotNone;
        }
    }

    // SetViewConstantBuffer() / UpdateGpuView()で、更新済みとします。
    SetConstantBuffer(pCommandBuffer,
        slotView,
        nn::gfx::ShaderStage_Vertex,
        m_ConstantMemoryPointer[ConstantBufferType_View],
        sizeof(View));

    // SetModelConstantBuffer() で、更新済みとします。
    SetConstantBuffer(pCommandBuffer,
        slotModelVs,
        nn::gfx::ShaderStage_Vertex,
        m_ConstantMemoryPointer[ConstantBufferType_Model],
        sizeof(Model));

    // UserPixelShaderにModelが無い場合はコンスタントバッファを設定しない
    if ( slotModelPs != nns::gfx::PrimitiveRenderer::ShaderSlotNone )
    {
        SetConstantBuffer(pCommandBuffer,
            slotModelPs,
            nn::gfx::ShaderStage_Pixel,
            m_ConstantMemoryPointer[ConstantBufferType_Model],
            sizeof(Model));
    }
    /*
    // ユーザ定義の定数バッファを設定します。
    // SetUserConstantBuffer()で、更新済みとします。
    if( m_ConstantMemoryOffset[ConstantBufferType_User] != InvalidOffset )
    {
    int slotUserVs = pDefaultShader->slotUser[0];
    int slotUserPs = pDefaultShader->slotUser[1];
    SetConstantBuffer( pCommandBuffer,
    slotUserVs,
    nn::gfx::ShaderStage_Vertex,
    m_ConstantMemoryOffset[ConstantBufferType_User],
    Constants_UserConstantBufferDataSizeMax);

    SetConstantBuffer( pCommandBuffer,
    slotUserPs,
    nn::gfx::ShaderStage_Pixel,
    m_ConstantMemoryOffset[ConstantBufferType_User],
    Constants_UserConstantBufferDataSizeMax);
    }
    */
    // サンプラ
    int idxSampler = ShaderSlotNone;
    if (SamplerType_CountMax != samplerType &&
        pTextureViewSlot != nullptr && pSamplerSlot != nullptr)
    {
        idxSampler = pDefaultShader->slotSamplerPs[samplerType];
    }

    // テクスチャとサンプラ
    if (ShaderSlotNone != idxSampler)
    {
        pCommandBuffer->SetTextureAndSampler(idxSampler,
            nn::gfx::ShaderStage_Pixel,
            *pTextureViewSlot, *pSamplerSlot);
    }

    // ドローコール
    const int numIndices = (0 < numUserIndices) ? numUserIndices : pShape->GetIndexCount();
    NN_SDK_ASSERT(0 < numIndices);

    nn::gfx::GpuAddress indexGpuAddress;
    pGpuBuffer->GetGpuAddress(&indexGpuAddress, pShape->GetIndexBuffer());
    ptrdiff_t indexOffset = indexBase * pShape->GetIndexBufferSize() / pShape->GetIndexCount();
    indexGpuAddress.Offset(indexOffset);
    pCommandBuffer->DrawIndexed(topology,
        pShape->GetIndexBufferFormat(), indexGpuAddress, numIndices, userBaseVertex);

    // 描画コール数をカウントします。
    m_DrawCallCount++;
}

void Renderer::DrawPrimitiveInternal( ShapeType shapeType,
                                      ShaderVariation shaderVariation,
                                      nn::gfx::CommandBuffer* pCommandBuffer,
                                      Model* pModel,
                                      const SamplerType samplerType,
                                      const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                      const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    const nn::gfx::util::PrimitiveShape* pShape = m_pMeshSet->GetShape( shapeType );

    if ( pShape != NULL )
    {
        DrawPrimitiveInternal( pShape->GetPrimitiveTopology(), shaderVariation, pCommandBuffer,
                               pShape, pModel, 0, 0, 0,
                               samplerType, pTextureViewSlot, pSamplerSlot );
    }
    else
    {
        const PrimitiveMesh* pMesh = m_pMeshSet->Get( shapeType );
        NN_SDK_ASSERT_NOT_NULL( pMesh );

        // pShape がnullだったら、何もせずにreturnする。
        if ( pMesh == nullptr )
        {
            return;
        }

        nn::gfx::PrimitiveTopology topology = TopologyTable[ shapeType ];

        DrawPrimitiveInternal( topology, shaderVariation, pCommandBuffer,
                               pMesh, pModel, 0, 0, 0,
                               samplerType, pTextureViewSlot, pSamplerSlot );
    }
}

// 以下は、各メッシュ描画関数

void Renderer::DrawPoint( nn::gfx::CommandBuffer* pCommandBuffer,
                          const nn::util::Vector3fType& center ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    nn::util::Vector3fType scale = {1.f, 1.f, 1.f};
    MakeModelConstantBuffer( &model, scale, center, m_Color );
    DrawPrimitiveInternal( ShapeType_Point, ShaderVariation_P, pCommandBuffer, &model );
}

void Renderer::DrawLine( nn::gfx::CommandBuffer* pCommandBuffer,
                const nn::util::Vector3fType& begin,
                const nn::util::Vector3fType& end ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    nn::util::Vector3fType line;
    nn::util::VectorSubtract( &line, end, begin );

    nn::util::MatrixT4x3fType modelMatrix;
    nn::util::MatrixIdentity( &modelMatrix );
    nn::util::MatrixSetTranslate( &modelMatrix, begin );

    nn::util::MatrixSetAxisX( &modelMatrix, line );

    nn::util::MatrixT4x4fType modelMtx4x4f;
    nn::util::MatrixConvert(&modelMtx4x4f, modelMatrix);
    nn::util::MatrixStore(&model.u_userMatrix, modelMtx4x4f);

    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
#endif

    DrawPrimitiveInternal( ShapeType_Line, ShaderVariation_P, pCommandBuffer, &model );
}

void Renderer::DrawTriangle( nn::gfx::CommandBuffer* pCommandBuffer,
                             const Surface surface,
                             const nn::util::Vector3fType& center,
                             const nn::util::Vector3fType& size ) NN_NOEXCEPT
{
    DrawTriangleImpl(pCommandBuffer, surface, center, size, nullptr, nullptr);
}

void Renderer::DrawTriangle( nn::gfx::CommandBuffer* pCommandBuffer,
                             const nn::util::Vector3fType& center,
                             const nn::util::Vector3fType& size,
                             const nn::gfx::DescriptorSlot& textureViewSlot,
                             const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawTriangleImpl(pCommandBuffer, Surface_Solid, center, size, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawTriangleImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                                 const Surface surface,
                                 const nn::util::Vector3fType& center,
                                 const nn::util::Vector3fType& size,
                                 const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                 const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    MakeModelConstantBuffer( &model, size, center, m_Color );

    const ShapeType shapeType = ( surface == Surface_Wired ) ? ShapeType_TriangleWired : ShapeType_Triangle;
    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;

    ShaderVariation shaderVariation;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
    }
#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot );
}


void Renderer::DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                              const nn::gfx::PrimitiveTopology topology,
                              const PrimitiveMesh* pUserMesh,
                              const int indexBase,
                              const int indexCount,
                              const int vertexBase )

{
    Model model;
    SetDefault( &model );
    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    if ( topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
        topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineListAdjacency ||
        topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip ||
        topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStripAdjacency )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    ShaderVariation shaderVariation;
    const VertexFormat vertexFormat = pUserMesh->GetVertexFormat();

    if ( vertexFormat & VertexFormat_Color )
    {
        shaderVariation = ShaderVariation_PUvC;
    }
    else
    {
        shaderVariation = ShaderVariation_P;
    }

    DrawPrimitiveInternal( topology, shaderVariation, pCommandBuffer, pUserMesh, &model,
                           indexBase, indexCount, vertexBase,
                           SamplerType_CountMax, nullptr, nullptr );
}

void Renderer::DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                             const nn::gfx::PrimitiveTopology topology,
                             const PrimitiveMesh* pUserMesh,
                             const int indexBase,
                             const int indexCount,
                             const int vertexBase,
                             const nn::gfx::DescriptorSlot& textureViewSlot,
                             const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );
    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    if ( topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
        topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineListAdjacency ||
        topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip ||
        topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStripAdjacency )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    ShaderVariation shaderVariation;
    const VertexFormat vertexFormat = pUserMesh->GetVertexFormat();

    if ( vertexFormat & VertexFormat_Color )
    {
        shaderVariation = ShaderVariation_PUvC;
    }
    else
    {
        shaderVariation = ShaderVariation_PUv;
    }

    DrawPrimitiveInternal( topology, shaderVariation, pCommandBuffer, pUserMesh, &model, indexBase, indexCount, vertexBase, SamplerType_Texture, &textureViewSlot, &samplerSlot );
}

void Renderer::DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                             const nn::gfx::PrimitiveTopology topology,
                             const PrimitiveMesh* pUserMesh ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );
    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    if ( topology== nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
         topology== nn::gfx::PrimitiveTopology::PrimitiveTopology_LineListAdjacency ||
         topology== nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip ||
         topology== nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStripAdjacency )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    ShaderVariation shaderVariation;
    const VertexFormat vertexFormat = pUserMesh->GetVertexFormat();

    if ( vertexFormat & VertexFormat_Color)
    {
        shaderVariation = ShaderVariation_PC;
    }
    else
    {
        shaderVariation = ShaderVariation_P;
    }

    DrawPrimitiveInternal( topology, shaderVariation, pCommandBuffer, pUserMesh, &model,
                           0, 0, 0, SamplerType_CountMax, nullptr, nullptr );
}

void Renderer::DrawUserMesh( nn::gfx::CommandBuffer* pCommandBuffer,
                             const nn::gfx::PrimitiveTopology topology,
                             const PrimitiveMesh* pUserMesh,
                             const nn::gfx::DescriptorSlot& textureViewSlot,
                             const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );
    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    if ( topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
         topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineListAdjacency ||
         topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip ||
         topology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStripAdjacency )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    ShaderVariation shaderVariation;
    const VertexFormat vertexFormat = pUserMesh->GetVertexFormat();

    if ( vertexFormat & VertexFormat_Color )
    {
        shaderVariation = ShaderVariation_PUvC;
    }
    else
    {
        shaderVariation = ShaderVariation_PUv;
    }

    DrawPrimitiveInternal( topology, shaderVariation, pCommandBuffer, pUserMesh, &model, 0, 0, 0, SamplerType_Texture, &textureViewSlot, &samplerSlot );
}



void Renderer::DrawSphere(nn::gfx::CommandBuffer* pCommandBuffer,
                               const Surface surface,
                               const Subdiv subdiv,
                               const nn::util::Vector3fType& center,
                               const float diameter) NN_NOEXCEPT
{
    DrawSphereImpl(pCommandBuffer, surface, subdiv, center, diameter, nullptr, nullptr);
}

void Renderer::DrawSphere(nn::gfx::CommandBuffer* pCommandBuffer,
                          const Subdiv subdiv,
                          const nn::util::Vector3fType& center,
                          const float diameter,
                          const nn::gfx::DescriptorSlot& textureViewSlot,
                          const nn::gfx::DescriptorSlot& samplerSlot) NN_NOEXCEPT
{
    DrawSphereImpl(pCommandBuffer, Surface_Solid, subdiv, center, diameter, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawSphereImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                               const Surface surface,
                               const Subdiv subdiv,
                               const nn::util::Vector3fType& center,
                               const float diameter,
                               const nn::gfx::DescriptorSlot* pTextureViewSlot,
                               const nn::gfx::DescriptorSlot* pSamplerSlot) NN_NOEXCEPT
{
    static const ShapeType s_TexturedShapeTable[2] =
    {
        ShapeType_SphereTexturedCoarse,
        ShapeType_SphereTextured,
    };

    static const ShapeType s_ShapeTable[3][2] =
    {
        { ShapeType_SphereWiredCoarse, ShapeType_SphereWired },
        { ShapeType_SphereCoarse, ShapeType_Sphere },
        { ShapeType_SphereCoarse, ShapeType_Sphere },
    };

    ShapeType shapeType;
    ShaderVariation shaderVariation;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType = s_TexturedShapeTable[subdiv];
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = s_ShapeTable[surface][subdiv];
    }

    // 定数バッファ
    nn::util::Vector3fType scale
        = NN_UTIL_VECTOR_3F_INITIALIZER( diameter * 0.5f, diameter * 0.5f, diameter * 0.5f );
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    MakeModelConstantBuffer( &model, scale, center, m_Color );

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;
    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot);
}

void Renderer::DrawCircle( nn::gfx::CommandBuffer* pCommandBuffer,
                    const Surface surface,
                    const Subdiv subdiv,
                    const nn::util::Vector3fType& center,
                    float diameter) NN_NOEXCEPT
{
    DrawCircleImpl(pCommandBuffer, surface, subdiv, center, diameter, nullptr, nullptr);
}

void Renderer::DrawCircle( nn::gfx::CommandBuffer* pCommandBuffer,
                    const Subdiv subdiv,
                    const nn::util::Vector3fType& center,
                    float diameter,
                    const nn::gfx::DescriptorSlot& textureViewSlot,
                    const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawCircleImpl(pCommandBuffer, Surface_Solid, subdiv, center, diameter, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawCircleImpl(nn::gfx::CommandBuffer* pCommandBuffer,
                                const Surface surface,
                                const Subdiv subdiv,
                                const nn::util::Vector3fType& center,
                                float diameter,
                                const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    static const ShapeType s_TexturedShapeTable[2] =
    {
        ShapeType_CircleTexturedCoarse,
        ShapeType_CircleTextured
    };

    static const ShapeType s_ShapeTable[3][2] =
    {
        { ShapeType_CircleWiredCoarse, ShapeType_CircleWired },
        { ShapeType_CircleCoarse, ShapeType_Circle },
        { ShapeType_CircleCoarse, ShapeType_Circle },
    };

    ShapeType shapeType;
    ShaderVariation shaderVariation;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType = s_TexturedShapeTable[subdiv];
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = s_ShapeTable[surface][subdiv];
    }

    // 定数バッファ
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, diameter, diameter, diameter);

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixSetScale( &modelMatrix, scale );
    nn::util::MatrixSetAxisW( &modelMatrix, center );

    ConvertMatrix4x3ToFloatT4x4( &model.u_userMatrix, modelMatrix );

    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;
    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot);
}

void Renderer::DrawCube(nn::gfx::CommandBuffer* pCommandBuffer,
                        const Surface surface,
                        const nn::util::Vector3fType& center,
                        const nn::util::Vector3fType& size) NN_NOEXCEPT
{
    DrawCubeImpl(pCommandBuffer, surface, center, size, nullptr, nullptr);
}

void Renderer::DrawCube(nn::gfx::CommandBuffer* pCommandBuffer,
                        const nn::util::Vector3fType& center,
                        const nn::util::Vector3fType& size,
                        const nn::gfx::DescriptorSlot& textureViewSlot,
                        const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawCubeImpl(pCommandBuffer, Surface_Solid, center, size, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawCubeImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                             const Surface surface,
                             const nn::util::Vector3fType& center,
                             const nn::util::Vector3fType& size,
                             const nn::gfx::DescriptorSlot* pTextureViewSlot,
                             const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    MakeModelConstantBuffer( &model, size, center, m_Color );

    ShapeType shapeType;
    ShaderVariation shaderVariation;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType       = ShapeType_CubeTextured;
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = (surface == Surface_Wired) ? ShapeType_CubeWired : ShapeType_Cube;
    }

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;
    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot);
}

void Renderer::DrawQuad( nn::gfx::CommandBuffer* pCommandBuffer,
                         const nn::util::Vector3fType& center,
                         const nn::util::Vector3fType& size ) NN_NOEXCEPT
{
    // 定数バッファ
    Model model;
    SetDefault( &model );
    MakeModelConstantBufferForQuad( &model, size, center, m_Color );
    DrawPrimitiveInternal( ShapeType_Quad, ShaderVariation_PN, pCommandBuffer, &model, SamplerType_CountMax, nullptr, nullptr );
}

void Renderer::DrawQuad( nn::gfx::CommandBuffer* pCommandBuffer,
                         const nn::util::Vector3fType& center,
                         const nn::util::Vector3fType& size,
                         const nn::gfx::DescriptorSlot& textureViewSlot,
                         const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    MakeModelConstantBufferForQuad( &model, size, center, m_Color );

    DrawPrimitiveInternal( ShapeType_QuadTextured, ShaderVariation_PUv, pCommandBuffer, &model, SamplerType_Texture, &textureViewSlot, &samplerSlot );
}

void Renderer::DrawQuadTextureArray( nn::gfx::CommandBuffer* pCommandBuffer,
                                     const nn::util::Vector3fType& center,
                                     const nn::util::Vector3fType& size,
                                     const nn::gfx::DescriptorSlot& textureViewSlot,
                                     const nn::gfx::DescriptorSlot& samplerSlot,
                                     const float lod ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    MakeModelConstantBufferForQuad( &model, size, center, m_Color );
    model.u_layer.x = lod;
    DrawPrimitiveInternal( ShapeType_QuadTextured, ShaderVariation_TextureArray, pCommandBuffer, &model, SamplerType_TextureArray, &textureViewSlot, &samplerSlot );
}

void Renderer::DrawQuadCubemap( nn::gfx::CommandBuffer* pCommandBuffer,
                                const nn::util::Vector3fType& center,
                                const nn::util::Vector3fType& size,
                                const nn::gfx::DescriptorSlot& textureViewSlot,
                                const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    MakeModelConstantBufferForQuad( &model, size, center, m_Color );
    DrawPrimitiveInternal( ShapeType_QuadTextured, ShaderVariation_Cubemap, pCommandBuffer, &model, SamplerType_TextureCubemap, &textureViewSlot, &samplerSlot );
}

void Renderer::DrawScreenQuad( nn::gfx::CommandBuffer* pCommandBuffer,
                               const nn::gfx::DescriptorSlot& textureViewSlot,
                               const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    const nn::util::Vector3fType size
        = NN_UTIL_VECTOR_3F_INITIALIZER( 1.f, 1.f, 1.f );
    const nn::util::Vector3fType center
        = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, 0.f, 0.f );

    MakeModelConstantBuffer( &model, size, center, m_Color );
    DrawPrimitiveInternal( ShapeType_ScreenQuad, ShaderVariation_PUv, pCommandBuffer, &model, SamplerType_Texture, &textureViewSlot, &samplerSlot );
}

void Renderer::DrawScreenQuadYFlip( nn::gfx::CommandBuffer* pCommandBuffer,
                                    const nn::gfx::DescriptorSlot& textureViewSlot,
                                    const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // Y 軸を反転
    const nn::util::Vector3fType size
        = NN_UTIL_VECTOR_3F_INITIALIZER( 1.f, -1.f, 1.f );
    const nn::util::Vector3fType center
        = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, 0.f, 0.f );

    MakeModelConstantBuffer( &model, size, center, m_Color );
    DrawPrimitiveInternal( ShapeType_ScreenQuad, ShaderVariation_PUv, pCommandBuffer, &model, SamplerType_Texture, &textureViewSlot, &samplerSlot );
}

void Renderer::Draw2DRect( nn::gfx::CommandBuffer* pCommandBuffer,
                           const float x,
                           const float y,
                           const float width,
                           const float height ) NN_NOEXCEPT
{
    // 定数バッファ
    Model model;
    SetDefault( &model );
    const nn::util::Vector3fType size
        = NN_UTIL_VECTOR_3F_INITIALIZER(2.f * width / m_ScreenWidth, 2.f * height / m_ScreenHeight, 2.f); // [-1,1]x[-1,1]
    const nn::util::Vector3fType center
        = NN_UTIL_VECTOR_3F_INITIALIZER(-1.f + 2.f * x / m_ScreenWidth, 1.f - 2.f * (y + height) / m_ScreenHeight, 0.f);

    MakeModelConstantBufferForQuad( &model, size, center, m_Color );
    DrawPrimitiveInternal( ShapeType_Quad, ShaderVariation_PN, pCommandBuffer, &model, SamplerType_Texture, nullptr, nullptr );
}

void Renderer::Draw2DFrame( nn::gfx::CommandBuffer* pCommandBuffer,
                            const float x,
                            const float y,
                            const float width,
                            const float height) NN_NOEXCEPT
{
    Draw2DLine(pCommandBuffer, x, y, x + width, y);
    Draw2DLine(pCommandBuffer, x, y + height, x + width, y + height);

    Draw2DLine(pCommandBuffer, x, y, x, y + height);
    Draw2DLine(pCommandBuffer, x + width, y, x + width, y + height);
}

void Renderer::Draw2DRect( nn::gfx::CommandBuffer* pCommandBuffer,
                    const float x,
                    const float y,
                    const float width,
                    const float height,
                    const nn::gfx::DescriptorSlot& textureViewSlot,
                    const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    const nn::util::Vector3fType size
        = NN_UTIL_VECTOR_3F_INITIALIZER(2.f * width / m_ScreenWidth, 2.f * height / m_ScreenHeight, 2.f); // [-1,1]x[-1,1]
    const nn::util::Vector3fType center
        = NN_UTIL_VECTOR_3F_INITIALIZER(-1.f + 2.f * x / m_ScreenWidth, 1.f - 2.f * (y + height) / m_ScreenHeight, 0.f);

    MakeModelConstantBufferForQuad( &model, size, center, m_Color );

    DrawPrimitiveInternal( ShapeType_ScreenQuad, ShaderVariation_PUv, pCommandBuffer, &model, SamplerType_Texture, &textureViewSlot, &samplerSlot );
}

void Renderer::Draw2DLine( nn::gfx::CommandBuffer* pCommandBuffer,
                           const float beginX,
                           const float beginY,
                           const float endX,
                           const float endY ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    const nn::util::Vector3fType begin
        = NN_UTIL_VECTOR_3F_INITIALIZER( -1.f + 2.f * beginX / m_ScreenWidth, 1.f - 2.f * beginY / m_ScreenHeight, 0.f );
    const nn::util::Vector3fType end
        = NN_UTIL_VECTOR_3F_INITIALIZER( -1.f + 2.f * endX / m_ScreenWidth, 1.f - 2.f * endY / m_ScreenHeight, 0.f );

    nn::util::Vector3fType line;
    nn::util::VectorSubtract( &line, end, begin );

    nn::util::MatrixT4x3fType modelMatrix;
    nn::util::MatrixIdentity( &modelMatrix );
    nn::util::MatrixSetTranslate( &modelMatrix, begin );

    nn::util::MatrixSetAxisX( &modelMatrix, line );

    nn::util::MatrixT4x4fType modelMtx4x4f;
    nn::util::MatrixConvert(&modelMtx4x4f, modelMatrix);
    nn::util::MatrixStore(&model.u_userMatrix, modelMtx4x4f);

    model.u_color = m_Color;

#if NN_GFX_IS_TARGET_NVN
    nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
#endif

    DrawPrimitiveInternal( ShapeType_Line, ShaderVariation_P, pCommandBuffer, &model, SamplerType_Texture, nullptr, nullptr );
}

void Renderer::DrawCone( nn::gfx::CommandBuffer* pCommandBuffer,
                    const Surface surface,
                    const nn::util::Vector3fType& center,
                    float radius,
                    float height ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );

    DrawCone( pCommandBuffer, surface, center, scale );
}

void Renderer::DrawCone( nn::gfx::CommandBuffer* pCommandBuffer,
                    const nn::util::Vector3fType& center,
                    float radius,
                    float height,
                    const nn::gfx::DescriptorSlot& textureViewSlot,
                    const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );

    DrawCone( pCommandBuffer, center, scale, textureViewSlot, samplerSlot );
}

void Renderer::DrawCone(nn::gfx::CommandBuffer* pCommandBuffer,
    const Surface surface,
    const nn::util::Vector3fType& center,
    const nn::util::Vector3fType& size) NN_NOEXCEPT
{
    DrawConeImpl(pCommandBuffer, surface, center, size, nullptr, nullptr);
}

void Renderer::DrawCone(nn::gfx::CommandBuffer* pCommandBuffer,
                        const nn::util::Vector3fType& center,
                        const nn::util::Vector3fType& size,
                        const nn::gfx::DescriptorSlot& textureViewSlot,
                        const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawConeImpl(pCommandBuffer, Surface_Solid, center, size, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawConeImpl(nn::gfx::CommandBuffer* pCommandBuffer,
                            const Surface surface,
                            const nn::util::Vector3fType& center,
                            const nn::util::Vector3fType& size,
                            const nn::gfx::DescriptorSlot* pTextureViewSlot,
                            const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    nn::util::Vector3fType arrangeSize
        = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f, 1.0f, 0.5f);
    nn::util::Vector3fType arrangedSize;
    nn::util::VectorMultiply(&arrangedSize, size, arrangeSize);

    MakeModelConstantBuffer( &model, arrangedSize, center, m_Color );

    ShapeType shapeType;
    ShaderVariation shaderVariation;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType = ShapeType_ConeTextured;
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = (surface == Surface_Wired) ?
            ShapeType_ConeWired : ShapeType_Cone;
    }

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif
    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;
    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot );
}

void Renderer::DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  float radius,
                  float height ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );


    DrawCapsule(pCommandBuffer, surface, center, scale );
}

void Renderer::DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                  const nn::util::Vector3fType& center,
                  float radius,
                  float height,
                  const nn::gfx::DescriptorSlot& textureViewSlot,
                  const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );

    DrawCapsule(pCommandBuffer, center, scale, textureViewSlot, samplerSlot);
}

void Renderer::DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size ) NN_NOEXCEPT
{
    DrawCapsuleImpl(pCommandBuffer, surface, center, size, nullptr, nullptr);
}

void Renderer::DrawCapsule( nn::gfx::CommandBuffer* pCommandBuffer,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size,
                  const nn::gfx::DescriptorSlot& textureViewSlot,
                  const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawCapsuleImpl(pCommandBuffer, Surface_Solid, center, size, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawCapsuleImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size,
                  const nn::gfx::DescriptorSlot* pTextureViewSlot,
                  const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    nn::util::Vector3fType cylinderArrangeSize
        = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f, 1.0f, 0.5f);
    nn::util::Vector3fType hemiSphereArrangeSize
        = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f, 0.5f, 0.5f);
    nn::util::Vector3fType arrangedSize;
    nn::util::VectorMultiply(&arrangedSize, size, cylinderArrangeSize);

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixSetScale(&modelMatrix, arrangedSize);
    nn::util::MatrixSetTranslate(&modelMatrix, center);

    ConvertMatrix4x3ToFloatT4x4( &model.u_userMatrix, modelMatrix );
    model.u_color = m_Color;

    // Pipeの描画
    ShapeType shapeType;
    ShaderVariation shaderVariation;
    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType = ShapeType_PipeTextured;
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = (surface == Surface_Wired) ?
            ShapeType_PipeWired : ShapeType_Pipe;
    }

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot);

    nn::util::VectorMultiply(&arrangedSize, size, hemiSphereArrangeSize);

    // 上部半球の描画
    nn::util::MatrixSetScale( &modelMatrix, arrangedSize );
    nn::util::Vector3fType translate;
    nn::util::VectorSet( &translate, nn::util::VectorGetX(center), nn::util::VectorGetY(size), nn::util::VectorGetZ(center) );
    nn::util::MatrixSetAxisW( &modelMatrix, translate );
    ConvertMatrix4x3ToFloatT4x4( &model.u_userMatrix, modelMatrix );

    if (pTextureViewSlot != nullptr)
    {
        shapeType = ShapeType_UpperHalfSphereTextured;
    }
    else
    {
        shapeType = (surface == Surface_Wired) ? ShapeType_UpperHalfSphereWired : ShapeType_UpperHalfSphere;
    }

    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot );

    // 下部半球の描画
    // 上半球を Y 軸反転して作成する。
    nn::util::Vector3fType mirrorSize;
    nn::util::VectorSet( &mirrorSize, nn::util::VectorGetX(arrangedSize), -nn::util::VectorGetY(arrangedSize), nn::util::VectorGetZ(arrangedSize) );
    nn::util::MatrixSetScale( &modelMatrix, mirrorSize );
    nn::util::MatrixSetAxisW( &modelMatrix, center );
    ConvertMatrix4x3ToFloatT4x4( &model.u_userMatrix, modelMatrix );
    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot );

}

void Renderer::DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  float radius,
                  float height ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );


    DrawPipe(pCommandBuffer, surface, center, scale );
}

void Renderer::DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                  const nn::util::Vector3fType& center,
                  float radius,
                  float height,
                  const nn::gfx::DescriptorSlot& textureViewSlot,
                  const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );

    DrawPipe(pCommandBuffer, center, scale, textureViewSlot, samplerSlot);
}

void Renderer::DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size ) NN_NOEXCEPT
{
    DrawPipeImpl(pCommandBuffer, surface, center, size, nullptr, nullptr);
}

void Renderer::DrawPipe( nn::gfx::CommandBuffer* pCommandBuffer,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size,
                  const nn::gfx::DescriptorSlot& textureViewSlot,
                  const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawPipeImpl(pCommandBuffer, Surface_Solid, center, size, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawPipeImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                  const Surface surface,
                  const nn::util::Vector3fType& center,
                  const nn::util::Vector3fType& size,
                  const nn::gfx::DescriptorSlot* pTextureViewSlot,
                  const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    nn::util::Vector3fType pipeArrangeSize
        = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f, 1.0f, 0.5f);
    nn::util::Vector3fType arrangedSize;
    nn::util::VectorMultiply(&arrangedSize, size, pipeArrangeSize);

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixSetScale(&modelMatrix, arrangedSize);
    nn::util::MatrixSetTranslate(&modelMatrix, center);

    ConvertMatrix4x3ToFloatT4x4( &model.u_userMatrix, modelMatrix );
    model.u_color = m_Color;

    // Pipeの描画
    ShapeType shapeType;
    ShaderVariation shaderVariation;
    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType = ShapeType_PipeTextured;
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = (surface == Surface_Wired) ?
            ShapeType_PipeWired : ShapeType_Pipe;
    }

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot);
}

void Renderer::DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                             const Surface surface,
                             const nn::util::Vector3fType& center,
                             float radius,
                             float height ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );

    DrawCylinder(pCommandBuffer, surface, center, scale );
}

void Renderer::DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                             const nn::util::Vector3fType& center,
                             float radius,
                             float height,
                             const nn::gfx::DescriptorSlot& textureViewSlot,
                             const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, radius * 2, height, radius * 2 );

    DrawCylinder(pCommandBuffer, center, scale, textureViewSlot, samplerSlot);
}

void Renderer::DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                             const Surface surface,
                             const nn::util::Vector3fType& center,
                             const nn::util::Vector3fType& size ) NN_NOEXCEPT
{
    DrawCylinderImpl(pCommandBuffer, surface, center, size, nullptr, nullptr);
}

void Renderer::DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                             const nn::util::Vector3fType& center,
                             const nn::util::Vector3fType& size,
                             const nn::gfx::DescriptorSlot& textureViewSlot,
                             const nn::gfx::DescriptorSlot& samplerSlot ) NN_NOEXCEPT
{
    DrawCylinderImpl(pCommandBuffer, Surface_Solid, center, size, &textureViewSlot, &samplerSlot);
}

void Renderer::DrawCylinderImpl( nn::gfx::CommandBuffer* pCommandBuffer,
                                 const Surface surface,
                                 const nn::util::Vector3fType& center,
                                 const nn::util::Vector3fType& size,
                                 const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                 const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    Model model;
    SetDefault( &model );

    // 簡易シェーダディングパラメータを有効に
    if ( surface == Surface_Normal )
    {
        model.rate = 1.0f;
    }

    nn::util::Vector3fType cylinderArrangeSize
        = NN_UTIL_VECTOR_3F_INITIALIZER(0.5f, 1.0f, 0.5f);
    nn::util::Vector3fType arrangedSize;
    nn::util::VectorMultiply(&arrangedSize, size, cylinderArrangeSize);

    nn::util::Matrix4x3fType modelMatrix;
    nn::util::MatrixSetScale(&modelMatrix, arrangedSize);
    nn::util::MatrixSetTranslate(&modelMatrix, center);

    ConvertMatrix4x3ToFloatT4x4( &model.u_userMatrix, modelMatrix );
    model.u_color = m_Color;

    // Cylinderの描画
    ShapeType shapeType;
    ShaderVariation shaderVariation;
    const SamplerType samplerType = (pTextureViewSlot == nullptr) ? SamplerType_CountMax : SamplerType_Texture;

    if (pTextureViewSlot != nullptr)
    {
        shaderVariation = ShaderVariation_PNUv;
        shapeType = ShapeType_CylinderTextured;
    }
    else
    {
        shaderVariation = (surface == Surface_Wired) ? ShaderVariation_P : ShaderVariation_PN;
        shapeType = (surface == Surface_Wired) ?
            ShapeType_CylinderWired : ShapeType_Cylinder;
    }

#if NN_GFX_IS_TARGET_NVN
    if ( surface == Surface_Wired )
    {
        nvnCommandBufferSetLineWidth( pCommandBuffer->ToData()->pNvnCommandBuffer, this->m_Param.lineWidth );
    }
#endif

    DrawPrimitiveInternal( shapeType, shaderVariation, pCommandBuffer, &model, samplerType, pTextureViewSlot, pSamplerSlot);
}

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
