﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/console/detail/console_Hid.h>

#include <limits>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_NpadCommon.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#if defined(NN_BUILD_CONFIG_SPEC_NX)
#include <nn/hid/hid_NpadJoyDual.h>
#endif
#include <nn/util/util_ScopeExit.h>

namespace nns {
namespace console {
namespace detail {
namespace {
PadButtonSet GetPadButtonSet(const nn::hid::DebugPadButtonSet& pad) NN_NOEXCEPT
{
    PadButtonSet s = {};
    s.Set<PadButton::A>(pad.Test<nn::hid::DebugPadButton::A>());
    s.Set<PadButton::B>(pad.Test<nn::hid::DebugPadButton::B>());
    s.Set<PadButton::X>(pad.Test<nn::hid::DebugPadButton::X>());
    s.Set<PadButton::Y>(pad.Test<nn::hid::DebugPadButton::Y>());
    s.Set<PadButton::L>(pad.Test<nn::hid::DebugPadButton::L>());
    s.Set<PadButton::R>(pad.Test<nn::hid::DebugPadButton::R>());
    s.Set<PadButton::ZL>(pad.Test<nn::hid::DebugPadButton::ZL>());
    s.Set<PadButton::ZR>(pad.Test<nn::hid::DebugPadButton::ZR>());
    s.Set<PadButton::Plus>(pad.Test<nn::hid::DebugPadButton::Start>());
    s.Set<PadButton::Minus>(pad.Test<nn::hid::DebugPadButton::Select>());
    s.Set<PadButton::Left>(pad.Test<nn::hid::DebugPadButton::Left>());
    s.Set<PadButton::Up>(pad.Test<nn::hid::DebugPadButton::Up>());
    s.Set<PadButton::Right>(pad.Test<nn::hid::DebugPadButton::Right>());
    s.Set<PadButton::Down>(pad.Test<nn::hid::DebugPadButton::Down>());
    return s;
}
PadButtonSet GetPadButtonSet(nn::hid::DebugPadButtonSet&& pad) NN_NOEXCEPT
{
    auto v = std::move(pad);
    return GetPadButtonSet(v);
}
PadButtonSet GetPadButtonSet(const nn::hid::NpadButtonSet& pad) NN_NOEXCEPT
{
    PadButtonSet s = {};
    s.Set<PadButton::A>(pad.Test<nn::hid::NpadButton::A>());
    s.Set<PadButton::B>(pad.Test<nn::hid::NpadButton::B>());
    s.Set<PadButton::X>(pad.Test<nn::hid::NpadButton::X>());
    s.Set<PadButton::Y>(pad.Test<nn::hid::NpadButton::Y>());
    s.Set<PadButton::L>(pad.Test<nn::hid::NpadButton::L>());
    s.Set<PadButton::R>(pad.Test<nn::hid::NpadButton::R>());
    s.Set<PadButton::ZL>(pad.Test<nn::hid::NpadButton::ZL>());
    s.Set<PadButton::ZR>(pad.Test<nn::hid::NpadButton::ZR>());
    s.Set<PadButton::Plus>(pad.Test<nn::hid::NpadButton::Plus>());
    s.Set<PadButton::Minus>(pad.Test<nn::hid::NpadButton::Minus>());
    s.Set<PadButton::Left>(pad.Test<nn::hid::NpadButton::Left>());
    s.Set<PadButton::Up>(pad.Test<nn::hid::NpadButton::Up>());
    s.Set<PadButton::Right>(pad.Test<nn::hid::NpadButton::Right>());
    s.Set<PadButton::Down>(pad.Test<nn::hid::NpadButton::Down>());
    return s;
}
PadButtonSet GetPadButtonSet(nn::hid::NpadButtonSet&& pad) NN_NOEXCEPT
{
    auto v = std::move(pad);
    return GetPadButtonSet(v);
}

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
template <size_t N, typename T = void>
nn::util::BitFlagSet<N, T> ExtractRepeatedInput(std::pair<bool, int>(*counts)[N], nn::util::BitFlagSet<N, T>&& set0, const std::pair<int, int>& interval) NN_NOEXCEPT
#else
template <size_t N>
struct IntType
{
    NN_STATIC_ASSERT(N <= std::numeric_limits<int>::max());
    static const int value;
};
template <size_t N>
const int IntType<N>::value = static_cast<int>(N);

template <size_t N, typename T = void>
nn::util::BitFlagSet<IntType<N>::value, T> ExtractRepeatedInput(std::pair<bool, int>(*counts)[N], nn::util::BitFlagSet<IntType<N>::value, T>&& set0, const std::pair<int, int>& interval) NN_NOEXCEPT
#endif
{
    nn::util::BitFlagSet<N, T> repeated = {};
    for (auto i = 0; i < set0.GetCount(); ++i)
    {
        auto& count = (*counts)[i];
        if (set0.Test(i))
        {
            if ((++count.second) >= (count.first ? interval.first : interval.second))
            {
                count.first = false;
                count.second = 0;
                repeated.Set(i);
            }
            else
            {
                repeated.Reset(i);
            }
        }
        else
        {
            count.first = true;
            count.second = 0;
            repeated.Reset(i);
        }
    }
    return repeated;
}

class DefaultKeyboardEventListener
    : public KeyboardEventListenerBase
{
public:
    virtual void OnKeyPressedImpl(const nn::hid::KeyboardKeySet&, const nn::hid::KeyboardModifierSet&) NN_NOEXCEPT final NN_OVERRIDE
    {
    }
    virtual void OnKeyReleasedImpl(const nn::hid::KeyboardKeySet&, const nn::hid::KeyboardModifierSet&) NN_NOEXCEPT final NN_OVERRIDE
    {
    }
    virtual void OnKeyRepeatedImpl(const nn::hid::KeyboardKeySet&, const nn::hid::KeyboardModifierSet&) NN_NOEXCEPT final NN_OVERRIDE
    {
    }
} g_DefaultKeyboardEventListner;

class DefaultPadEventListener
    : public PadEventListenerBase
{
public:
    virtual void OnButtonPressedImpl(const PadButtonSet&) NN_NOEXCEPT final NN_OVERRIDE
    {
    }
    virtual void OnButtonReleasedImpl(const PadButtonSet&) NN_NOEXCEPT final NN_OVERRIDE
    {
    }
    virtual void OnButtonRepeatedImpl(const PadButtonSet&) NN_NOEXCEPT final NN_OVERRIDE
    {
    }
} g_DefaultPadEventListener;
} // ~namespace nns::console::detail::<anonymous>

// ---------------------------------------------------------------------------------------------

PadContext::PadContext(int intervalFirst, int interval) NN_NOEXCEPT
    : m_LastSelectedPad(SelectedPad::None)
    , m_DebugPadContext(intervalFirst, interval)
    , m_NpadContextNo1(nn::hid::NpadId::No1, 16, 8)
    , m_NpadContextHandheld(nn::hid::NpadId::Handheld, 16, 8)
{
}

void PadContext::SetEventListener(PadEventListenerBase* pEventListener) NN_NOEXCEPT
{
    m_DebugPadContext.SetEventListener(pEventListener);
    m_NpadContextNo1.SetEventListener(pEventListener);
    m_NpadContextHandheld.SetEventListener(pEventListener);
}
void PadContext::UnsetEventListener() NN_NOEXCEPT
{
    m_DebugPadContext.UnsetEventListener();
    m_NpadContextNo1.UnsetEventListener();
    m_NpadContextHandheld.UnsetEventListener();
}

void PadContext::Update() NN_NOEXCEPT
{
    switch (m_LastSelectedPad)
    {
    case SelectedPad::None:
        if (m_DebugPadContext.TryUpdate())
        {
            NN_LOG("[nns::console::detail::PadContext] Switched to DebugPad\n");
            m_LastSelectedPad = SelectedPad::DebugPad;
            return;
        }
        else if (m_NpadContextNo1.TryUpdate())
        {
            NN_LOG("[nns::console::detail::PadContext] Switched to Npad-No1\n");
            m_LastSelectedPad = SelectedPad::NpadNo1;
        }
        else if (m_NpadContextHandheld.TryUpdate())
        {
            NN_LOG("[nns::console::detail::PadContext] Switched to Npad-Handheld\n");
            m_LastSelectedPad = SelectedPad::NpadHandheld;
        }
        break;

    case SelectedPad::DebugPad:
        if (!m_DebugPadContext.TryUpdate())
        {
            NN_LOG("[nns::console::detail::PadContext] DebugPad disconnected\n");
            m_LastSelectedPad = SelectedPad::None;
        }
        break;

    case SelectedPad::NpadNo1:
        if (!m_NpadContextNo1.TryUpdate())
        {
            NN_LOG("[nns::console::detail::PadContext] Npad-No1 disconnected\n");
            m_LastSelectedPad = SelectedPad::None;
        }
        break;

    case SelectedPad::NpadHandheld:
        if (!m_NpadContextHandheld.TryUpdate())
        {
            NN_LOG("[nns::console::detail::PadContext] Npad-Handlheld disconnected\n");
            m_LastSelectedPad = SelectedPad::None;
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

std::pair<nn::hid::AnalogStickState, nn::hid::AnalogStickState> PadContext::GetAnalogStickState() const NN_NOEXCEPT
{
    switch (m_LastSelectedPad)
    {
    case SelectedPad::None:
        {
            decltype(GetAnalogStickState()) v = {};
            return v;
        }
        break;

    case SelectedPad::DebugPad:
        return m_DebugPadContext.GetAnalogStickState();

    case SelectedPad::NpadNo1:
        return m_NpadContextNo1.GetAnalogStickState();

    case SelectedPad::NpadHandheld:
        return m_NpadContextHandheld.GetAnalogStickState();

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// ---------------------------------------------------------------------------------------------

DebugPadContext::DebugPadContext(int intervalFirst, int interval) NN_NOEXCEPT
    : m_RepeatInterval(intervalFirst, interval)
    , m_ButtonRepeatedCounts {}
    , m_HasPrevious(false)
{
    UnsetEventListener();
}

void DebugPadContext::SetEventListener(PadEventListenerBase* pEventListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventListener);
    m_pEventListener = pEventListener;
}

void DebugPadContext::UnsetEventListener() NN_NOEXCEPT
{
    m_pEventListener = &g_DefaultPadEventListener;
}

bool DebugPadContext::TryUpdate() NN_NOEXCEPT
{
    nn::hid::DebugPadState state;
    nn::hid::GetDebugPadState(&state);
    if (!state.attributes.Test<nn::hid::DebugPadAttribute::IsConnected>())
    {
        if (m_HasPrevious)
        {
            auto prev = GetPadButtonSet(m_PreviousState.buttons);
            if (m_pEventListener && prev.IsAnyOn())
            {
                m_pEventListener->OnButtonReleased(prev);
            }
            m_HasPrevious = false;
            return true;
        }
        return false;
    }

    if (m_HasPrevious)
    {
        if (state.samplingNumber == m_PreviousState.samplingNumber)
        {
            return true;
        }

        auto pressedButtons = GetPadButtonSet((~m_PreviousState.buttons) & state.buttons);
        auto releasedButtons = GetPadButtonSet(m_PreviousState.buttons & (~state.buttons));
        auto repeatedButtons = GetPadButtonSet(ExtractRepeatedInput<>(
            &m_ButtonRepeatedCounts, (m_PreviousState.buttons & state.buttons), m_RepeatInterval));

        if (m_pEventListener && pressedButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonPressed(pressedButtons);
        }
        if (m_pEventListener && releasedButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonReleased(releasedButtons);
        }
        if (m_pEventListener && repeatedButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonRepeated(repeatedButtons);
        }
    }
    else
    {
        auto padButtons = GetPadButtonSet(state.buttons);
        if (m_pEventListener && padButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonPressed(padButtons);
        }
        m_HasPrevious = true;
    }

    m_PreviousState = state;
    return true;
}

std::pair<nn::hid::AnalogStickState, nn::hid::AnalogStickState> DebugPadContext::GetAnalogStickState() const NN_NOEXCEPT
{
    if (m_HasPrevious)
    {
        decltype(GetAnalogStickState()) v = {};
        return v;
    }
    return decltype(GetAnalogStickState())(m_PreviousState.analogStickL, m_PreviousState.analogStickR);
}

// ---------------------------------------------------------------------------------------------
namespace {
template <typename T>
NpadContext::NpadState ConvertNpadState(const T& rhs) NN_NOEXCEPT
{
    NpadContext::NpadState state;
    state.samplingNumber = rhs.samplingNumber;
    state.buttons = rhs.buttons;
    state.analogStickL = rhs.analogStickL;
    state.analogStickR = rhs.analogStickR;
    state.attributes = rhs.attributes;
    return state;
}
} // ~namespace <anonymous>
NpadContext::NpadContext(const nn::hid::NpadIdType& id, int intervalFirst, int interval) NN_NOEXCEPT
    : m_Id(id)
    , m_RepeatInterval(intervalFirst, interval)
    , m_ButtonRepeatedCounts {}
    , m_HasPrevious(false)
{
    UnsetEventListener();
}

void NpadContext::SetEventListener(PadEventListenerBase* pEventListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventListener);
    m_pEventListener = pEventListener;
}

void NpadContext::UnsetEventListener() NN_NOEXCEPT
{
    m_pEventListener = &g_DefaultPadEventListener;
}

NpadContext::NpadState NpadContext::GetNpadState() const NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(m_Id);
    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, m_Id);
        return ConvertNpadState(state);
    }
    else if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, m_Id);
        return ConvertNpadState(state);
    }
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    else if (style.Test<nn::hid::NpadStyleJoyDual>())
    {
        nn::hid::NpadJoyDualState state;
        nn::hid::GetNpadState(&state, m_Id);
        return ConvertNpadState(state);
    }
#endif

    // 接続されていないか、非サポートのスタイル
    NpadState state;
    state.attributes.Reset();
    return state;
}
bool NpadContext::TryUpdate() NN_NOEXCEPT
{
    auto state = GetNpadState();
    if (!state.attributes.Test<nn::hid::NpadAttribute::IsConnected>())
    {
        if (m_HasPrevious)
        {
            auto prev = GetPadButtonSet(m_PreviousState.buttons);
            if (m_pEventListener && prev.IsAnyOn())
            {
                m_pEventListener->OnButtonReleased(prev);
            }
            m_HasPrevious = false;
            return true;
        }
        return false;
    }

    if (m_HasPrevious)
    {
        if (state.samplingNumber == m_PreviousState.samplingNumber)
        {
            return true;
        }

        auto pressedButtons = GetPadButtonSet((~m_PreviousState.buttons) & state.buttons);
        auto releasedButtons = GetPadButtonSet(m_PreviousState.buttons & (~state.buttons));
        auto repeatedButtons = GetPadButtonSet(ExtractRepeatedInput<>(
            &m_ButtonRepeatedCounts, (m_PreviousState.buttons & state.buttons), m_RepeatInterval));

        if (m_pEventListener && pressedButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonPressed(pressedButtons);
        }
        if (m_pEventListener && releasedButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonReleased(releasedButtons);
        }
        if (m_pEventListener && repeatedButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonRepeated(repeatedButtons);
        }
    }
    else
    {
        auto padButtons = GetPadButtonSet(state.buttons);
        if (m_pEventListener && padButtons.IsAnyOn())
        {
            m_pEventListener->OnButtonPressed(padButtons);
        }
        m_HasPrevious = true;
    }

    m_PreviousState = state;
    return true;
}

std::pair<nn::hid::AnalogStickState, nn::hid::AnalogStickState> NpadContext::GetAnalogStickState() const NN_NOEXCEPT
{
    if (m_HasPrevious)
    {
        decltype(GetAnalogStickState()) v = {};
        return v;
    }
    return decltype(GetAnalogStickState())(m_PreviousState.analogStickL, m_PreviousState.analogStickR);
}

// ---------------------------------------------------------------------------------------------

KeyboardContext::KeyboardContext(int intervalFirst, int interval) NN_NOEXCEPT
    : m_RepeatInterval(intervalFirst, interval)
    , m_KeyRepeatedCounts {}
    , m_ModifierRepeatedCounts {}
    , m_HasPrevious(false)
{
    UnsetEventListener();
}

void KeyboardContext::SetEventListener(KeyboardEventListenerBase* pEventListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventListener);
    m_pEventListener = pEventListener;
}

void KeyboardContext::UnsetEventListener() NN_NOEXCEPT
{
    m_pEventListener = &g_DefaultKeyboardEventListner;
}

void KeyboardContext::Update() NN_NOEXCEPT
{
    nn::hid::KeyboardState state;
    nn::hid::GetKeyboardState(&state);

    if (m_HasPrevious)
    {
        if (state.samplingNumber == m_PreviousState.samplingNumber)
        {
            return;
        }

        auto pressedKeys = ((~m_PreviousState.keys) & state.keys);
        auto releasedKeys = (m_PreviousState.keys & (~state.keys));
        auto repeatedKeys = ExtractRepeatedInput<>(&m_KeyRepeatedCounts, (m_PreviousState.keys & state.keys), m_RepeatInterval);

        auto pressedModifiers = ((~m_PreviousState.modifiers) & state.modifiers);
        auto releasedModifiers = (m_PreviousState.modifiers & (~state.modifiers));
        auto repeatedModifiers = ExtractRepeatedInput<>(
            &m_ModifierRepeatedCounts, (m_PreviousState.modifiers & state.modifiers), m_RepeatInterval);

        if (m_pEventListener && (pressedKeys.IsAnyOn() || pressedModifiers.IsAnyOn()))
        {
            m_pEventListener->OnKeyPressed(pressedKeys, pressedModifiers);
        }
        if (m_pEventListener && (releasedKeys.IsAnyOn() || releasedModifiers.IsAnyOn()))
        {
            m_pEventListener->OnKeyReleased(releasedKeys, releasedModifiers);
        }
        if (m_pEventListener && (repeatedKeys.IsAnyOn() || repeatedModifiers.IsAnyOn()))
        {
            m_pEventListener->OnKeyRepeated(repeatedKeys, repeatedModifiers);
        }
    }
    else
    {
        if (m_pEventListener && (state.keys.IsAnyOn() || state.modifiers.IsAnyOn()))
        {
            m_pEventListener->OnKeyPressed(state.keys, state.modifiers);
        }
        m_HasPrevious = true;
    }

    m_PreviousState = state;
}


} // ~namespace nns::console::detail
}
}
