﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iostream>
#include <iomanip>

#include "audio_Loudness.h"
#include <algorithm>

void LoudnessUtil::Step::Reset()
{
    memset(square, 0, sizeof(square));
}

LoudnessUtil::LoudnessUtil(int sampleRate, int channelCount, int sampleCount, int sampleFormat)
{
    m_SampleRate = sampleRate;
    if (sampleRate == 48000)
    {
        m_ShelvCoeff = &ShelvingFilterCoeff48kHz;
        m_HPFCoeff = &HighPassFilterCoeff48kHz;
    }
//     else if (sampleRate == 32000)
//     {
//         m_ShelvCoeff = &ShelvingFilterCoeff32kHz;
//         m_HPFCoeff = &HighPassFilterCoeff32kHz;
//     }
    else
    {
        assert(false); // non supported sample rate
    }

    m_ChannelCount = channelCount;
    m_SampleCountPerAudioFrame = sampleCount;
    m_StepSampleCount = m_SampleRate * StepInterval / 1000;
    m_GatingSampleCount = m_SampleRate * MomentaryInterval / 1000;

    Reset();
}

void LoudnessUtil::Reset()
{
    freeStepList = nullptr;
    filledStepList = nullptr;
    fillingStep = nullptr;
    lastFreeStep = nullptr;
    lastFilledStep = nullptr;

    m_Integrated = 0.f;
    m_IntegratedCount = 0;

    m_CalcIndex = 0;

    ResetValueBuffer();
    m_StepFilledCount = 0;
    for (auto& state : m_ShelvState)
    {
        state.context[0] = 0.0f;
        state.context[1] = 0.0f;
    }
    for (auto& state : m_HighPassState)
    {
        state.context[0] = 0.0f;
        state.context[1] = 0.0f;
    }
    for (auto& step : m_Steps)
    {
        step.Reset();
        PushFreeStep(&step);
    }
}

int LoudnessUtil::GetStepCount()
{
    return StepCount;
}

void LoudnessUtil::PushFreeStep(Step* step)
{
    PushStep(&freeStepList, &lastFreeStep, step);
}

void LoudnessUtil::ResetValueBuffer()
{
    m_MomentaryPower.Reset();
    m_ShortTermPower.Reset();
    m_AbsGatedPower.Reset();
    m_AbsGatedLoudness.Reset();
    m_Integrated = 0.f;
    m_IntegratedCount = 0;
}

LoudnessUtil::Step* LoudnessUtil::GetFreeStep()
{
    return PopStep(&freeStepList, &lastFreeStep);
}

LoudnessUtil::Step* LoudnessUtil::GetFilledStep()
{
    return PopStep(&filledStepList, &lastFilledStep);
}

void LoudnessUtil::PushFilledStep(Step* step)
{
    PushStep(&filledStepList, &lastFilledStep, step);
}

LoudnessUtil::Step* LoudnessUtil::PopStep(Step** list, Step** lastStep)
{
    if (*list)
    {
        auto p = *list;
        *list = (*list)->next;
        if (*list == nullptr)
        {
            *lastStep = nullptr;
        }
        return p;
    }
    return nullptr;
}

void LoudnessUtil::PushStep(Step** list, Step** lastStep, Step* step)
{
    if (*list == nullptr)
    {
        assert(*lastStep == nullptr);
        *list = step;
        *lastStep = step;
        step->next = nullptr;
        return;
    }

    if (*lastStep)
    {
        (*lastStep)->next = step;
    }
    step->next = nullptr;
    *lastStep = step;
}

float LoudnessUtil::BiquadFilter(const FilterCoeff& coeff, BiquadFilterState& state, float sample)
{
    float h1 = state.context[0];
    float h2 = state.context[1];

    float x = sample;
    float y = coeff.b0 * x + h1;
    h1 = x * coeff.b1 - y * coeff.a1 + h2;
    h2 = x * coeff.b2 - y * coeff.a2;

    state.context[0] = h1;
    state.context[1] = h2;

    return y;
}

float LoudnessUtil::KFilitering(float sample, int channel)
{
    auto f = BiquadFilter(*m_ShelvCoeff, m_ShelvState[channel], sample);
    return BiquadFilter(*m_HPFCoeff, m_HighPassState[channel], f);
}

int LoudnessUtil::FillSteps(const int16_t* samples, int sampleCount)
{
    auto left = sampleCount;
    int blockPerChannel = m_SampleCountPerAudioFrame;
    int blockPerFrame = m_SampleCountPerAudioFrame * m_ChannelCount;
    int blockConsumed = 0;
    int consumedFrames = 0;
    while (left > 0)
    {
        // Get new Step
        if (fillingStep == nullptr)
        {
            fillingStep = GetFreeStep();
            if (fillingStep == nullptr)
            {
                // All Step are consumed. Calc to free a Step.
                fillingStep = GetFilledStep();
                assert(fillingStep);
                UpdateValueBuffer(fillingStep);
            }
        }

        // Filling a Step. Note "samples" are block interleaved.
        auto cnt = std::min(blockPerChannel - blockConsumed,
                   std::min(m_StepSampleCount - m_StepFilledCount, left));
        for (auto c = 0; c < m_ChannelCount; c++)
        {
            // "s" means "(sample buffer base) + (consumed count) + (channel offset)"
            auto s =  samples + (consumedFrames * blockPerFrame) + (c * blockPerChannel);
            for (auto i = 0; i < cnt; ++i)
            {
                auto f = *s++ / static_cast<float>(std::numeric_limits<int16_t>::max() + 1);
                 f = KFilitering(f, c);
                fillingStep->square[c] += f * f;
            }
        }

        m_StepFilledCount += cnt;
        blockConsumed += cnt;
        left -= cnt;

        // Push filled Step into filledStepList.
        if (m_StepFilledCount >= m_StepSampleCount)
        {
            fillingStep->index = m_CalcIndex;
            m_CalcIndex = (m_CalcIndex >= std::numeric_limits<uint64_t>::max()) ? 0 : m_CalcIndex + 1;

            PushFilledStep(fillingStep);
            fillingStep = nullptr;
            m_StepFilledCount = 0;
        }

        // Move to next audio frame
        if (blockConsumed >= blockPerChannel)
        {
            ++consumedFrames;
            blockConsumed = 0;
        }
    }
    return sampleCount - left;
}

void LoudnessUtil::Update(const void* buffer, size_t bufferSize)
{
    auto samples = reinterpret_cast<const int16_t*>(buffer);
    auto cnt = static_cast<int>(bufferSize / sizeof(int16_t));

    if (cnt % (m_SampleCountPerAudioFrame * m_ChannelCount))
    {
        assert(false);
    }
    assert(FillSteps(samples, cnt / m_ChannelCount) == cnt / m_ChannelCount);
}

void LoudnessUtil::UpdateValueBuffer(const Step* pStep)
{
    float j = 0.f;
    for (auto c = 0; c < m_ChannelCount; ++c)
    {
        j += ChannelWeightings[c] * pStep->square[c];
    }
    m_MomentaryPower.PushValue(j);
    m_ShortTermPower.PushValue(j);

    auto testVal = CalcLKFS(m_MomentaryPower);
    if (testVal > AbsoluteThreshold)
    {
        auto t = m_MomentaryPower.Sum();
        m_AbsGatedPower.PushValue(t);
        m_AbsGatedLoudness.PushValue(testVal);
    }

    if (m_AbsGatedLoudness.PushedCount() > 0)
    {
        auto relL = CalcLKFS(m_AbsGatedPower.Sum() / (m_AbsGatedLoudness.PushedCount() * m_StepSampleCount * m_MomentaryPower.Length())) + RelativeThreshold;
        m_Integrated = 0.f;
        m_IntegratedCount = 0;
        for (auto i = 0; i < m_AbsGatedLoudness.PushedCount(); ++i)
        {
            if (m_AbsGatedLoudness[i] > relL)
            {
                m_Integrated += m_AbsGatedPower[i];
                ++m_IntegratedCount;
            }
        }
    }
}

bool LoudnessUtil::Calc(LoudnessValues* pOutValues)
{
    bool consumed = false;
    if (auto s = GetFilledStep())
    {
        consumed = true;

        UpdateValueBuffer(s);

        pOutValues->index = s->index;
        pOutValues->momentary = CalcLKFS(m_MomentaryPower);
        pOutValues->shortTerm = CalcLKFS(m_ShortTermPower);
        pOutValues->integrated = CalcLKFS(m_Integrated / (m_IntegratedCount * m_StepSampleCount * m_MomentaryPower.Length()));

        s->Reset();
        PushFreeStep(s);
    }

    return consumed;
}

int LoudnessUtil::GetValues(LoudnessValues* pOutValues, int valueCount)
{
    // clear pOutValues
    for (auto i = 0; i < valueCount; ++i)
    {
        pOutValues[i].index = 0l;
        pOutValues[i].momentary = 0.f;
        pOutValues[i].shortTerm = 0.f;
    }

    int consumed = 0;
    for (; consumed < valueCount; ++consumed)
    {
        if (Calc(&pOutValues[consumed]) == false)
        {
            break;
        }
    }

    return consumed;
}
