﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AssetFileDataHolder.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This files defines some structures that hold
 *  the data provided by an asset file created by
 *  Tutorial02 as well as a class that holds lists
 *  of each section type that made up an single
 *  asset file.
 */

#include <vector>
#include <nn/nn_Assert.h>
#include <nvntutorial/AssetFileDataHolder.h>
#include <nvntutorial/TextureIDManager.h>
#include <nvntutorial/MemoryPool.h>

/*
 * NVNProgramData::Initialize
 * --------------------------
 * Sets up some default values for the structure.
 */
void NVNProgramData::Initialize()
{
    m_pProgramName = NULL;
    m_ShaderStages = 0;
    m_ShaderType = ShaderTypes::ShaderType_Num;
    m_ShaderScratchMemorySize = 0;
}

/*
 * NVNProgramData::SetName
 * --------------------------
 * Sets the name of the shader program.
 */
void NVNProgramData::SetName(const char* pName, int length)
{
    m_pProgramName = new char[length + 1];
    memset(m_pProgramName, 0, length + 1);
    memcpy(m_pProgramName, pName, length);
}

/*
 * NVNProgramData::Finalize
 * ------------------------
 * Cleans up the memory allocated for the object
 * and any NVN objects associated with it.
 */
void NVNProgramData::Finalize()
{
    nvnProgramFinalize(&m_Program);

    if (m_pProgramName)
    {
        delete[] m_pProgramName;
        m_pProgramName = NULL;
    }

    for (size_t i = 0; i < m_ShaderBuffers.size(); ++i)
    {
        nvnBufferFinalize(m_ShaderBuffers[i]);
        delete m_ShaderBuffers[i];
    }

    m_ShaderBuffers.clear();
    m_ShaderStages = 0;

    m_ShaderMemoryPool.Shutdown();

    m_ShaderScratchMemorySize = 0;
}

/*
 * NVNModelData::BindShaderProgram
 * ------------------------
 * Binds the shader program.
 */
void NVNProgramData::BindShaderProgram(NVNcommandBuffer* pCommandBuffer)
{
    nvnCommandBufferBindProgram(pCommandBuffer, &m_Program, m_ShaderStages);
}

/*
 * NVNModelData::Initialize
 * ------------------------
 * Sets up some default values for the structure.
 */
void NVNModelData::Initialize()
{
    m_pModelName = NULL;
    m_VertexBufferSize = 0;
}

/*
 * NVNModelData::SetName
 * ---------------------
 * Sets the model's name.
 */
void NVNModelData::SetName(const char* pName, int length)
{
    m_pModelName = new char[length + 1];
    memset(m_pModelName, 0, length + 1);
    memcpy(m_pModelName, pName, length);
}

/*
 * NVNModelData::Finalize
 * ----------------------
 * Cleans up the memory allocated for the object
 * and any NVN objects associated with it.
 */
void NVNModelData::Finalize()
{
    nvnBufferFinalize(&m_VertexBuffer);

    nvnBufferFinalize(&m_IndexBuffer);

    if(m_pModelName)
    {
        delete[] m_pModelName;
        m_pModelName = NULL;
    }

    m_BufferMemoryPool.Shutdown();

    for (size_t i = 0; i < m_Model.m_VertexAttributes.size(); ++i)
    {
        VertexAttribute& attr = m_Model.m_VertexAttributes[i];
        if(attr.m_pData)
        {
            delete reinterpret_cast<char*>(attr.m_pData);
            attr.m_pData = NULL;
        }
    }

    if (m_Model.m_IndexData.m_pData)
    {
        delete reinterpret_cast<char*>(m_Model.m_IndexData.m_pData);
        m_Model.m_IndexData.m_pData = NULL;
    }
}

/*
 * NVNModelData::AddVertexAttribute
 * --------------------------------
 * Copies the given vertex attribute data into
 * the model data structure.
 */
void NVNModelData::AddVertexAttribute(const VertexAttributeHeader& header)
{
    VertexAttribute attribute;
    attribute.m_Name = std::string(header.m_pAttributeName, header.m_AttributeNameLength);
    attribute.m_DataSize = header.m_AttributeDataSize;
    attribute.m_ElementSize = header.m_AttributeDataSize / header.m_AttributeStride;
    attribute.m_Stride = header.m_AttributeStride;
    attribute.m_NvnFormat = header.m_NvnFormat;
    attribute.m_Location = static_cast<uint32_t>(-1);
    attribute.m_pData = new char[header.m_AttributeDataSize];

    memcpy(attribute.m_pData, header.m_AttributeData, header.m_AttributeDataSize);
    m_Model.m_VertexAttributes.push_back(attribute);
}

/*
 * NVNModelData::SetupAttributeStatesNVN
 * -------------------------------------
 * Sets up NVNvertexAttibState and NVNvertexStream state
 * classes for each of the loaded vertex attributes.
 */
void NVNModelData::SetupAttributeStatesNVN(int32_t (*GetLocation)(const std::string&))
{
    m_VertexAttributeStates.resize(m_Model.m_NumVertexAttributes);
    m_VertexStreamStates.resize(m_Model.m_NumVertexAttributes);

    for (uint32_t i = 0; i < m_Model.m_NumVertexAttributes; ++i)
    {
        VertexAttribute& attribute = m_Model.m_VertexAttributes[i];
        uint32_t location = static_cast<uint32_t>(GetLocation(attribute.m_Name));

        attribute.m_Location = location;

        if (location >= m_Model.m_NumVertexAttributes)
            NN_ASSERT(0, "location outside of vector range.");

            /*
             * NVN Vertex Attribute and Stream States
             * --------------------------------------
             * The NVNvertexAttributeState object is used to
             * define the format of a given vertex attribute. It
             * is also associated with an NVNvertexStreamState
             * object. The stream state controls the stride and
             * the divisor of the attribute. The attribute state
             * and stream state objects are bound to a command
             * buffer in arrays. The index into the attribute
             * state array of a given attribute determines it's
             * binding location in the shader. The stream index
             * held by the attribute state is an index into the
             * bound stream state array.
             */
        NVNvertexAttribState* pAttributeState = &m_VertexAttributeStates[location];

        nvnVertexAttribStateSetDefaults(pAttributeState);
        nvnVertexAttribStateSetStreamIndex(pAttributeState, location);
        nvnVertexAttribStateSetFormat(pAttributeState, static_cast<NVNformat>(attribute.m_NvnFormat), 0);

        NVNvertexStreamState* pStreamState = &m_VertexStreamStates[location];
        nvnVertexStreamStateSetDefaults(pStreamState);
        nvnVertexStreamStateSetStride(pStreamState, attribute.m_Stride);

    }
}

/*
 * NVNTextureData::Initialize
 * --------------------------
 * Sets up some default values for the structure.
 */
void NVNTextureData::Initialize()
{
    m_TextureID = 0;
    m_SamplerID = 0;
    m_TextureInitialized = false;
    m_SamplerInitialized = false;
    m_TextureHandle = NULL;
    m_pTextureName = NULL;
    m_TextureDataSize = 0;
    m_GpuVersion = 0;
    m_Alignment = 0;
    m_Width = 0;
    m_Height = 0;
    m_Depth = 0;
    m_NvnTextureTarget = NVN_TEXTURE_TARGET_2D;
    m_NvnFormat = NVN_FORMAT_NONE;
    m_MipLevels = 0;
}

/*
 * NVNTextureData::SetName
 * -----------------------
 * Sets the name of the texture.
 */
void NVNTextureData::SetName(const char* pName, int length)
{
    m_pTextureName = new char[length + 1];
    memset(m_pTextureName, 0, length + 1);
    memcpy(m_pTextureName, pName, length);
}

/*
 * NVNTextureData::Finalize
 * ------------------------
 * Cleans up the memory allocated for the object
 * and any NVN objects associated with it.
 */
void NVNTextureData::Finalize()
{
    if (m_TextureInitialized)
    {
        nvnTextureFinalize(&m_Texture);
    }

    if (m_SamplerInitialized)
    {
        nvnSamplerFinalize(&m_Sampler);
    }

    m_TextureMemoryPool.Shutdown();

    if (m_pTextureName)
    {
        delete[] m_pTextureName;
        m_pTextureName = NULL;
    }

    m_TextureHandle = NULL;
}

/*
 * AssetFileDataHolder Constructor
 * -------------------------------
 * Empty default constuctor.
 */
AssetFileDataHolder::AssetFileDataHolder()
{
}

/*
 * AssetFileDataHolder Destructor
 * ------------------------------
 * Calls the Finalize funcion to clean up if needed.
 */
AssetFileDataHolder::~AssetFileDataHolder()
{
    Finalize();
}

/*
 * AssetFileDataHolder::Finalize
 * -----------------------------
 * Cleans up the memory allocated for the objects.
 */
void AssetFileDataHolder::Finalize()
{
    for (size_t i = 0; i < m_ProgramData.size(); ++i)
    {
        if (m_ProgramData[i])
        {
            m_ProgramData[i]->Finalize();
            delete m_ProgramData[i];
        }
    }

    for (size_t i = 0; i < m_ModelData.size(); ++i)
    {
        if (m_ModelData[i])
        {
            m_ModelData[i]->Finalize();
            delete m_ModelData[i];
        }
    }

    for (size_t i = 0; i < m_TextureData.size(); ++i)
    {
        if (m_TextureData[i])
        {
            m_TextureData[i]->Finalize();
            delete m_TextureData[i];
        }
    }
}

/*
 * AssetFileDataHolder::AddProgramData
 * -----------------------------------
 * Adds a shader program data structure to the data holder.
 */
void AssetFileDataHolder::AddProgramData(NVNProgramData* pProgramData)
{
    m_ProgramData.push_back(pProgramData);
}

/*
 * AssetFileDataHolder::AddModelData
 * ---------------------------------
 * Adds a model data structure to the data holder.
 */
void AssetFileDataHolder::AddModelData(NVNModelData* pModelData)
{
    m_ModelData.push_back(pModelData);
}

/*
 * AssetFileDataHolder::AddTextureData
 * -----------------------------------
 * Adds a texture data structure to the data holder.
 */
void AssetFileDataHolder::AddTextureData(NVNTextureData* pTextureData)
{
    m_TextureData.push_back(pTextureData);
}

/*
 * AssetFileDataHolder::GetProgramData
 * -----------------------------------
 * Get the list of shader program data structures.
 */
std::vector<NVNProgramData*>& AssetFileDataHolder::GetProgramData()
{
    return m_ProgramData;
}

/*
 * AssetFileDataHolder::GetModelData
 * ---------------------------------
 * Get the list of model data structures.
 */
std::vector<NVNModelData*>& AssetFileDataHolder::GetModelData()
{
    return m_ModelData;
}

/*
 * AssetFileDataHolder::GetTextureData
 * -----------------------------------
 * Get the list of texture data structures.
 */
std::vector<NVNTextureData*>& AssetFileDataHolder::GetTextureData()
{
    return m_TextureData;
}

/*
 * AssetFileDataHolder::SetupAttributeStatesNVN
 * --------------------------------------------
 * Sets up the vertex attribute states and vertex stream
 * states for each model.
 */
void AssetFileDataHolder::SetupAttributeStatesNVN(int32_t(*GetLocation)(const std::string&))
{
    for (size_t i = 0; i < m_ModelData.size(); ++i)
    {
        m_ModelData[i]->SetupAttributeStatesNVN(GetLocation);
    }
}

/*
 * AssetFileDataHolder::SetupTextureSamplerHandle
 * ----------------------------------------------
 * Builds a sampler using the passed in NVNsamplerBuilder,
 * registers the sampler for each texture, and creates
 * the texture handle.
 */
void AssetFileDataHolder::SetupTextureSamplerHandle(NVNdevice* pDevice, TextureIDManager* pTextureIDManager, NVNsamplerBuilder* pSamplerBuilder)
{
    for (size_t i = 0; i < m_TextureData.size(); ++i)
    {
            /* Initialize the sampler used for the texture and register it with the pool. */
        NVNboolean samplerSuccess = nvnSamplerInitialize(&m_TextureData[i]->m_Sampler, pSamplerBuilder);
        NN_ASSERT(samplerSuccess, "Failed to initialize sampler\n");
        m_TextureData[i]->m_SamplerInitialized = true;
        m_TextureData[i]->m_SamplerID = pTextureIDManager->RegisterSampler(&m_TextureData[i]->m_Sampler);

            /* Combine the texture ID and the sampler ID to make a 64-bit texture handle. */
        m_TextureData[i]->m_TextureHandle = nvnDeviceGetTextureHandle(pDevice, m_TextureData[i]->m_TextureID, m_TextureData[i]->m_SamplerID);
    }
}

