﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneScanning.h"

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_Result.h>

namespace {
// スレッドスタック
const size_t ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ ThreadStackSize ];
// スキャンバッファ
const size_t  ssBuffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_ssScanBuffer[ ssBuffSize ];
}

nn::os::EventType SceneScanning::m_event;
SceneScanning::SceneScanning(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    m_animCnt = 0;
    strcpy(m_animStr, ".");
}

void SceneScanning::Initialize() NN_NOEXCEPT
{
    nn::os::InitializeEvent(&m_event, false, nn::os::EventClearMode_AutoClear);
    m_prevTick = nn::os::GetSystemTick();

    nn::Result result;
    result = nn::os::CreateThread(
            &m_scanThread, ScanThreadFunc, NULL, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create scan thread." );

    nn::os::StartThread( &m_scanThread );
}

void SceneScanning::Finalize() NN_NOEXCEPT
{
    nn::os::FinalizeEvent(&m_event);
}

void SceneScanning::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::B) )
    {
        // scanを止める
        nn::wlan::Infra::StopScan();
        nn::os::WaitThread(&m_scanThread);
        nn::os::DestroyThread(&m_scanThread);

        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( nn::os::TryWaitEvent(&m_event) )
    {
        nn::os::WaitThread(&m_scanThread);
        nn::os::DestroyThread(&m_scanThread);

        m_distributor->SetScanResult(g_ssScanBuffer, ssBuffSize);
        m_SceneChanger->ChangeScene(eScene_ScanResult);
    }
}

void SceneScanning::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;

    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );

    if( (nn::os::GetSystemTick() - m_prevTick).ToTimeSpan().GetMilliSeconds() > 500 )
    {
        m_animCnt++;
        strcat(m_animStr, ".");
        if( m_animCnt > TEXT_ANM_STR_LEN - 2 )
        {
            m_animCnt = 0;
            strcpy(m_animStr, ".");
        }
        m_prevTick = nn::os::GetSystemTick();
    }
    textWriter.SetCursor( MY_INITIAL_X + FONT_SIZE, MY_INITIAL_Y );
    textWriter.Print( "Now scanning %s", m_animStr );

    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    textWriter.Print( "[B]:Cancel" );

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
}

void SceneScanning::ScanThreadFunc(void* arg)
{
    nn::Result result;

    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Passive,
            {1, 6, 11},
            0,  // all channel scan
            120,
            0,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    result = nn::wlan::Infra::StartScan(g_ssScanBuffer, ssBuffSize, scanParam);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SignalEvent(&m_event);
}
