﻿using System;
using System.Linq;
using System.Net.Sockets;
using System.Runtime.InteropServices;

namespace WowFunctionTestServer
{
    class ReceivedEventArgs : EventArgs
    {
        public readonly byte[] Data;

        public ReceivedEventArgs(byte[] data)
        {
            Data = data;
        }
    }

    class Client
    {
        public readonly Socket Socket;

        public event EventHandler<ReceivedEventArgs> Received;
        public event EventHandler Closed;

        [DllImport("libc")]
        private static extern int setsockopt(int sockfd, int level, int optname, [MarshalAs(UnmanagedType.LPArray, SizeConst = 1)] int[] optva, int optlen);

        [DllImport("libc")]
        private static extern void perror([MarshalAs(UnmanagedType.LPStr)] string s);

        public Client(Socket socket, int idle, int interval, int count)
        {
            Socket = socket;

            // TCP Keep Alive を有効にして、各オプションを設定
            // この辺はサーバー環境設定で行うもの。あとで取り除くはず
            Socket.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.KeepAlive, true);
            if (Environment.OSVersion.Platform == PlatformID.Unix || Environment.OSVersion.Platform == PlatformID.MacOSX)
            {
                int ret;
                var sockfd = Socket.Handle.ToInt32();
                var opt = new int[2];
                var optlen = new int[1];

                // TCP_KEEPIDLE
                opt[0] = idle;
                ret = setsockopt(sockfd, 6, 4, opt, 4);
                Console.WriteLine("setsockopt:{0}", ret);
                perror("socket");

                // TCP_KEEPINTVL
                opt[0] = interval;
                ret = setsockopt(sockfd, 6, 5, opt, 4);
                Console.WriteLine("setsockopt:{0}", ret);
                perror("socket");

                // TCP_KEEPCNT
                opt[0] = count;
                ret = setsockopt(sockfd, 6, 6, opt, 4);
                Console.WriteLine("setsockopt:{0}", ret);
                perror("socket");

                // SO_LINGER
                opt[0] = 1;
                opt[1] = 0;
                ret = setsockopt(sockfd, 1, 13, opt, 8);
                Console.WriteLine("setsockopt:{0}", ret);
                perror("socket");
            }
            else
            {
                var tcp_keepalive = new byte[12];
                Array.Copy(BitConverter.GetBytes((ulong) 1), 0, tcp_keepalive, 0, 4);     // onoff
                Array.Copy(BitConverter.GetBytes((ulong) idle * 1000), 0, tcp_keepalive, 4, 4);  // keepalivetime (milliseconds)
                Array.Copy(BitConverter.GetBytes((ulong) interval * 1000), 0, tcp_keepalive, 8, 4);  // keepaliveinterval (milliseconds)

                var ret = Socket.IOControl(IOControlCode.KeepAliveValues, tcp_keepalive, null);
            }

            var buffer = new byte[1024];
            Socket.BeginReceive(buffer, 0, buffer.Length, SocketFlags.None, AsyncReceiveCallback, buffer);
        }

        public void Fin()
        {
            Socket.Shutdown(SocketShutdown.Both);
        }

        public void Rst()
        {
            Socket.Close();
        }

        public void Send(byte[] data)
        {
            Socket.Send(data);
        }

        void AsyncReceiveCallback(IAsyncResult asyncResult)
        {
            var buffer = (byte[]) asyncResult.AsyncState;

            int received = 0;
            SocketError error;
            try
            {
                received = Socket.EndReceive(asyncResult, out error);
            }
            catch (ObjectDisposedException)
            {
            }
            if (received == 0)
            {
                // ソケットが閉じられた
                Closed?.Invoke(this, new EventArgs());
                return;
            }

            Received?.Invoke(this, new ReceivedEventArgs(buffer.Take(received).ToArray()));

            Socket.BeginReceive(buffer, 0, buffer.Length, SocketFlags.None, AsyncReceiveCallback, buffer);
        }
    }
}
