﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../common.h"

//#define ENABLE_5GHZ_DEBUG

namespace {

const size_t  txDataSize = 1400; // 1400Bytes
uint8_t  g_txBuffer[ txDataSize ];
uint8_t  g_rxBuffer[2048]; // 受信用バッファ
const size_t ThreadStackSize = 4096;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRx[ ThreadStackSize ];   // 受信用スレッドスタック
nn::os::ThreadType  g_ThreadRx;
static bool g_ExitFlagRx = false;

// データパス用
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRxSoc[ ThreadStackSize ];   // 受信用スレッドスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackTxSoc[ ThreadStackSize ];   // 送信用スレッドスタック
nn::os::ThreadType  g_ThreadRxSoc;
nn::os::ThreadType  g_ThreadTxSoc;
}

// BSSの作成と破棄を繰り返すだけ
#define SEND_ACTION_FRAME
void TestCreateBss()
{
    NN_LOG("Start %s\n", __FUNCTION__);

    nn::wlan::InitializeLocalManager();
    nn::wlan::Local::OpenMasterMode();

#if defined(SEND_ACTION_FRAME)
    // Beacon用途ActionFrameの作成
    nn::wlan::MacAddress macAddr;
    nn::wlan::Local::GetMacAddress(&macAddr);
    nn::Bit8 oui[3] = { 0x00, 0x22, 0xAA };
    WlanTest::GenerateTxDataForLocal(g_txBuffer, nn::wlan::MacAddress::CreateBroadcastMacAddress(), macAddr, 20);
    g_txBuffer[0] = 0x7F;  // Category
    std::memcpy(&g_txBuffer[1], &oui[0], 3);  // OUI
    g_txBuffer[4] = nn::wlan::ActionFrameType_Beacon;  // Subtype
#endif

    // BSSの設定
    nn::wlan::MasterBssParameters param;
    param.channel = -1;
    param.autoKeepAlive = false; // inactivePeriod = 0で同等のことが出来るので使用しない。（一応値は入れておく）
    param.basicRates = nn::wlan::RateSetLegacy_11gMask;
    param.supportedRates = nn::wlan::RateSetLegacy_11gMask;
    param.beaconInterval = 100;
    param.hiddenSsid = true;
    param.inactivePeriod = 10; // 10sec
    param.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
    param.security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
    std::memcpy(&param.security.key[0], &WlanTest::staticAesKey[0], sizeof(WlanTest::staticAesKey));
    param.ssid.Set(WlanTest::localMasterSsid);

    nn::wlan::ConnectionStatus connectionStatus;
    for( int i = 0; i < 20; i++ )
    {
#if defined(SEND_ACTION_FRAME)
        // ActionFrameのセット
        nn::wlan::Local::SetActionFrameWithBeacon(reinterpret_cast<uint8_t*>(g_txBuffer), 20);
        NN_LOG("Set action frame with beacon\n");
#endif
        nn::wlan::Local::CreateBss(param);

        // BSS作成チャンネルの取得
        nn::wlan::Local::GetConnectionStatus(&connectionStatus);
        NN_LOG("new BSS starts on ch%d\n", connectionStatus.channel);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

#if defined(SEND_ACTION_FRAME)
        // ActionFrameの停止
        nn::wlan::Local::CancelActionFrameWithBeacon();
#endif
        // BSSの破棄
        nn::wlan::Local::DestroyBss();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    nn::wlan::Local::CloseMasterMode();
    nn::wlan::FinalizeLocalManager();

    NN_LOG("End %s\n", __FUNCTION__);
}

void RxThreadFunc(void* arg)
{
    nn::Result result;
    g_ExitFlagRx = true;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);

    while( g_ExitFlagRx )
    {
        size_t rxSize = 0;
        result = nn::wlan::Local::GetFrameRaw(g_rxBuffer, sizeof(g_rxBuffer), &rxSize, *rxId); // ブロックされる
        if( result.IsSuccess() )
        {
            NN_LOG("Get data frame from client\n");
            WlanTest::DumpBuffer(g_rxBuffer, rxSize);
            if( g_rxBuffer[17] == 0xAB && g_rxBuffer[18] == 0xCD )
            {
                NN_LOG("Disconnection frame received.\n");
            }
        }
        else if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() )
        {
            NN_LOG("GetFrame Cancelled.\n");
        }
    }
}

void TestCaseLocalAp()
{
    nn::Result      result;

    // 接続に変化があった際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType connectionEvent;

    // 接続状態を格納するためのオブジェクト
    nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
    nn::Bit32 clientStatusBitMap = 0;

    nn::wlan::InitializeLocalManager();

    nn::wlan::Local::OpenMasterMode();
    NN_LOG("WlanMaster:OpenMasterMode done\n");

    nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    NN_LOG("WLANClient: GetConnectionEvent\n");

    nn::wlan::MacAddress macAddr;
    nn::wlan::Local::GetMacAddress(&macAddr);

    // VIE付加
    uint32_t indexA; // 追加したVIEのindexを受け取るための変数
    uint8_t pVie[] = {0x00, 0x22, 0xAA, 0x02, 0xBB, 0x55, 0x16};
    nn::wlan::Local::AddIe(&indexA, nn::wlan::ManagementFrameType_Beacon, pVie, sizeof(pVie));
    NN_LOG("IE index : %d\n", indexA);

    // Beacon用途ActionFrameのセット
    nn::Bit8 oui[3] = { 0x00, 0x22, 0xAA };
    WlanTest::GenerateTxDataForLocal(g_txBuffer, nn::wlan::MacAddress::CreateBroadcastMacAddress(), macAddr, 20);
    g_txBuffer[0] = 0x7F;  // Category
    std::memcpy(&g_txBuffer[1], &oui[0], 3);  // OUI
    g_txBuffer[4] = nn::wlan::ActionFrameType_Beacon;  // Subtype
    nn::wlan::Local::SetActionFrameWithBeacon(reinterpret_cast<uint8_t*>(g_txBuffer), 20);
    NN_LOG("Set action frame with beacon\n");

    nn::wlan::MasterBssParameters param;
    param.channel = 1;
    param.autoKeepAlive = false; // inactivePeriod = 0で同等のことが出来るので使用しない。（一応値は入れておく）
    param.basicRates = nn::wlan::RateSetLegacy_11gMask;
    param.supportedRates = nn::wlan::RateSetLegacy_11gMask;
    param.beaconInterval = 100;
    param.hiddenSsid = true;
    param.inactivePeriod = 10; // 10sec
    param.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
    param.security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
    std::memcpy(&param.security.key[0], &WlanTest::staticAesKey[0], sizeof(WlanTest::staticAesKey));
    param.ssid.Set(WlanTest::localMasterSsid);
    nn::wlan::Local::CreateBss(param);
    NN_LOG("BSS starts on 1ch\n");

    // ActionFrame定期送信が確認しやすいように3秒ほど放置する
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    // ActionFrame定期送信の停止
    NN_LOG("Cancel action frame with beacon\n");
    nn::wlan::Local::CancelActionFrameWithBeacon();

    // 停止しているか確認しやすいように3秒ほど停止状態を作る
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    nn::wlan::Local::DestroyBss();
    NN_LOG("BSS ends on 1ch\n");

    nn::wlan::Local::DeleteIe(indexA);

    // 異なるAP設定でリスタート
    uint32_t indexB = 10;
    nn::wlan::Local::AddIe(&indexB, nn::wlan::ManagementFrameType_Beacon, WlanTest::localVendorIe, sizeof(WlanTest::localVendorIe));
    NN_LOG("IE index : %d\n", indexB);

    // DestroyBssでセットしたActionFrameはクリアされているので、再度セットする
    nn::wlan::Local::SetActionFrameWithBeacon(reinterpret_cast<uint8_t*>(g_txBuffer), 20);
    NN_LOG("Set action frame with beacon\n");

#if defined(ENABLE_5GHZ_DEBUG)
    param.channel = nn::wlan::WirelessChannel_48ch;  // 5GHzデバッグでは36ch, 40ch, 44ch, 48chの4つが使えます。
#else
    param.channel = -1;  // 自動チャンネル選択
#endif
    param.ssid.Set(WlanTest::localMasterSsid);
    param.inactivePeriod = 10;
    nn::wlan::Local::CreateBss(param);

    // BSS作成チャンネルの取得
    nn::wlan::ConnectionStatus connectionStatus;
    nn::wlan::Local::GetConnectionStatus(&connectionStatus);
    NN_LOG("new BSS starts on ch%d\n", connectionStatus.channel);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    // Beacon用途ActionFrameの更新
    g_txBuffer[1] = 0x40;
    g_txBuffer[2] = 0xD2;
    g_txBuffer[3] = 0x8A;
    nn::wlan::Local::SetActionFrameWithBeacon(reinterpret_cast<uint8_t*>(g_txBuffer), 30);

    // BSSを作成したら、BCでデータ送信を開始する。
    // その前にマッチング用データを登録しておく。そうしないと送信許可されない。

    /*
     * マッチング用データ登録作業 ここから--------------------------------------------------------------------------------------
     */
    // マッチング用データ（適当）
    nn::wlan::ReceivedDataMatchInfo matchInfo1 = {
            { 0x00, 0x22, 0xAA, 0x00, 0x01 },  // OUI + PI
            5
    };
    nn::wlan::ReceivedDataMatchInfo matchInfo2 = {
            { 0x00, 0x22, 0xAA, 0xAB, 0xCD },  // OUI + PI
            5
    };
    uint32_t rxId;  // 受信エントリ番号受取り用変数
    uint16_t ethertypes[] = { 0x88b7 };  // 受信したいプロトコルID（EtherType）

    // 受信エントリ作成。
    result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30); // capacityは適当。とりあえず30個は保持できるようにしておく。
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_LOG("WlanMaster:Created Rx Entry [%d]\n", rxId);

    // 作成した受信エントリにマッチング用データを登録。
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo1);
    NN_SDK_ASSERT( result.IsSuccess() );
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo2);
    NN_SDK_ASSERT( result.IsSuccess() );
    /*
     * マッチング用データ登録作業 ここまで--------------------------------------------------------------------------------------
     */

    // 受信スレッド作成
    result = nn::os::CreateThread( &g_ThreadRx, RxThreadFunc, &rxId, g_ThreadStackRx, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadRx." );
    nn::os::StartThread( &g_ThreadRx );

    // 最初の1台の接続を待つ
    NN_LOG("WlanMaster:waiting for connection or disconnection from client...\n");
    nn::os::WaitSystemEvent(&connectionEvent);
    {
        // CLIENTが接続または切断してきた
        NN_LOG("WlanMaster:Connection event signaled\n");
        nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);

        for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
        {
            if( ( 0x00000001 << i ) & clientStatusBitMap )
            {
                WlanTest::PrintClientStatus(&clientStatus[i]);
            }
        }
    }

    // 5秒ごとにCLIENTとの接続状態を確認し、接続中の相手にデータ送信。1台も繋がっていなかったらループを抜ける
    nn::os::Tick statusTick = nn::os::GetSystemTick();
    bool txSwitch = true;  // 通常のデータフレームを送信するかActionFrameを送信するか切り替えるためのスイッチ変数
    while( 1 )
    {
        if( nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true )
        {
            NN_LOG("WlanMaster:Connection event signaled\n");
            nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);

            // 一応ビットダンプして更新場所表示
            NN_LOG("-------------- Updated Client Bit Map --------------\n");
            for( int8_t i = nn::wlan::ConnectableClientsCountMax - 1; i >= 0 ; i-- )
            {
                NN_LOG( "%d", (clientStatusBitMap >> i ) & 0x1 );
            }
            NN_LOG("\n---------------------------------------------------\n");
            for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
            {
                if( ( 0x00000001 << i ) & clientStatusBitMap )
                {
                    WlanTest::PrintClientStatus(&clientStatus[i]);
                }
            }
        }

        // 5秒毎に接続中のCLIENTに対しデータ送信
        if( (nn::os::GetSystemTick() - statusTick).ToTimeSpan().GetMilliSeconds() >= 5000 )
        {
            nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);
            bool clientExist = false;
            for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
            {
                if( clientStatus[i].state == nn::wlan::ConnectionState_Connected )
                {
                    clientExist = true;
                    WlanTest::PrintClientStatus(&clientStatus[i]);
                    if( txSwitch == true )
                    {
                        // 接続中のClientに対しUnicast送信
                        // 送信データ生成
                        // 宛先 : Clientアドレス
                        // 送信元 : 自局MACアドレス
                        WlanTest::GenerateTxDataForLocal(g_txBuffer, clientStatus[i].clientMacAddress, macAddr, 72);
                        nn::wlan::Local::PutFrameRaw(reinterpret_cast<uint8_t*>(g_txBuffer), 72);
                    }
                    else
                    {
                        // ActionFrameの送信
                        nn::Bit8 oui[3] = { 0x00, 0x22, 0xAA };
                        WlanTest::GenerateTxDataForLocal(g_txBuffer, clientStatus[i].clientMacAddress, macAddr, 128);
                        g_txBuffer[0] = 0x7F;  // Category
                        std::memcpy(&g_txBuffer[1], &oui[0], 3);  // OUI
                        g_txBuffer[4] = nn::wlan::ActionFrameType_Local;  // Subtype
                        nn::wlan::Local::PutActionFrameOneShot(nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                                reinterpret_cast<uint8_t*>(g_txBuffer), 128, static_cast<nn::wlan::WirelessChannel>(connectionStatus.channel), 3);
                    }
                    txSwitch ^= 1;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            statusTick = nn::os::GetSystemTick();

            // 1台も接続していなかったら抜ける
            if( clientExist == false )
            {
                NN_LOG("No client exist.\n");
                break;
            }
        }
    }

    // 1秒待ってからBSS止める
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    // Beacon用途ActionFrameの停止
    nn::wlan::Local::CancelActionFrameWithBeacon();

    // IEの消去
    nn::wlan::Local::DeleteIe(indexB);

    // 受信のキャンセルと受信スレッドの破棄
    g_ExitFlagRx = false;
    nn::wlan::Local::CancelGetFrame(rxId);
    nn::os::WaitThread( &g_ThreadRx );
    nn::os::DestroyThread( &g_ThreadRx );
    // 受信エントリ削除
    result = nn::wlan::Local::DeleteRxEntry(rxId);
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_LOG("WlanMaster:Delete Rx Entry [%d]\n", rxId);

    nn::wlan::Local::DestroyBss();  // 内部で全体にDeauthを投げているので、終了前に切断を明示的に行う必要はない。
    NN_LOG("BSS ends on %dch\n", connectionStatus.channel);

    nn::wlan::Local::CloseMasterMode();
    NN_LOG("WlanMaster:CloseMode done\n");

    NN_LOG("\n\n End WlanTest \n\n");
    nn::wlan::FinalizeLocalManager();
} //NOLINT(impl/function_size)

void TestCaseLocalApWithDataPath()
{
    nn::Result      result;

    nn::os::SystemEventType connectionEvent;
    nn::wlan::ConnectionStatus connectionStatus;
    nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
    nn::Bit32 clientStatusBitMap = 0;

    // 初期化
    nn::wlan::InitializeLocalManager();
    nn::wlan::Local::OpenMasterMode();
    nn::wlan::Local::GetConnectionEvent(&connectionEvent);
    for( int i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
    {
        std::memset(&clientStatus[i], 0x00, sizeof(nn::wlan::ClientStatus));
    }

    // VIE付加
    uint32_t ieId; // 追加したVIEのindexを受け取るための変数
    nn::wlan::Local::AddIe(&ieId, nn::wlan::ManagementFrameType_Beacon, WlanTest::localVendorIe, sizeof(WlanTest::localVendorIe));

    nn::wlan::MasterBssParameters param;
    param.channel = -1;
    param.autoKeepAlive = false; // inactivePeriod = 0で同等のことが出来るので使用しない。（一応値は入れておく）
    param.basicRates = nn::wlan::RateSetLegacy_11gMask;
    param.supportedRates = nn::wlan::RateSetLegacy_11gMask;
    param.beaconInterval = 100;
    param.hiddenSsid = true;
    param.inactivePeriod = 10; // 10sec
    param.security.privacyMode = nn::wlan::SecurityMode_Open;//StaticAes;
    param.security.groupPrivacyMode = nn::wlan::SecurityMode_Open;//StaticAes;
    std::memcpy(&param.security.key[0], &WlanTest::staticAesKey[0], sizeof(WlanTest::staticAesKey));
    param.ssid.Set(WlanTest::localMasterSsid);
    nn::wlan::Local::CreateBss(param);

    // BSS作成チャンネルの取得
    nn::wlan::Local::GetConnectionStatus(&connectionStatus);
    NN_LOG("new BSS starts on ch%d\n", connectionStatus.channel);

    // マッチング用データ（適当）
    nn::wlan::ReceivedDataMatchInfo matchInfo1 = {
            { 0x00, 0x22, 0xAA, 0x00, 0x01 },  // OUI + PI
            5
    };
    uint32_t rxId;  // 受信エントリ番号受取り用変数
    uint16_t ethertypes[] = { 0x88b7 };  // 受信したいプロトコルID（EtherType）
    // 受信エントリ作成。
    result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30); // capacityは適当。とりあえず30個は保持できるようにしておく。
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_LOG("WlanMaster:Created Rx Entry [%d]\n", rxId);
    // 作成した受信エントリにマッチング用データを登録。
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo1);
    NN_SDK_ASSERT( result.IsSuccess() );

    // データ通信用スレッドの開始および実行
    {
        nn::wlan::InitializeSocketManager();
        result = nn::os::CreateThread( &g_ThreadRxSoc, WlanTest::DataPathRxThreadFunc, NULL, g_ThreadStackRxSoc, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadRxSoc." );
        nn::os::StartThread( &g_ThreadRxSoc );
        result = nn::os::CreateThread( &g_ThreadTxSoc, WlanTest::DataPathTxThreadFunc, NULL, g_ThreadStackTxSoc, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadTxSoc." );
        nn::os::StartThread( &g_ThreadTxSoc );
    }

    // 最初の1台の接続を待つ。1分間接続が無ければ終了する。
    NN_LOG("WlanMaster:waiting for connection or disconnection from client...\n");
    if( nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(60)) == true )
    {
        // CLIENTが接続または切断してきた
        NN_LOG("WlanMaster:1st Connection event signaled\n");
        nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);

        for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
        {
            if( ( 0x00000001 << i ) & clientStatusBitMap )
            {
                WlanTest::PrintClientStatus(&clientStatus[i]);
            }
        }
    }

    // 5秒ごとにCLIENTとの接続状態を確認し、接続中の相手にデータ送信。1台も繋がっていなかったらループを抜ける
    // 30秒後にBSS破棄
    nn::os::Tick statusTick = nn::os::GetSystemTick();
    nn::os::Tick finishTick = nn::os::GetSystemTick();
    while( 1 )
    {
        if( nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true )
        {
            NN_LOG("WlanMaster:Connection event signaled\n");
            nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);
            for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
            {
                if( ( 0x00000001 << i ) & clientStatusBitMap )
                {
                    WlanTest::PrintClientStatus(&clientStatus[i]);
                }
            }
        }

        // 5秒毎に接続中のCLIENTの状態表示
        if( (nn::os::GetSystemTick() - statusTick).ToTimeSpan().GetMilliSeconds() >= 5000 )
        {
            nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);
            bool clientExist = false;
            for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
            {
                if( clientStatus[i].state == nn::wlan::ConnectionState_Connected )
                {
                    clientExist = true;
                    WlanTest::PrintClientStatus(&clientStatus[i]);
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
            statusTick = nn::os::GetSystemTick();

            // 1台も接続していなかったら抜ける
            if( clientExist == false )
            {
                NN_LOG("No client exist.\n");
                break;
            }
        }

        if( (nn::os::GetSystemTick() - finishTick).ToTimeSpan().GetSeconds() >= 30 )
        {
            break;
        }
    }

    // 1秒待ってからBSS止める
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    // 受信エントリ削除
    result = nn::wlan::Local::DeleteRxEntry(rxId);
    NN_SDK_ASSERT( result.IsSuccess() );

    nn::wlan::Local::DestroyBss();  // 内部で全体にDeauthを投げているので、終了前に切断を明示的に行う必要はない。
    NN_LOG("BSS ends on %dch\n", connectionStatus.channel);

    nn::wlan::Local::DeleteIe(ieId);

    nn::wlan::Local::CloseMasterMode();

    // データパスを閉じる
    {
        nn::os::WaitThread( &g_ThreadRxSoc );
        nn::os::DestroyThread( &g_ThreadRxSoc );
        nn::os::WaitThread( &g_ThreadTxSoc );
        nn::os::DestroyThread( &g_ThreadTxSoc );
        nn::wlan::FinalizeSocketManager();  // データパス用のセッションの終了
    }
    nn::wlan::FinalizeLocalManager();

} //NOLINT(impl/function_size)


void TestCaseLocalApWithMultipleClients()
{
    const int ConnectableCount = 1;

    nn::Result      result;

    // 接続に変化があった際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType connectionEvent;

    // 接続状態を格納するためのオブジェクト
    nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
    nn::Bit32 clientStatusBitMap = 0;

    nn::wlan::InitializeLocalManager();
    nn::wlan::Local::OpenMasterMode();
    nn::wlan::Local::GetConnectionEvent(&connectionEvent);

    nn::wlan::MacAddress macAddr;
    nn::wlan::Local::GetMacAddress(&macAddr);

    nn::wlan::MasterBssParameters param;
    param.channel = 11;
    param.autoKeepAlive = false; // inactivePeriod = 0で同等のことが出来るので使用しない。（一応値は入れておく）
    param.basicRates = nn::wlan::RateSetLegacy_11gMask;
    param.supportedRates = nn::wlan::RateSetLegacy_11gMask;
    param.beaconInterval = 100;
    param.hiddenSsid = false;
    param.inactivePeriod = 10; // 10sec
    param.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
    param.security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
    std::memcpy(&param.security.key[0], &WlanTest::staticAesKey[0], sizeof(WlanTest::staticAesKey));
    param.ssid.Set(WlanTest::localMasterSsid);
    nn::wlan::Local::CreateBss(param);
    NN_LOG("BSS starts on 11ch\n");

    /*
     * マッチング用データ登録作業 ここから--------------------------------------------------------------------------------------
     */
    // マッチング用データ（適当）
    nn::wlan::ReceivedDataMatchInfo matchInfo1 = {
            { 0x00, 0x22, 0xAA, 0x00, 0x01 },  // OUI + PI
            5
    };
    uint32_t rxId;  // 受信エントリ番号受取り用変数
    uint16_t ethertypes[] = { 0x88b7 };  // 受信したいプロトコルID（EtherType）

    // 受信エントリ作成。
    result = nn::wlan::Local::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 30); // capacityは適当。とりあえず30個は保持できるようにしておく。
    NN_SDK_ASSERT( result.IsSuccess() );

    // 作成した受信エントリにマッチング用データを登録。
    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo1);
    NN_SDK_ASSERT( result.IsSuccess() );
    /*
     * マッチング用データ登録作業 ここまで--------------------------------------------------------------------------------------
     */

    // データ通信用スレッドの開始および実行
    {
        nn::wlan::InitializeSocketManager();
        result = nn::os::CreateThread( &g_ThreadRxSoc, WlanTest::DataPathRxThreadFunc, NULL, g_ThreadStackRxSoc, ThreadStackSize, nn::os::DefaultThreadPriority );
        NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadRxSoc." );
        nn::os::StartThread( &g_ThreadRxSoc );
    }

    // 接続を待つ
    nn::wlan::MacAddress clientMac[ConnectableCount];
    int connectedCount = 0;
    NN_LOG("WlanMaster:waiting for connection or disconnection from client...\n");

    while(1)
    {
        nn::os::WaitSystemEvent(&connectionEvent);

        NN_LOG("WlanMaster:Connection event signaled\n");
        nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);

        for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
        {
            if( ( 0x00000001 << i ) & clientStatusBitMap )
            {
                WlanTest::PrintClientStatus(&clientStatus[i]);
                if( clientStatus[i].state == nn::wlan::ConnectionState_Connected )
                {
                    clientMac[connectedCount] = clientStatus[i].clientMacAddress;
                    connectedCount++;
                }
            }
        }

        if( connectedCount >= ConnectableCount )
            break;
    }

    NN_LOG("%d clients connected to master:\n", ConnectableCount);
    for(int i = 0; i < ConnectableCount; i++)
    {
        char str[nn::wlan::MacAddress::MacStringSize];
        clientMac[i].GetString(str);
        NN_LOG("   [%d]%s\n", i + 1, str);
    }


    nn::os::Tick startTick = nn::os::GetSystemTick();
    nn::os::Tick statusTick = nn::os::GetSystemTick();
    int32_t pktCnt = 0;
    while( 1 )
    {
        if( nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true )
        {
            NN_LOG("WlanMaster:Connection event signaled\n");
            nn::wlan::Local::GetClientStatus(clientStatus, &clientStatusBitMap);

            for( int8_t i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
            {
                if( ( 0x00000001 << i ) & clientStatusBitMap )
                {
                    WlanTest::PrintClientStatus(&clientStatus[i]);
                }
            }
        }

        // 1秒毎に接続中のCLIENTに対しデータ送信
        if( (nn::os::GetSystemTick() - statusTick).ToTimeSpan().GetMilliSeconds() >= 1000 )
        {
            pktCnt++;
            for( int i = 0; i < ConnectableCount; i++ )
            {
                // 接続中のClientに対しUnicast送信
                // 送信データ生成
                // 宛先 : Clientアドレス
                // 送信元 : 自局MACアドレス
                // 送信データ生成
                const uint32_t headerSize = nn::wlan::MacAddress::MacAddressSize * 2;

                // Etherヘッダ埋め
                std::memcpy(&g_txBuffer[0], clientMac[i].GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
                std::memcpy(&g_txBuffer[nn::wlan::MacAddress::MacAddressSize], macAddr.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

                // Payload埋め
                // EtherType
                g_txBuffer[headerSize]     = 0x08;
                g_txBuffer[headerSize + 1] = 0x00;

                // パケットカウント埋め込み
                std::memcpy(reinterpret_cast<int32_t*>(&g_txBuffer[headerSize + 2]), &pktCnt, sizeof(pktCnt));

                uint32_t j = 0;
                for( int i = headerSize + 2 + sizeof(pktCnt); i < 72; i++ )
                {
                    g_txBuffer[i] = static_cast<uint8_t>(j & 0xFF);
                    j++;
                }
                nn::wlan::Socket::PutFrameRaw(reinterpret_cast<uint8_t*>(g_txBuffer), 72);
            }
            statusTick = nn::os::GetSystemTick();
        }

        // 5分後にBSS破棄する
        if( (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetSeconds() >= 20 )
        {
            break;
        }
    }

    // 1秒待ってからBSS止める
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    // 受信エントリ削除
    result = nn::wlan::Local::DeleteRxEntry(rxId);
    NN_SDK_ASSERT( result.IsSuccess() );
    NN_LOG("WlanMaster:Delete Rx Entry [%d]\n", rxId);

    nn::wlan::Local::DestroyBss();  // 内部で全体にDeauthを投げているので、終了前に切断を明示的に行う必要はない。
    NN_LOG("BSS ends on 11ch\n");

    nn::wlan::Local::CloseMasterMode();

    // データパスを閉じる
    {
        nn::os::WaitThread( &g_ThreadRxSoc );
        nn::os::DestroyThread( &g_ThreadRxSoc );
        nn::wlan::FinalizeSocketManager();  // データパス用のセッションの終了
    }

    nn::wlan::FinalizeLocalManager();
} //NOLINT(impl/function_size)

extern "C" void nnMain()
{
    NN_LOG("\n\n Start WlanTest LocalMaster\n\n");
    WlanTest::SystemInitialize();
    TestCaseLocalAp();
    //TestCreateBss();
/*
    for( int i = 1; i < 11; i++ )
    {
        NN_LOG("[%d]Start Local Master\n", i);
        TestCaseLocalApWithDataPath();
        //TestCaseLocalApWithMultipleClients();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    }
*/
    WlanTest::SystemFinalize();
}

